/*
 Copyright (c) 2002 RIPE

 All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of the author not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
 AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
 DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

------------------------------------------------------------------------------
 Module Header
 Filename          : sup_frontend.c
 Purpose           : Frontend for synchronous updates,
 Author            : Can Bican <can@ripe.net>
 Description       : Receives data from connections, determines the
                     port to send, and communicates the user with the
                     backend
 Language Version  : C
 Command Line      : sup_frontend
 Input Sources     : environment variables or standart input, depending
                     on the method
 Output Files      : none. feeds data directly to backend
 $Id: sup_frontend.c,v 1.32 2003/01/07 11:09:22 can Exp $
------------------------------------------------------------------------------
*/

#include "supconf.h"
#include "queue.h"
#include "input.h"

#define APP_LOG_DOMAIN "sup_frontend"

#define isEmpty(x) (((x)==NULL)||(strcmp(x,"")==0))

#define CONFFILE "./syncupdates.conf"

char appLogDomain[255];

/*
 * returns true if the agent can modify the origin of update
 * by setting a cgi variable
 */
gboolean canModifyOrigin(gchar * origin)
{
  FILE *f;
  gchar line[200];

  f = fopen(ACLFILE, "r");
  if (f == NULL) {
    if (TEST)
      return TRUE;
    else
      return FALSE;
  }
  while (fgets(line, 80, f)) {
    line[strlen(line) - 1] = 0;
    if ((line[0] != 0) && (line[0] != '#')) {
      if (strncmp(origin, line, strlen(line) - 1) == 0) {
	fclose(f);
	return TRUE;
      }
    }
  }
  fclose(f);
  if (TEST)
    return TRUE;
  else
    return FALSE;
}

/*
   Send the update, and output the result
*/
gchar *update(gchar * data, gchar * action, gchar * remoteAddr)
{
  gint queue, sockfd;
  gulong dataSize, bytesRead, bytesWritten;
  struct sockaddr_in serv_addr;
  gchar *resultAck = NULL, *ack, *tmpstr;
  gboolean streamFinished = FALSE;

  ack = g_new0(char, MAXDATALEN);
  queue = findQueue(data, appLogDomain);

  serv_addr.sin_family = AF_INET;
  serv_addr.sin_addr.s_addr = inet_addr(SERVERADDR);
  serv_addr.sin_port = htons(FIRSTPORT + queue);

  if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    return (g_strdup_printf("error in socket - %s", strerror(errno)));
  }
  if (connect(sockfd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) <
      0) {
    return (g_strdup_printf("error in connect - %s", strerror(errno)));
  }
  if (action != NULL) {
    if ((strncmp(action, "NEW", 3) == 0)
	|| (strncmp(action, "HELP", 4) == 0)) {
      tmpstr = g_strdup_printf("%s\n", action);
      bytesWritten = write(sockfd, tmpstr, strlen(tmpstr));
      if (bytesWritten != (strlen(tmpstr))) {
	g_free(tmpstr);
	return
	    ("error sending data to backend - in 'force new'/'help req'");
      }
      g_free(tmpstr);
    }
  }
  bytesWritten = write(sockfd, remoteAddr, strlen(remoteAddr));
  if (bytesWritten != strlen(remoteAddr)) {
    return ("error sending data to backend - in 'remoteaddr'");
  }
  dataSize = strlen(data);
  bytesWritten = write(sockfd, data, dataSize);
  if (bytesWritten != dataSize) {
    return ("error sending data to backend - in data");
  }
  write(sockfd, "\n\r\n.\r\n", 6);
  while (!streamFinished) {
    bytesRead = read(sockfd, ack, MAXDATALEN);
    if (bytesRead == 0) {
      streamFinished = TRUE;
    } else {
      if (bytesRead == -1) {
	sprintf(ack, "error in read - %s", strerror(errno));
	streamFinished = TRUE;
      } else {
	ack[bytesRead] = 0;
      }
      if (resultAck == NULL) {
	resultAck = g_strdup(ack);
      } else {
	tmpstr = resultAck;
	resultAck = g_strdup_printf("%s%s", resultAck, ack);
	g_free(tmpstr);
      }
    }
  }
  close(sockfd);
  g_free(ack);
  return (resultAck);
}

/*
   User communication
*/
void displayError(gint code, gchar * remark, gchar * txt)
{
  if (!TEST) {
    printf("Content-Type: text/plain\n");
  }
  printf("Status: %d %s\n\n%s\n", code, remark, txt);
}

/*
   Process the message to be printed, formatting it
*/
void processMessage(gchar * data, gchar * action, gchar * remoteAddr)
{
  gchar *resultUpdate;

  resultUpdate = update(data, action, remoteAddr);
  if (g_strncasecmp(resultUpdate, "error", 5) == 0) {
    displayError(506, resultUpdate, "");
  } else {
    displayError(200, "Acknowledgement follows", resultUpdate);
  }
  g_free(resultUpdate);
}

/*
   Print the help message
*/
void helpMessage(gchar * remoteAddr)
{
  processMessage("NONE", "HELP", remoteAddr);
}

/*
   MAIN Function
*/
int main(int argc, char **argv)
{
  int res;
  gchar *ORIGIN, *HELP, *NEW, *DATA;
  gchar *REMOTE_ADDRN, *REMOTE_ADDR, *REQUEST_METHOD, *HTTP_USER_AGENT;
  gchar *emptyVal = "NONE", *confName;
  FILE *f;

  TEST = FALSE;
  if (argc == 2) {
    if (strcmp(argv[1], "TEST") == 0) {
      TEST = TRUE;
    }
  }
  if (!TEST) {
    g_snprintf(appLogDomain, 255, "%s%d", APP_LOG_DOMAIN, (int)getpid());
  } else {
    g_snprintf(appLogDomain, 255, "%s", APP_LOG_DOMAIN);
  }
  if (!TEST) {
    logSethandler(appLogDomain);
  }
  confName = g_strdup(CONFFILE);
  f = fopen(confName, "r");
  if (f == NULL) {
    g_free(confName);
    confName = g_strdup(getenv("SYNCUPDATES_CONF"));
    f = fopen(confName, "r");
    if (f == NULL) {
      fprintf(stderr, "No configuration file.\n");
      exit(1);
    } else {
      fclose(f);
    }
  } else {
    fclose(f);
  }
  readConfFile(confName, appLogDomain);
  res = input_init();
  if (res != CGIERR_NONE) {
    displayError(506, g_strdup_printf("cgi error %d %s", res,
				      input_strerror(res)), "");
    exit(res);
  }
  REMOTE_ADDR = getenv("REMOTE_ADDR");
  if (REMOTE_ADDR == NULL) {
    REMOTE_ADDR = emptyVal;
  }
  REMOTE_ADDRN = g_strdup_printf("%s\n", REMOTE_ADDR);
  REQUEST_METHOD = getenv("REQUEST_METHOD");
  if (REQUEST_METHOD == NULL) {
    REQUEST_METHOD = emptyVal;
  }
  HTTP_USER_AGENT = getenv("HTTP_USER_AGENT");
  if (HTTP_USER_AGENT == NULL) {
    HTTP_USER_AGENT = emptyVal;
  }
  g_log(appLogDomain, G_LOG_LEVEL_INFO, "from %s method %s agent %s",
	REMOTE_ADDR, REQUEST_METHOD, HTTP_USER_AGENT);
  DATA = input_getentrystr("DATA");
  HELP = input_getentrystr("HELP");
  NEW = input_getentrystr("NEW");
  ORIGIN = input_getentrystr("ORIGIN");
  if (canModifyOrigin(REMOTE_ADDR)) {
    if ((ORIGIN != NULL) && (strcmp(ORIGIN, "") != 0)) {
      g_free(REMOTE_ADDRN);
      REMOTE_ADDRN = g_strdup_printf("%s\n", ORIGIN);
    }
  } else {
    g_log(appLogDomain, G_LOG_LEVEL_INFO,
	  "origin specified from invalid host %s", REMOTE_ADDR);
  }
  if ((!isEmpty(DATA)) && (strlen(DATA) > MAXDATALEN)) {
    displayError(413, "Data exceeds maximum allowed size", "");
  } else if (isEmpty(DATA)) {
    if ((isEmpty(HELP)) || (g_strcasecmp(HELP, "no") == 0)) {
      displayError(418, "No input", "");
    } else if (g_strcasecmp(HELP, "no") != 0) {
      helpMessage(REMOTE_ADDRN);
    } else {
      displayError(419, "Command not understood", "");
    }
  } else {
    if ((!isEmpty(HELP)) && (g_strcasecmp(HELP, "no") != 0)) {
      helpMessage(REMOTE_ADDRN);
    } else if ((!isEmpty(NEW)) && (g_strcasecmp(NEW, "no") != 0)) {
      processMessage(DATA, "NEW", REMOTE_ADDRN);
    } else {
      processMessage(DATA, NULL, REMOTE_ADDRN);
    }
  }
  input_quit();
  g_free(REMOTE_ADDRN);
  g_free(confName);
  return 0;
}
