/*
 Copyright (c) 2002 RIPE

 All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of the author not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
 AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
 DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

------------------------------------------------------------------------------
 Module Header
 Filename          : sup_backend.c
 Purpose           : Backend for synchronous updates,
 Author            : Can Bican <can@ripe.net>
 Description       : Feeds data to dbupdate. Middle layer between the
                     CGI and dbupdate
 Language Version  : C
 Command Line      : sup_backend

 $Id: sup_backend.c,v 1.22 2003/02/07 21:10:24 can Exp $
------------------------------------------------------------------------------
*/

#include "supconf.h"

#define APP_LOG_DOMAIN "sup_backend"

char appLogDomain[255];
pid_t childPid[255];

/*
 * Kills children before exit
 */
void killChildren(int s)
{
  int p = 0;

  while (childPid[p] > 0) {
    g_log(appLogDomain, G_LOG_LEVEL_DEBUG, "Killing backend on pid %d", (int)childPid[p]);
    kill(childPid[p], SIGTERM);
    p++;
  }
  unlink(PIDFILE);
}

/*
   executes the command given by path, returns the output
   returns NULL on failure
*/
gchar *my_spawn(const gchar * path)
{
  FILE *f;
  gchar buf[MAXDATALEN], *tmpStr, *resultStr = NULL;

  g_return_val_if_fail(path != NULL, FALSE);

  g_log(appLogDomain, G_LOG_LEVEL_DEBUG, "spawning %s",path);
  f = popen(path, "r");
  if (f == NULL) {
    return NULL;
  } else {
    while (fgets(buf, MAXDATALEN, f)) {
      if (resultStr == NULL) {
        if (buf!=NULL)
          resultStr = g_strdup(buf);
        else resultStr=g_strdup("");
      } else {
	tmpStr = g_strdup_printf("%s%s", resultStr, buf);
	g_free(resultStr);
	resultStr = tmpStr;
      }
    }
  }
  pclose(f);
  return (resultStr);
}

/*
   Gets the request coming from the socket, passes it to dbupdate, and
   replies with the output from dbupdate
*/
void processRequest(int port)
{
  gulong bufSize, s, f;
  gchar *cmdLine, *c, *tmpnam, *outStr, *errStr, *buf,*origin,*p,*tmpBuf;
  gboolean inputFinished = FALSE,isNew=FALSE,isHelp=FALSE;
  gint bufOffset;

  c = g_new0(char, 1024);
  tmpnam = g_strdup("/tmp/tmpsupbeXXXXXX");
  buf = g_new0(char, MAXDATALEN);
  bufSize = 0;
  while (!inputFinished) {
    s = read(port, c, 1000);
    if (s > 0) {
      c[s] = 0;
      if (buf[0] == 0) {
	g_snprintf(buf, MAXDATALEN, "%s", c);
      } else {
	      tmpBuf=g_strconcat(buf,c,NULL);
	      g_free(buf);
	      buf=tmpBuf;
      }
      bufSize = strlen(buf);
      if (bufSize >= 5) {
	if (g_strncasecmp((buf + bufSize - 5), "\r\n.\r\n", 5) == 0) {
	  inputFinished = TRUE;
	  bufSize = bufSize - 5;
	  buf[bufSize] = 0;
	}
      }
    } else if (s == 0) {
      inputFinished = TRUE;
    } else {
      g_log(appLogDomain, G_LOG_LEVEL_ERROR, "read failed: %s",
	    strerror(errno));
    }
  }
  f = mkstemp(tmpnam);
  bufOffset=0;
  if (strncmp(buf,"NEW",3)!=0) {
    isNew=FALSE;
    if (strncmp(buf,"HELP",4)!=0) {
      isHelp=FALSE;
    } else {
      g_log(appLogDomain,G_LOG_LEVEL_DEBUG,"help requested");
      isHelp=TRUE;
      bufOffset=bufOffset+5;
    }
  } else {
    g_log(appLogDomain,G_LOG_LEVEL_DEBUG,"new forced");
    isNew=TRUE;
    bufOffset = bufOffset+4;
  }
  origin=g_strdup(buf+bufOffset);
  p=strchr(origin,'\n');
  if (p!=NULL) {
    bufOffset=bufOffset+p-origin+1;
    p[0]=0;
  }
  if (origin!=NULL) {
    if (strncmp(origin,"NONE",4)==0) {
      g_free(origin);
      origin=g_strdup_printf("127.0.0.1");
    }
  }
  g_log(appLogDomain,G_LOG_LEVEL_DEBUG,"origin is: %s",origin);
  if (isNew) {
    cmdLine = g_strdup_printf("%s -o \"%s\" -e -f %s", DBUPDATE, origin, tmpnam);
  } else {
    if (isHelp) {
      cmdLine = g_strdup_printf("%s -o \"%s\" -h -f %s", DBUPDATE, origin, tmpnam);
    } else {
      cmdLine = g_strdup_printf("%s -o \"%s\" -f %s", DBUPDATE, origin, tmpnam);
    }
  }
  if (f == -1) {
    g_log(appLogDomain, G_LOG_LEVEL_ERROR, "mkstemp failed: %s",
	  strerror(errno));
  }
  write(f, buf+bufOffset, bufSize-bufOffset);
  close(f);
  outStr = my_spawn(cmdLine);
  if (outStr != NULL) {
    write(port, outStr, strlen(outStr));
  } else {
    errStr =
	g_strdup_printf("error connecting dbupdate");
    write(port, errStr, strlen(errStr));
    g_free(errStr);
  }
  unlink(tmpnam);
  g_free(c);
  g_free(tmpnam);
  g_free(outStr);
  g_free(buf);
  g_free(cmdLine);
  g_free(origin);
}

/*
   Starts a server for the given port
*/
void startServer(int port)
{
  int clilen, socketfd, newsocketfd;
  struct sockaddr_in peer, servaddr, cliaddr;
  socklen_t peerlen;
  int opt = TRUE;

  g_log(appLogDomain, G_LOG_LEVEL_DEBUG, "Starting backend on port %d pid %d",
	port,(int)getpid());
  if ((socketfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
    g_log(appLogDomain, G_LOG_LEVEL_ERROR, "socket: %s", strerror(errno));
    exit(errno);
  }
  if (setsockopt
      (socketfd, SOL_SOCKET, SO_REUSEADDR, (char *) &opt, sizeof(opt)) < 0) {
    g_log(appLogDomain, G_LOG_LEVEL_ERROR, "setsockopt: %s",
	  strerror(errno));
    exit(errno);
  }
  servaddr.sin_family = AF_INET;
  servaddr.sin_addr.s_addr = htonl(INADDR_ANY);
  servaddr.sin_port = htons(port);
  if (bind(socketfd, (struct sockaddr *) &servaddr, sizeof(servaddr)) < 0) {
    g_log(appLogDomain, G_LOG_LEVEL_ERROR, "bind: %s", strerror(errno));
    exit(errno);
  }
  if (listen(socketfd, 5) < 0) {
    g_log(appLogDomain, G_LOG_LEVEL_ERROR, "listen: %s", strerror(errno));
    exit(errno);
  }
  for (;;) {
    clilen = sizeof(cliaddr);
    newsocketfd = accept(socketfd, (struct sockaddr *) &cliaddr, &clilen);
    if (newsocketfd < 0) {
      g_log(appLogDomain, G_LOG_LEVEL_ERROR, "accept: %s",
	    strerror(errno));
      exit(errno);
    }
    peerlen = sizeof(peer);
    if ((getpeername(newsocketfd, (struct sockaddr *) &peer, &peerlen)) <
	0) {
      g_log(appLogDomain, G_LOG_LEVEL_ERROR, "getpeername: %s",
	    strerror(errno));
      exit(errno);
    }
    g_log(appLogDomain, G_LOG_LEVEL_INFO, "new connection from %s",
	  inet_ntoa(peer.sin_addr));
    processRequest(newsocketfd);
    close(newsocketfd);
  }
}

/* 
   MAIN FUNCTION
   Fork one server for each queue, and exit 
*/
int main(int argc, char **argv)
{
  int port = 0, cp = 0;
  pid_t p;
  char c;
  char *confFile = NULL;
  FILE *pidf;

  g_snprintf(appLogDomain, 255, "%s", APP_LOG_DOMAIN);
  logSethandler(appLogDomain);
  while ((c = getopt(argc, argv, "f:")) != -1) {
    switch (c) {
    case 'f':
      confFile = optarg;
      break;
    case '?':
      if (isprint(optopt))
	fprintf(stderr, "Unknown option -%c\n", optopt);
      else
	fprintf(stderr, "Unknown option character \\x%x\n", optopt);
      return (1);
    default:
      abort();
    }
  }
  if (confFile == NULL) {
    fprintf(stderr, "Please specify a configuration file.\n%s -f <conffile>\n",argv[0]);
    exit(1);
  }
  readConfFile(confFile, appLogDomain);
  p = fork();
  if (p == -1) {
    g_log(appLogDomain, G_LOG_LEVEL_ERROR, "fork: %s", strerror(errno));
  } else if (p == 0) {
    while (port < QUEUECOUNT) {
      p = fork();
      if (p == -1) {
	g_log(appLogDomain, G_LOG_LEVEL_ERROR, "fork: %s",
	      strerror(errno));
      } else if (p == 0) {
	startServer(port + FIRSTPORT);
      } else {
	childPid[cp] = p;
	childPid[cp + 1] = (-1);
	cp++;
	port++;
      }
    }
    if (signal(SIGTERM, killChildren) != 0) {
      g_log(appLogDomain, G_LOG_LEVEL_ERROR, "signal(SIGTERM,...): %s",
	    strerror(errno));
    }
    wait(NULL);
  } else {
    pidf = fopen(PIDFILE, "w");
    if (pidf == NULL) {
      g_log(appLogDomain, G_LOG_LEVEL_ERROR, "fopen %s: %s", PIDFILE,
	    strerror(errno));
    }
    fprintf(pidf, "%d\n", (int)p);
    fclose(pidf);
  }
  return 0;
}
