# Rails

ActiveLdap supports Rails 4.0 or later.

## Install

To install, simply add the following code to your Gemfile:

```ruby
gem 'activeldap', :require => 'active_ldap/railtie'
```

You should also depend on an LDAP adapter such as Net::LDAP
or Ruby/LDAP. The following example uses Ruby/LDAP:

```ruby
gem 'ruby-ldap'
```

Bundler will install the gems automatically when you run
`bundle install`.

## Configuration

You can use a LDAP configuration per environment. They are in
a file named 'ldap.yml' in the config directory of your
rails app. This file has a similar function to the
'database.yml' file that allows you to set your database
connection configurations per environment. Similarly, the
ldap.yml file allows configurations to be set for
development, test, and production environments.

You can generate 'config/ldap.yml' by the following command:

```console
% script/rails generate active_ldap:scaffold
```

You need to modify 'config/ldap.yml' generated by
`active_ldap:scaffold`. For instance, the development entry
would look something like the following:

```yaml
development:
  host: 127.0.0.1
  port: 389
  base: dc=localhost
  bind_dn: cn=admin,dc=localhost
  password: secret
```

When your application starts up,
ActiveLdap::Base.setup_connection will be called with the
parameters specified for your current environment.

You can replace default orm generators with gems one
to skip `active_ldap prefix` in `config/application.rb`:

```ruby
config.app_generators.orm :active_ldap
```

## Model

You can generate a User model that represents entries under
ou=Users by the following command:

```console
% script/rails generate active_ldap:model User --dn-attribute uid --classes person PosixAccount
```

It generates the following app/model/user.rb:

```ruby
class User < ActiveLdap::Base
  ldap_mapping :dn_attribute => "uid",
               :prefix => "ou=Users",
               :classes => ["person", "PosixAccount"]
end
```

You can add relationships by modifying app/model/user.rb:

```ruby
class User < ActiveLdap::Base
  ldap_mapping :dn_attribute => 'uid',
               :prefix => "ou=Users",
               :classes => ['person', 'posixAccount']
  belongs_to :primary_group,
             :class_name => "Group",
             :foreign_key => "gidNumber",
             :primary_key => "gidNumber"
  belongs_to :groups,
             :many => 'memberUid'
end
```

You can also generate a Group model by the following command:

```console
% script/rails generate active_ldap:model Group --classes PosixGroup
```

app/model/group.rb:

```ruby
class Group < ActiveLdap::Base
  ldap_mapping :dn_attribute => "cn",
               :prefix => "ou=Groups",
               :classes => ["PosixGroup"]
end
```

You can add relationships by modifying app/model/group.rb:

```ruby
class Group < ActiveLdap::Base
  ldap_mapping :dn_attribute => "cn",
               :prefix => "ou=Groups",
               :classes => ["PosixGroup"]
  has_many :members,
           :class_name => "User",
           :wrap => "memberUid"
  has_many :primary_members,
           :class_name => "Group",
           :foreign_key => "gidNumber",
           :primary_key => "gidNumber"
end
```

You can also generate a Ou model by the following command:

```console
% script/rails generate active_ldap:model Ou --prefix '' --classes organizationalUnit
```

```ruby
class Ou < ActiveLdap::Base
  ldap_mapping :dn_attribute => "cn",
               :prefix => "",
               :classes => ["organizationalUnit"]
end
```
