/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/degenerate/cdf' );
import Degenerate = require( './../../../../../base/dists/degenerate/ctor' );
import entropy = require( './../../../../../base/dists/degenerate/entropy' );
import logcdf = require( './../../../../../base/dists/degenerate/logcdf' );
import logpdf = require( './../../../../../base/dists/degenerate/logpdf' );
import logpmf = require( './../../../../../base/dists/degenerate/logpmf' );
import mean = require( './../../../../../base/dists/degenerate/mean' );
import median = require( './../../../../../base/dists/degenerate/median' );
import mgf = require( './../../../../../base/dists/degenerate/mgf' );
import mode = require( './../../../../../base/dists/degenerate/mode' );
import pdf = require( './../../../../../base/dists/degenerate/pdf' );
import pmf = require( './../../../../../base/dists/degenerate/pmf' );
import quantile = require( './../../../../../base/dists/degenerate/quantile' );
import stdev = require( './../../../../../base/dists/degenerate/stdev' );
import variance = require( './../../../../../base/dists/degenerate/variance' );

/**
* Interface describing the `degenerate` namespace.
*/
interface Namespace {
	/**
	* Degenerate distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param mu - constant value of distribution
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 5.0 );
	* // returns 0.0
	*
	* var mycdf = ns.cdf.factory( 5.0 );
	*
	* y = mycdf( 3.0 );
	* // returns 0.0
	*
	* y = mycdf( 6.0 );
	* // returns 1.0
	*/
	cdf: typeof cdf;

	/**
	* Degenerate distribution.
	*/
	Degenerate: typeof Degenerate;

	/**
	* Returns the entropy of a degenerate distribution.
	*
	* @param mu - constant value of distribution
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 0.1 );
	* // returns 0.0
	*
	* @example
	* var v = ns.entropy( 0.5 );
	* // returns 0.0
	*
	* @example
	* var v = ns.entropy( 10.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.entropy( NaN );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Degenerate distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param mu - constant value of distribution
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 2.0, 5.0 );
	* // returns -Infinity
	*
	* var mylogcdf = ns.logcdf.factory( 5.0 );
	*
	* y = mylogcdf( 3.0 );
	* // returns -Infinity
	*
	* y = mylogcdf( 6.0 );
	* // returns 0.0
	*/
	logcdf: typeof logcdf;

	/**
	* Degenerate distribution natural logarithm of probability density function (logPDF).
	*
	* @param x - input value
	* @param mu - value at which to center the distribution
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 0.0 );
	* // returns -Infinity
	*
	* var mylogpdf = ns.logpdf.factory( 10.0 );
	*
	* y = mylogpdf( 10.0 );
	* // returns Infinity
	*/
	logpdf: typeof logpdf;

	/**
	* Degenerate distribution natural logarithm of probability mass function (PMF).
	*
	* @param x - input value
	* @param mu - value at which to center the distribution
	* @returns evaluated logPMF
	*
	* @example
	* var y = ns.logpmf( 2.0, 0.0 );
	* // returns -Infinity
	*
	* var mylogpmf = ns.logpmf.factory( 10.0 );
	*
	* var y = mylogpmf( 10.0 );
	* // returns 0.0
	*/
	logpmf: typeof logpmf;

	/**
	* Returns the expected value of a degenerate distribution.
	*
	* @param mu - constant value of distribution
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 0.1 );
	* // returns 0.1
	*
	* @example
	* var v = ns.mean( 0.5 );
	* // returns 0.5
	*
	* @example
	* var v = ns.mean( 10.0 );
	* // returns 10.0
	*
	* @example
	* var v = ns.mean( NaN );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a degenerate distribution.
	*
	* @param mu - constant value of distribution
	* @returns median
	*
	* @example
	* var v = ns.median( 0.1 );
	* // returns 0.1
	*
	* @example
	* var v = ns.median( 0.5 );
	* // returns 0.5
	*
	* @example
	* var v = ns.median( 10.0 );
	* // returns 10.0
	*
	* @example
	* var v = ns.median( NaN );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Degenerate distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param mu - value at which to center the distribution
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 2.0, 0.0 );
	* // returns 1.0
	*
	* var mymgf = ns.mgf.factory( 10.0 );
	*
	* y = mymgf( 0.1 );
	* // returns ~2.718
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of a degenerate distribution.
	*
	* @param mu - constant value of distribution
	* @returns mode
	*
	* @example
	* var v = ns.mode( 0.1 );
	* // returns 0.1
	*
	* @example
	* var v = ns.mode( 0.5 );
	* // returns 0.5
	*
	* @example
	* var v = ns.mode( 10.0 );
	* // returns 10.0
	*
	* @example
	* var v = ns.mode( NaN );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Degenerate distribution probability density function (PDF).
	*
	* @param x - input value
	* @param mu - value at which to center the distribution
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 0.0 );
	* // returns 0.0
	*
	* var mypdf = ns.pdf.factory( 10.0 );
	*
	* y = mypdf( 10.0 );
	* // returns Infinity
	*/
	pdf: typeof pdf;

	/**
	* Degenerate distribution probability mass function (PMF).
	*
	* @param x - input value
	* @param mu - value at which to center the distribution
	* @returns evaluated PMF
	*
	* @example
	* var y = ns.pmf( 2.0, 0.0 );
	* // returns 0.0
	*
	* var mypmf = ns.pmf.factory( 10.0 );
	*
	* y = mypmf( 10.0 );
	* // returns 1.0
	*/
	pmf: typeof pmf;

	/**
	* Degenerate distribution quantile function.
	*
	* @param p - input value
	* @param mu - constant value of the distribution
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.5, 2.0 );
	* // returns 2.0
	*
	* var myQuantile = ns.quantile.factory( 10.0 );
	*
	* y = myQuantile( 0.5 );
	* // returns 10.0
	*/
	quantile: typeof quantile;

	/**
	* Returns the standard deviation of a degenerate distribution.
	*
	* @param mu - constant value of distribution
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 0.1 );
	* // returns 0.0
	*
	* @example
	* var v = ns.stdev( 0.5 );
	* // returns 0.0
	*
	* @example
	* var v = ns.stdev( 10.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.stdev( NaN );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a degenerate distribution.
	*
	* @param mu - constant value of distribution
	* @returns variance
	*
	* @example
	* var v = ns.variance( 0.1 );
	* // returns 0.0
	*
	* @example
	* var v = ns.variance( 0.5 );
	* // returns 0.0
	*
	* @example
	* var v = ns.variance( 10.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.variance( NaN );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Degenerate distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
