#-------------------------------------------------------------------------
# MODULE mkPrint.tcl
#
# This module is a part of the Visual Ising Machine
#-------------------------------------------------------------------------


#-------------------------------------------------------------------------
#
# mkPrint w
#
# Builds a toplevel dialog box for the PostScript printing utility
#
# Arguments:
#	w -	window
#-------------------------------------------------------------------------
proc mkPrint {w} {

	catch {destroy $w}
	toplevel $w
	wm title $w "VIM \[Print\]"
	wm iconname $w "Print"


	set topF [frame $w.topF -relief raised -bd 1]
	set deviceF [mkPrDevice $topF]
	set optionF [mkPrOption $topF]
	set dlgBtnsF   [mkPrDlgButtons $w]

	pack $deviceF -side left -anchor nw -fill both -expand yes -padx 20 -pady 20 
	pack $optionF -side left -anchor n -fill x -expand yes -padx 20 -pady 20 
	pack $topF -side top -fill both -expand yes
	pack $dlgBtnsF -side bottom -fill both -expand yes

}


#-------------------------------------------------------------------------
#
# mkPrDevice top
#
# Builds a frame that holds radiobuttons, etc to select an
# outputdevice
#
# Arguments:
#	top -	toplevel frame
#-------------------------------------------------------------------------
proc mkPrDevice {top} {
	global outputDevice 
	global outputFileName
	
	set w [tixLabelFrame $top.deviceF -label "Device"]

	# This frame is used to give the labelframe a fixed width.
	# If you omit it, the labelFrame will adjust its size to
	# the biggest widget it's containing, thus messing up the
	# layout
	frame $w.sizingF -width 285
	pack $w.sizingF -side top
		
	set prF [frame $w.printerF]
	set fiF [frame $w.fileF]

	radiobutton $prF.r1 -text "printer" -width 8\
		-variable outputDevice -value 1 \
		-relief groove 

	tixComboBox $prF.cmb -type editable -width 26 
	$prF.cmb appendhistory "lpr"
	$prF.cmb appendhistory "gs "	

	radiobutton $fiF.r2 -text "file" -width 8 \
		-variable outputDevice -value 0 \
		-relief groove

	entry $fiF.ent -width 20 -textvariable outputFileName
	button $fiF.btn -text "Browse..." \
		 -command "mkPrFileBox $top prFileBoxOkCmd" 

	pack $prF.r1 -anchor nw -pady 5
	pack $prF.cmb -anchor se
	pack $fiF.r2  -anchor sw -pady 5
	pack $fiF.btn -side right -anchor se 
	pack $fiF.ent -side right -anchor s -padx 10
	pack $prF -padx 10 -pady 5 -fill both -expand yes
	pack $fiF -padx 10 -pady 5 -fill both -expand yes


	$prF.r1 configure -command {
		set path .vim.printdlg.topF.deviceF.f2

		$path.printerF.cmb configure -state normal
		$path.fileF.ent configure -state disabled
		$path.fileF.btn configure -state disabled
		unset path
	}
	$fiF.r2 configure -command {
		set path .vim.printdlg.topF.deviceF.f2

		$path.printerF.cmb configure -state disabled
		$path.fileF.ent configure -state normal
		focus $path.fileF.ent
		$path.fileF.btn configure -state normal

		unset path
	}


	if {$outputDevice == 1} {
		$prF.r1 invoke
	} else {
		$fiF.r2 invoke
	}
	return $top.deviceF

}


#-------------------------------------------------------------------------
#
# mkPrOption top
#
# Builds a frame that holds two radiobuttons to select color or b&w.
#
# Arguments:
#	top -	toplevel frame
#-------------------------------------------------------------------------
proc mkPrOption {top} {
	global colorOptions
	
	set lbF [tixLabelFrame $top.optionF -label "Options"]
	set w [frame $lbF.radioF]
	
	radiobutton $w.r1 -text "color" -width 10\
		-variable colorOption -value 1\
		-relief groove
	radiobutton $w.r2 -text "grayscale" -width 10\
		-variable colorOption -value 0\
		-relief groove

	pack $w.r1 $w.r2
	pack $w -padx 10 -pady 10
	
	return $top.optionF	
}


#-------------------------------------------------------------------------
#
# mkPrDlgButtons top
#
# Builds a frame that holds some dialog buttons
#
# Arguments:
#	top -	toplevel frame
#-------------------------------------------------------------------------
proc mkPrDlgButtons {top} {
	set btns [tixDlgBtns $top.btns]

	$btns add print -text Print -width 7 -command "prPrintCmd"
#	$btns add preview -text Preview -width 7
	$btns add close -text Close -width 7 -command "destroy $top"
	$btns add help -text Help -width 7 -state disabled

 	return $btns
}


#-------------------------------------------------------------------------
#
# mkPrFileBox top okcmd helpcmd
#
# Builds a filedialog to select an output file
#
# Arguments:
#	top     -	toplevel window
#	okcmd   -	command to exec when Ok is pressed
#	helpcmd -	command to exec when Help is pressed
#-------------------------------------------------------------------------
proc mkPrFileBox {top okcmd {helpcmd {}} } {
	set w [toplevel $top.fileBox]
	wm minsize $w 10 10
	set btns [tixStdDlgBtns $w.btns]
	tixFileSelectBox $w.fsb -command "destroy $w; $okcmd" 	
	wm title $w "VIM \[select output file\]"

	$btns ok config -command "$w.fsb invoke"
	$btns apply config -text Filter -command "$w.fsb filter"
	$btns cancel config -command "destroy $w"
	if {$helpcmd == {}} {
		$btns help config -state disabled
	} else {
		$btns help config -command $helpcmd
	}
	pack $btns -side bottom -fill both
	pack $w.fsb -fill both -expand yes
	
	return $w.fsb
}


#-------------------------------------------------------------------------
#
# prFileBoxOkCmd filename
#
# Sets the global var outputFileName to <filename>
#
# Arguments:
#	filename -	a string containing the chosen file
#-------------------------------------------------------------------------
# TODO:  filename parsen op path en naam
#	 en zorgen dat in het entryfield alleen de filename te zien is
#-------------------------------------------------------------------------
proc prFileBoxOkCmd {filename} {
	global outputFileName

	set outputFileName $filename
}


#-------------------------------------------------------------------------
#
# prPrintCmd
#
# This procedure is invoked by pressing the Print button
#
#-------------------------------------------------------------------------
proc prPrintCmd {} {
	global state

	if {$state == "PLAYING"} {
		mkStdErrorBox {
Error: Simulation running!

You cannot print a frame while the simulation is still running!
Pause or stop the simulation first, then try again.
}

		return
	}

	# else...
	global outputDevice
	global outputFileName
	global outputDeviceName	
	global colorOption

	if {$outputDevice == 0} {
		remote ".ising print -color $colorOption \
				-file \"${outputFileName}\""
	} else {
		remote ".ising print -color $colorOption \
				-device \"${outputDeviceName}\""
	}
}

