# Bindings for entry widgets


# Is there a better way to ring the bell???
proc beep {} {	catch "puts stderr {} nonewline"}


# Entry traversal

proc e_recenter {e} {
	$e view [expr [$e index insert] - ([lindex [$e configure -width] 4])]
}

proc e_select_next_line {e} {
	if {[catch "$e index sel.last"]} { set start insert
	} elseif {([$e index sel.first] > [$e index insert]) ||
		  ([$e index sel.last] < [$e index insert])} {
		$e select clear
		set start insert
	} else {set start sel.first}
	$e select from $start
	$e select to end
}

# Location of the entry's mark (-1 is an invalid value)
set e_mark -1

proc e_select_region {e} {
	global e_mark
	if {($e_mark < 0)} {beep
	} else {if {[$e index insert] > $e_mark} {
			set start $e_mark
			set end insert
		} else {set start insert
			set end $e_mark}
		$e select from $start
		$e select to $end
}}

proc e_move_char {e d} {
	set i [expr "[$e index insert] $d"]
	if {($i > [$e index end]) || ($i < 0)} {beep
	} else {$e icursor $i
		e_recenter $e
}}

proc e_exchange_dot_and_mark {e} {
	global e_mark
	if {($e_mark < 0)} {beep ; return}
	set i [$e index insert]
	$e icursor $e_mark
	set e_mark $i
	e_recenter $e
}


# Editing Selection

proc e_kill_next_line {e} {
	.e_kill delete 0 end
	.e_kill insert 0 [string range [$e get] [$e index insert] \
						[$e index end]]
	$e delete insert end
	.e_kill select from 0
	.e_kill select to end
}

proc e_kill_region {e} {
	global e_mark
	if {($e_mark < 0)} {beep ; return}
	.e_kill delete 0 end
	if {[$e index insert] > $e_mark} {
		set start $e_mark
		set end insert
	} else {set start insert
		set end $e_mark}
	.e_kill insert 0 [string range [$e get] [$e index $start] \
						 [$e index $end]]
	.e_kill select from 0
	.e_kill select to end
	$e delete $start $end
	e_recenter $e
}

proc e_paste_selection {e} {
	if {[catch {set chars [selection get]}]} {beep
	} else {$e insert insert $chars
		e_recenter $e
}}

proc e_delete_selection {e} {
	if {[catch "$e index sel.last"]} {beep ; return}
	$e delete sel.first sel.last
}


# General editing

proc e_delete_char {e} {
	if {([$e index insert] == [$e index end])} {beep
	} else {$e delete insert
		e_recenter $e
}}

proc e_backspace {e} {
	if {([$e index insert] == 0)} {beep
	} else {$e delete [expr "[$e index insert] - 1"]
		e_recenter $e
}}

proc e_unquote {e c} {
	$e insert insert $c
	bind $e <Key> ""
	bind $e <Control-Key> ""
	e_recenter $e
}

proc e_key_quote {e c} {
	if {([regexp . $c])} {
		beep
		bind $e <Key> ""
		bind $e <Control-Key> ""
}}

proc e_quote_insert {e} {
	bind $e <Key> "e_key_quote $e %A"
	bind $e <Control-Key> "e_unquote $e %A"
}

proc e_transpose_chars {e} {
	set i [$e index insert]
	if {($i < 2) || ($i >= [$e index end])} {beep ; return}

	set c1 [string index [$e get] [expr "$i-1"]]
	set c2 [string index [$e get] [expr "$i"]]
	$e delete [expr "$i-1"] [expr "$i"]
	$e insert insert $c2
	$e insert insert $c1
	e_recenter $e
}

proc e_transpose_words {e} {
	set i [$e index insert]
	if {($i < 2) || ($i >= [$e index end])} {beep ; return}

	set y [expr "[$e index insert]-1"]
	set c2 [string index [$e get] $y]
	$e delete $y
	incr y -1

	e_back_word $e
	set x [$e index insert]
	set c1 [string range [$e get] $x $y]
	$e delete $x $y

	set y [$e index insert]
	e_forward_word $e
	set x [expr "[$e index insert]-1"]
	set c3 [string range [$e get] $y $x]
	$e delete $y $x

	$e insert insert $c3
	$e insert insert $c2
	$e insert insert $c1
	$e icursor [expr "[$e index insert]+1"]
	e_recenter $e
}

proc e_self_insert {e {c ""}} {
	if {(![regexp . $c])} {return}
	$e insert insert $c
	e_recenter $e
}

# Adjusts the widget's size. option is -height or -width, d is +n or -n where
# n is the increment/decrement value. w is the widget.
# This works for any widget with a -width or -height option.
proc widget_resize {w option d} {
	$w configure $option [expr "[lindex [$w configure $option] 4] $d"]
}


# Word traversal

# Moves cursor back one word
proc e_back_word {e} {
	set string [$e get]
	set curs [expr [$e index insert]-1]
	if {$curs < 0} {beep ; return}
	for {set x $curs} {$x > 0} {incr x -1} {
		if {([string first [string index $string $x] " \t"] < 0)
			&& ([string first [string index $string [expr $x-1]] \
			" \t"] >= 0)} {   break}}
	$e icursor $x
	e_recenter $e
}

proc e_delete_word_back {e} {
	if {([$e index insert] == 0)} {beep ; return}
	set y [expr "[$e index insert]-1"]
	e_back_word $e
	set x [$e index insert]
	$e delete $x $y
	e_recenter $e
}

# Moves cursor forward one word
proc e_forward_word {e} {
	set string [$e get]
	set curs [expr "[$e index insert]+1"]
	set end [$e index end]
	if {$curs > $end} {beep ; return}
	for {set x $curs} {$x < $end} {incr x} {
		if {([string first [string index $string $x] " \t"] >= 0)
			&& ([string first [string index $string [expr $x+1]] \
			" \t"] < 0)} {   break}}
	$e icursor $x
	e_recenter $e
}

proc e_delete_word_forward {e} {
	if {([$e index insert] > [$e index end])} {beep ; return}
	set y [$e index insert]
	e_forward_word $e
	set x [expr "[$e index insert]-1"]
	$e delete $y $x
	e_recenter $e
}

proc string_tolower {s} {return [string tolower $s]}
proc string_toupper {s} {return [string toupper $s]}
proc string_capitalize {s} {
	return "[string toupper [string index $s 0]][string tolower [string range $s 1 end]]"
}

proc e_filter_word {e filter} {
	if {[catch "$e index sel.last"]} {
		set start [$e index insert]
		e_forward_word $e
		set end [expr "[$e index insert]-1"]
		set selected 0
	} else {set start [$e index sel.first]
		set end [$e index sel.last]
		set selected 1
	}
	set w [string range [$e get] $start $end]
	set new_w [$filter $w]

	if {($w == $new_w)} {e_recenter $e ; return}

	$e delete $start $end
	$e insert insert $new_w
	$e icursor [expr "[$e index insert]+1"]
	if {($selected)} {
		$e select from $start
		$e select to insert
	}
	e_recenter $e
}


# Entry bindings
proc entrybind {} {
	bind Entry <Key> {e_self_insert %W %A}
	bind Entry <Return> {beep}
	bind Entry <Control-Key> {if {[regexp . %A]} {beep}}
	bind Entry <Meta-Key> {if {[regexp . %A]} {beep}}
	bind Entry <Control-Meta-Key> {if {[regexp . %A]} {beep}}
	bind Entry <Escape> {;}

	bind Entry <Control-Delete> "e_delete_selection %W"
	bind Entry <Control-space> {set e_mark [%W index insert]}
	bind Entry <Control-a> "%W icursor 0 ; e_recenter %W"
	bind Entry <Control-b> "e_move_char %W -1"
	bind Entry <Meta-b> {e_back_word %W}
	bind Entry <Control-c> "%W select from 0 ; %W select to end ; %W select clear"
	bind Entry <Meta-c> "e_filter_word %W string_capitalize"
	bind Entry <Control-d> "e_delete_char %W"
	bind Entry <Meta-d> "e_delete_word_forward %W"
	bind Entry <Control-e> "%W icursor end ; e_recenter %W"
	bind Entry <Control-f> "e_move_char %W +1"
	bind Entry <Meta-f> {e_forward_word %W}
	bind Entry <Control-d> "e_delete_char %W"
	bind Entry <Control-g> {eval [bind Text <Control-g>]}
	bind Entry <Control-h> "e_backspace %W"
	bind Entry <Meta-h> "e_delete_word_back %W"
	bind Entry <Control-i> {e_self_insert %W \t}
	bind Entry <Control-k> "e_kill_next_line %W"
	bind Entry <Control-K> "e_select_next_line %W"
	bind Entry <Control-l> "e_recenter %W"
	bind Entry <Meta-l> "e_filter_word %W string_tolower"
	bind Entry <Control-q> "e_quote_insert %W"
	bind Entry <Control-t> "e_transpose_chars %W"
	bind Entry <Meta-t> "e_transpose_words %W"
	bind Entry <Control-u> "%W icursor 0 ; e_kill_next_line %W"
	bind Entry <Meta-u> "e_filter_word %W string_toupper"
	bind Entry <Control-w> "e_kill_region %W"
	bind Entry <Control-W> "e_select_region %W"
	bind Entry <Meta-w> "%W select from 0 ; %W select to end"
	bind Entry <Control-x> "e_exchange_dot_and_mark %W"
	bind Entry <Control-y> "e_paste_selection %W"
	bind Entry <Control-bracketright> "widget_resize %W -width -1"
	bind Entry <Control-braceright> "widget_resize %W -width +1"

# Duplicate Keybindings
	bind Entry <Key-KP_Enter> "[bind Entry <Return>]"
	bind Entry <Delete> "[bind Entry <Control-h>]"
	bind Entry <Select> [bind Entry <Control-W>]
	bind Entry <Shift-Select> [bind Entry <Meta-w>]
	bind Entry <Left> [bind Entry <Control-b>]
	bind Entry <Shift-Left> [bind Entry <Control-a>]
	bind Entry <Right> [bind Entry <Control-f>]
	bind Entry <Shift-Right> [bind Entry <Control-e>]
	bind Entry <Insert> [bind Entry <Control-y> ]
	bind Entry <Control-v> [bind Entry <Control-y> ]
	bind Entry <ButtonRelease-3> [bind Entry <Control-y>]
	bind Entry <Button1-Delete> [bind Entry <Control-Delete>]
	bind Entry <Control-Left> [bind Entry <Control-bracketright>]
	bind Entry <Control-Right> [bind Entry <Control-braceright>]
}

entrybind
catch {entry .e_kill}
