# Bindings for entry widgets


# Is there a better way to ring the bell???
proc beep {} {	catch "puts stderr {} nonewline"}


# Entry traversal

proc e_recenter {e} {
	$e view [expr [$e index insert] - ([lindex [$e configure -width] 4])]
}

proc e_select_next_line {e} {
	if {[catch "$e index sel.last"]} { set start insert
	} elseif {([$e index sel.first] > [$e index insert]) ||
		  ([$e index sel.last] < [$e index insert])} {
		$e select clear
		set start insert
	} else {set start sel.first}
	$e select from $start
	$e select to end
}

# Location of the entry's mark (-1 is an invalid value)
set e_mark -1

proc e_select_region {e} {
	global e_mark
	if {($e_mark < 0)} {beep
	} else {if {[$e index insert] > $e_mark} {
			set start $e_mark
			set end insert
		} else {set start insert
			set end $e_mark}
		$e select from $start
		$e select to $end
}}

proc e_move_char {e d} {
	set i [expr "[$e index insert] $d"]
	if {($i > [$e index end]) || ($i < 0)} {beep
	} else {$e icursor $i
		e_recenter $e
}}

proc e_exchange_dot_and_mark {e} {
	global e_mark
	if {($e_mark < 0)} {beep ; return}
	set i [$e index insert]
	$e icursor $e_mark
	set e_mark $i
	e_recenter $e
}


# Editing Selection

# Removes contents of e, and replaces with new_string, leaving view intact.
proc e_replace {e new_string} {
	set new_index [expr [$e index insert] + [string length $new_string] - \
				[string length [$e get]]]
	set beginning [$e index @0]
	$e delete 0 end
	$e insert 0 $new_string
	$e view $beginning
	$e icursor $new_index
}

proc e_kill_next_line {e} {
	e_replace .e_kill [string range [$e get] [$e index insert] [$e index end]] 
	$e delete insert end
	.e_kill select from 0
	.e_kill select to end
}

proc e_kill_region {e} {
	global e_mark
	if {($e_mark < 0)} {beep ; return}
	if {[$e index insert] > $e_mark} {
		set start $e_mark
		set end insert
	} else {set start insert
		set end $e_mark}
	e_replace .e_kill [string range [$e get] [$e index $start] [$e index $end]]
	.e_kill select from 0
	.e_kill select to end
	$e delete $start $end
	e_recenter $e
}

proc e_paste_selection {e} {
	if {[catch {set chars [selection get]}]} {beep
	} else {$e insert insert $chars
		e_recenter $e
}}

proc e_delete_selection {e} {
	if {[catch "$e index sel.last"]} {beep ; return}
	$e delete sel.first sel.last
}


# General editing

proc e_delete_char {e} {
	if {([$e index insert] == [$e index end])} {beep
	} else {$e delete insert
		e_recenter $e
}}

proc e_backspace {e} {
	if {([$e index insert] == 0)} {beep
	} else {$e delete [expr "[$e index insert] - 1"]
		e_recenter $e
}}

proc e_unquote {e c} {
	global Keys
	$e insert insert $c
	parse_bindings $e $Keys(A_C_Key) {}
	e_recenter $e
}

proc e_key_quote {e c} {
	global Keys
	if {([regexp . $c])} {
		beep
		parse_bindings $e $Keys(A_C_Key) {}
}}

proc e_quote_insert {e} {
	parse_bindings $e \
Key				"e_key_quote $e %A" \
C-Key				"e_unquote $e %A"
}

proc e_transpose_chars {e} {
	set i [$e index insert]
	if {($i < 2) || ($i >= [$e index end])} {beep ; return}

	set c1 [string index [$e get] [expr "$i-1"]]
	set c2 [string index [$e get] [expr "$i"]]
	$e delete [expr "$i-1"] [expr "$i"]
	$e insert insert $c2
	$e insert insert $c1
	e_recenter $e
}

proc e_transpose_words {e} {
	set i [$e index insert]
	if {($i < 2) || ($i >= [$e index end])} {beep ; return}

	set y [expr "[$e index insert]-1"]
	set c2 [string index [$e get] $y]
	$e delete $y
	incr y -1

	e_back_word $e
	set x [$e index insert]
	set c1 [string range [$e get] $x $y]
	$e delete $x $y

	set y [$e index insert]
	e_forward_word $e
	set x [expr "[$e index insert]-1"]
	set c3 [string range [$e get] $y $x]
	$e delete $y $x

	$e insert insert $c3
	$e insert insert $c2
	$e insert insert $c1
	$e icursor [expr "[$e index insert]+1"]
	e_recenter $e
}

set overwrite_mode 0

proc e_self_insert {e {c ""}} {
	if {(![regexp . $c])} {return}
	$e insert insert $c
	global overwrite_mode ;	if $overwrite_mode {$e delete insert}
	e_recenter $e
}

# Adjusts the widget's size. option is -height or -width, d is +n or -n where
# n is the increment/decrement value. w is the widget.
# This works for any widget with a -width or -height option.
proc widget_resize {w option d} {
	$w configure $option [expr "[lindex [$w configure $option] 4] $d"]
}


# Word traversal

# Moves cursor back one word
proc e_back_word {e} {
	set string [$e get]
	set curs [expr [$e index insert]-1]
	if {$curs < 0} {beep ; return}
	for {set x $curs} {$x > 0} {incr x -1} {
		if {([string first [string index $string $x] " \t"] < 0)
			&& ([string first [string index $string [expr $x-1]] \
			" \t"] >= 0)} {   break}}
	$e icursor $x
	e_recenter $e
}

proc e_delete_word_back {e} {
	if {([$e index insert] == 0)} {beep ; return}
	set y [expr "[$e index insert]-1"]
	e_back_word $e
	set x [$e index insert]
	$e delete $x $y
	e_recenter $e
}

# Moves cursor forward one word
proc e_forward_word {e} {
	set string [$e get]
	set curs [expr "[$e index insert]+1"]
	set end [$e index end]
	if {$curs > $end} {beep ; return}
	for {set x $curs} {$x < $end} {incr x} {
		if {([string first [string index $string $x] " \t"] >= 0)
			&& ([string first [string index $string [expr $x+1]] \
			" \t"] < 0)} {   break}}
	$e icursor $x
	e_recenter $e
}

proc e_delete_word_forward {e} {
	if {([$e index insert] > [$e index end])} {beep ; return}
	set y [$e index insert]
	e_forward_word $e
	set x [expr "[$e index insert]-1"]
	$e delete $y $x
	e_recenter $e
}


# Filters that replace one string with another
proc string_tolower {s} {return [string tolower $s]}
proc string_toupper {s} {return [string toupper $s]}
proc string_capitalize {s} {
	return "[string toupper [string index $s 0]][string tolower [string range $s 1 end]]"
}

proc e_filter_word {e filter} {
	if {[catch "$e index sel.last"]} {
		set start [$e index insert]
		e_forward_word $e
		set end [expr "[$e index insert]-1"]
		set selected 0
	} else {set start [$e index sel.first]
		set end [$e index sel.last]
		set selected 1
	}
	set w [string range [$e get] $start $end]
	set new_w [$filter $w]

	if {($w == $new_w)} {e_recenter $e ; return}

	$e delete $start $end
	$e insert insert $new_w
	$e icursor [expr "[$e index insert]+1"]
	if {($selected)} {
		$e select from $start
		$e select to insert
	}
	e_recenter $e
}


# String replacement / completion

# Does completion to the string in an entry binding.
# Beeps unless exact completion was made.
proc e_complete_string {e possibilities {message_prelude "Possible values:\n"} \
				{function filter_completions}} {
	set completion [complete_string $function $possibilities [$e get]]
	global completion_message completion_index
	set completion_index -1
	if {[lindex $completion 1] != ""} {
		set completion_message [lindex $completion 1]
	} else {set completion_message ""}
	if {([lindex $completion 0] == "") || ([lindex $completion 1] != "")} {beep}
	if {[lindex $completion 0] != ""} {e_replace $e [lindex $completion 0]}
}

# Does completion to the substring in an entry binding before the insert cursor.
# Beeps unless completion was made and no message exists.
proc e_complete_multiple {e {completion_list ""}} { 
	set index [expr [$e index insert] - 1]
	set head [string range [$e get] 0 $index]
	set tail [string range [$e get] [$e index insert] end]
	set completion [multiple_completion $head $completion_list]
	global completion_message completion_index
	set completion_message [lindex $completion 1]
	set completion_index [lindex $completion 2]
	if {([lindex $completion 0] == "") || ([lindex $completion 1] != "")} {beep}
	if {[lindex $completion 0] != ""} {
		e_replace $e "[lindex $completion 0]$tail"
}}

# Shows list of completions, letting user pick one.
proc e_show_completion_dialog {e} {
	global completion_message completion_index
	if {$completion_message == ""} {beep ; return}
	if {[string match {Possible *} $completion_message]} {
		set view [expr [string length [$e get]] - $completion_index]
		set completion [eval tk_dialog_listbox .conf \
		   {{Completion Dialog}} [split $completion_message \n] $view]
		if {$completion != ""} {
			e_replace $e "[string range [$e get] 0 \
					$completion_index]$completion"}
	} else {tk_dialog .conf Error $completion_message info 0 OK
}}

proc toggle_overwrite_mode {} {
	global overwrite_mode
	if $overwrite_mode {set overwrite_mode 0} else {set overwrite_mode 1}
}


# Entry bindings
proc entrybind {m} {
	global Keys
	foreach binding [bind Entry] {
		if {[string match *Key* $binding]} {
			bind Entry $binding ""}}

	parse_bindings all \
C-g				{beep} \
M-o				{toggle_overwrite_mode}
	bind Text <Escape>	{switch_focus}

	parse_bindings Entry \
Escape		{switch_focus} \
Key		{e_self_insert %W %A} \
$Keys(C_m)	{beep} \
Tab		{e_complete_multiple %W} \
C-Tab		{e_show_completion_dialog %W} \
$Keys(C_Delete)	{e_delete_selection %W} \
C-space		{set e_mark [%W index insert]} \
$Keys(C_a)	{%W icursor 0 ; e_recenter %W} \
{C-b Left}	{e_move_char %W -1} \
M-b		{e_back_word %W} \
C-c		{%W select from 0 ; %W select to end ; %W select clear} \
M-c		{e_filter_word %W string_capitalize} \
C-d		{e_delete_char %W} \
M-d		{e_delete_word_forward %W} \
$Keys(C_e)	{%W icursor end ; e_recenter %W} \
$Keys(C_f)	{e_move_char %W +1} \
M-f		{e_forward_word %W} \
$Keys(C_h)	{e_backspace %W} \
M-h		{e_delete_word_back %W} \
C-k		{e_kill_next_line %W} \
C-K		{e_select_next_line %W} \
C-l		{e_recenter %W} \
M-l		{e_filter_word %W string_tolower} \
C-q		{e_quote_insert %W} \
C-t		{e_transpose_chars %W} \
M-t		{e_transpose_words %W} \
C-u		{%W icursor 0 ; e_kill_next_line %W} \
M-u		{e_filter_word %W string_toupper} \
C-w		{e_kill_region %W} \
{C-W Select}	{e_select_region %W} \
{M-w S-Select}	{%W select from 0 ; %W select to end} \
C-x		{e_exchange_dot_and_mark %W} \
$Keys(C_y)	{e_paste_selection %W}

	if {[winfo exists $m]} {parse_menu $m \
{Edit 0	{Kill 0	""			{Line 0 C-k}
					{Region 0 C-w}}
					{Yank 0 C-y}
	{Delete 0 ""	{Character 0 ""	{Previous 0 C-h}
					{Next 0 C-d}}
			{Word 0	""	{Previous 0 M-h}
					{Next 0 M-d}}
					{Selection 0 C-Delete}}
	{Case 0 ""			{Upper 0 M-u}
					{Lower 0 M-l}
					{Capitalize 0 M-c}}
					{Quote 0 C-q}
	{Transpose 0 ""			{Characters 0 C-t}
					{Words 0 M-t}}} \
{Entry 1			{"Switch Focus" 0 Escape}
				{"Abort" 0 C-g}
				{"Exit" 0 Return}
	separator
				{"Complete Word" 0 Tab}
				{"Show Completion Info" 16 C-Tab}
	separator
				{Reset 0 C-u}}

# No command, since the command merely changes the variable.
		$m.edit.m add checkbutton -label Overwrite -underline 1 \
			-variable overwrite_mode -offvalue 0 -onvalue 1 \
			-accelerator M-o

		$m.entry configure -state disabled
		$m.edit configure -state disabled
}}


entrybind $menu
catch {entry .e_kill}
