//$TextView$
overload Swap;
#include "TextView.h"

#include "TextCmd.h"
#include "CmdNo.h"
#include "String.h"
#include "View.h"
#include "BlankWin.h"
#include "Menu.h"
#include "Error.h"
#include "FindDialog.h"
#include "Document.h"
#include "RegularExp.h"
#include "ClipBoard.h"

static short CaretBits[]= {
#   include "images/Caret.image"
};  

const int cCaretHt = 4,
	  cCaretWd = 7;

static StaticBitmap CaretImage(Point(cCaretWd, cCaretHt), CaretBits);
static Point caretHotSpot(cCaretHt, 0);

inline void Swap(SelPoint &p1, SelPoint &p2)
{
    SelPoint tmp;

    tmp = p1; p1 = p2; p2 = tmp;
}

//----- TextView Methods --------------------------------------------------

MetaImpl(TextView, (I_B(updateSelection), I_B(inTextSelector), 
    I_I(caretState), I_CS(stopChars), I_O(Typeing),I_O(findChange), 
    I_O(scratchText), I_I(start.ch), I_I(start.line), I_P(start.viewp),
    I_I(end.ch), I_I(end.line), I_P(end.viewp)));

TextView::TextView(EvtHandler *eh, Rectangle r, Text *t, eTextJust m, eSpacing sp, 
		       bool doWrap, TextViewFlags fl, Point b, int id)
					    : (eh,r,t,m,sp,doWrap,fl,b,id) 
{
    Init();
}

void TextView::InitNew()
{
    View::InitNew();
    caretState= Off;
    inTextSelector= FALSE;
}  

void TextView::Init()
{
    PrivSetSelection(0,0,FALSE);
    Typeing = 0;
    stopChars = 0;
    updateSelection= TRUE;
    text->AddDependent(this);
    scratchText= text->GetScratchText(cMaxBatchedIns);
    caretState = Off;
    findChange= 0;
}

TextView::~TextView()
{
    if (text)
	text->RemoveDependent(this);
    SafeDelete(stopChars);
    if (findChange)
	findChange->Close();
    SafeDelete(findChange);
    SafeDelete(scratchText);
}

void TextView::Draw(Rectangle r)
{
    caretState= Off;  // secure, because if the caret is already visible
		      // drawing of the selection will be clipped of
    StaticTextView::Draw(r);
}

void TextView::Reformat()
{
    StaticTextView::Reformat();
    SetSelection(start.ch, end.ch, FALSE);
}

void TextView::DoHighlightSelection(HighlightState hs)
{
    if (caretState == hs)
	return;
    caretState= hs;
    if (!AnySelection() || (Caret() && inTextSelector))
	return;
    if (Caret())    
	DrawCaret(start.viewp, start.line, hs);
    else
	Invert(start.line, start.viewp, end.line, end.viewp);    
}

void TextView::SetNoSelection(bool redraw)
{
    if (redraw)
	DoHighlightSelection(Off);
    start.ch= end.ch= -1;
}

void TextView::PrivSetSelection(int s,int e, bool redraw)
{
    SelPoint p;
    s = max(0, min(s, text->Size()));
    e = max(0, min(e, text->Size()));
    if (redraw)
	DoHighlightSelection(Off);
    CharToPos(p.ch= s, &p.line, &p.viewp);
    start= p;
    if (e != s)
	CharToPos(p.ch= e, &p.line, &p.viewp);
    end= p;
    if (redraw) 
	DoHighlightSelection(On);
}

void TextView::SetSelection(int s,int e, bool redraw)
{
    DoneTypeing();
    PrivSetSelection(s, e, redraw);
}

void TextView::SetSelectionAndInvalidate(int s,int e)
{
    if (AnySelection())
	InvalidateRange(start.line, end.line);
    SetSelection(s, e, FALSE);
    InvalidateRange(start.line, end.line);
}

void TextView::SelectionAsString(byte *buf, int max)
{
    GetText()->CopyInStr(buf, max, start.ch, end.ch);
}

Text *TextView::SelectionAsText()
{
    return text->Save(start.ch, end.ch);
}

void TextView::RevealSelection()
{
    if (AnySelection()) {
	Point p(end.viewp.x+1, end.viewp.y + LineHeight(end.line));
	Rectangle r= NormRect(start.viewp, p).Expand(8);
	r.origin+= GetInnerOrigin();
	RevealRect(r, r.extent);
    }
}

Command *TextView::DoLeftButtonDownCommand(Point p, Token t, int clicks)
{
    SelPoint oldStart= start;
    SelPoint oldEnd= end;

    ResumeFormat();
    if (!Enabled())
	return gNoChanges;
    p -= GetInnerOrigin();
    if (clicks >= 2) {        
	DoHighlightSelection(Off);
	if (clicks >= 3)
	    return new ParagraphSelector(this, p);
	return new WordSelector(this, p);
    }

    if (t.Flags == eFlgShiftKey) {   
	SelPoint nextp;
	if (Caret()) 
	    DoHighlightSelection(Off);
	PointToPos(p, &nextp.viewp, &nextp.line, &nextp.ch);
	if (nextp.ch < (end.ch - start.ch)/2)  // extend at start
	    Swap(start, end);
	Invert(end.line, end.viewp, nextp.line, nextp.viewp);
	caretState= On;
	end= nextp;        
    } else {            
	DoHighlightSelection(Off);
	PointToPos (p, &start.viewp, &start.line, &start.ch);
	end= start;
    }
    if (t.Flags == (eFlgShiftKey|eFlgCntlKey) && clicks == 1)
	return new QuickPasteSelector(this, oldStart.ch, oldEnd.ch);
    else if (t.Flags == eFlgCntlKey && clicks == 1) 
	return new CopyDragPasteSelector(this, oldStart.ch, oldEnd.ch); 
    return new CharSelector(this);
}

void TextView::SetStopChars(char *stops)
{
    SafeDelete(stopChars);
    stopChars= strsave(stops);
}

const char *TextView::GetStopChars()
{
    return stopChars;
} 

Command *TextView::DoKeyCommand(int ch, Point, Token token)
{
    int nDelete= 0;
    Token t;
    bool newCmd;
    bool delSelection;       // delete the selection only

    if (TestFlag(eTextViewReadOnly) || !Enabled())
	return gNoChanges;
    if (token.Code == 'd' + 128 && gDebug) {
	Dump();
	return gNoChanges;
    }

    //---- map \r to to \n ???
    if (ch == '\r')
	ch= '\n';

    newCmd= (Typeing == 0 || Typeing->completed); // start new typing sequence?
 
    if (newCmd) {
	if (Typeing)
	    Typeing->SetFlag(eCmdDoDelete);
	Typeing= new TypeingCommand(this, cTYPEING, "typing");
    }

    delSelection= (newCmd && !Caret());
    if (ch != gBackspace || delSelection) {     
	for (;;) {
	    if (ch != gBackspace) {
		Typeing->AddChar();
		scratchText->Append(ch);
	    }
	    if (scratchText->Size() == cMaxBatchedIns)
		break;
	    t= gWindow->ReadEvent(0);
	    if (t.IsAscii() && (byte)t.Code != gBackspace 
		    // do not batch stopChars
		    && !(stopChars && index(stopChars, (byte)t.Code)) 
			&& ! TestFlag(eTextViewNoBatch))
		ch= (byte) t.Code;
	    else {
		gWindow->PushBackEvent(t);
		break;
	    } 
	}
	Paste(scratchText);
	scratchText->Empty();
    }
    else {
	for (;;) {
	    Typeing->DelChar();
	    nDelete++;
	    t= gWindow->ReadEvent(0);
	    if ((byte) t.Code != gBackspace) {
		gWindow->PushBackEvent(t);
		break;
	    } 
	} 
	DelChar(nDelete);
    }
    RevealSelection();
    return Typeing;    
}

Command *TextView::DoCursorKeyCommand(EvtCursorDir cd, Point p, Token)
{
    int charNo;

    if (!Enabled())
	return gNoChanges;
    switch (cd) {
    case eCrsLeft:
	charNo= start.ch-1;
	break;
    case eCrsRight:
	charNo= end.ch+1;
	break;
    case eCrsUp:
	ResumeFormat();
	charNo= CursorPos(start.ch, start.line, cd, p);
	break;
    case eCrsDown:
	ResumeFormat();
	charNo= CursorPos(end.ch, end.line, cd, p);
	break;
    }
    DoneTypeing();
    PrivSetSelection(charNo, charNo);
    RevealSelection();
    return gNoChanges;
}

int TextView::CursorPos(int at, int line, EvtCursorDir d, Point p)
{
    int charNo;
    Point basePoint,screenPoint;

    CharToPos (at, &line, &screenPoint);
    if (d == eCrsDown)
	line= min(nLines-1, line+1);
    else
	line= max(0, line-1);        
    basePoint= LineToPoint(line, TRUE) + Point(screenPoint.x, 0);
    PointToPos(basePoint, &p, &line, &charNo);
    return charNo;
}

Command *TextView::DoOtherEventCommand(Point, Token t)
{
    if (t.Code == eEvtLocMove)
	ResumeFormat();
    return gNoChanges;
}

GrCursor TextView::GetCursor(Point)
{
    return eCrsIBeam;
}

Command *TextView::DoIdleCommand()
{
    ResumeFormat(TRUE);
    return View::DoIdleCommand();
}

void TextView::DoneTypeing()
{
    if (Typeing)
	Typeing->completed= TRUE;
    text->ResetCurrentStyle();
    ResumeFormat();
}

void TextView::TypeingDeleted()
{
    Typeing= 0;  
}

bool TextView::DeleteRequest(int,int)
{
    return TRUE;
}

void TextView::Cut()
{
    if (!DeleteRequest(start.ch, end.ch) || !AnySelection())
	return;
    DoHighlightSelection(Off);
    updateSelection= FALSE;            
    text->Cut(start.ch, end.ch);
    updateSelection= TRUE; 
    PrivSetSelection(start.ch, start.ch, FALSE);
}

void TextView::Copy(Text *t)
{
    text->Copy(t, start.ch, end.ch);
}

void TextView::Paste (Text *t)
{ 
    if ((!Caret() && !DeleteRequest(start.ch, end.ch)) || !AnySelection())
	return;
    DoHighlightSelection(Off);
    // minor hack: if text is appended on the last line, 
    // fake last line to reach a mark 
    if (wrap && start.line == nLines-1) 
	MarkAtLine(start.line)->len= cMaxInt;
    updateSelection= FALSE;
    text->Paste(t, start.ch, end.ch);
    updateSelection= TRUE;
    int p= start.ch+t->Size();
    PrivSetSelection(p, p, FALSE);       
}

Command *TextView::InsertText(Text *t)
{
    if (TestFlag(eTextViewReadOnly) || !Enabled())
	return gNoChanges;

    bool newCmd;
    newCmd= (Typeing == 0 || Typeing->completed); // start new typing sequence?

    if (newCmd) {
	if (Typeing)
	    Typeing->SetFlag(eCmdDoDelete);
	Typeing= new TypeingCommand(this, cTYPEING, "typing");
    }

    Typeing->AddChar(t->Size());
    Paste(t);
    return Typeing;
}

void TextView::DelChar (int n)
{
    int newStart= max(0, start.ch-n);
    int changedLine= CharToLine(newStart);

    if (newStart == start.ch)
	return;
    if (newStart == 0 && start.ch == 0)
	return;
    if (DeleteRequest(newStart, start.ch)) {
	DoHighlightSelection(Off);
	updateSelection= FALSE;   
	text->Cut(newStart, start.ch);
	updateSelection= TRUE;
	PrivSetSelection(newStart, newStart, FALSE);       
    }
}

Text *TextView::SetText(Text *t)
{
    Text *old= text;
    old->RemoveDependent(this);
    bool anySel= AnySelection();
    PrivSetSelection(0, 0, FALSE); // reset old selection
    text= t;    // should remove possible reference from a command object ???
    SafeDelete(nextc);
    nextc= text->GetIterator();
    SafeDelete(scratchText);
    scratchText= text->GetScratchText(cMaxBatchedIns);
    text->AddDependent(this);
    Reformat();  
    ChangedWhat((void*)eReplacedText);
    DoneTypeing();
    PrivSetSelection(0, 0, FALSE); // set new selection
    if (!anySel)
	SetNoSelection(FALSE);  
    Typeing= 0;
    return old;
}       

void TextView::SetString(byte *str, int len)
{
    bool anySel= AnySelection();
    PrivSetSelection(0, 0, FALSE); 
    text->ReplaceWithStr(str, len);
    Reformat();  
    ChangedWhat((void*)eReplacedText);
    DoneTypeing();
    PrivSetSelection(0, 0, FALSE); 
    if (!anySel)
	SetNoSelection(FALSE);  
    Typeing= 0;
}
 
void TextView::DoUpdate(Object *op, void *vp)
{
    int d= 0, d1= 0;
    if (op == text) {
	TextChanges *tc= (TextChanges *)vp;
	Mark *m= 0;
	if (AnySelection())
	    m= MarkAtLine(start.line);
	int fl, tl; // lines corresponding from, to
	// optimize call to CharToLine 
	if (m && tc->from >= m->pos && 0 < m->pos - tc->from + m->len)
	    fl= start.line;
	else {
	    fl= CharToLine(tc->from);
	}
	if (tc->from == tc->to)
	    tl= fl;
	else {
	    m= 0;
	    if (AnySelection())
		m= MarkAtLine(end.line);
	    if (m && tc->to >= m->pos && tc->to < m->pos + m->len)
		tl= end.line;
	    else {
		tl= CharToLine(tc->to);
	    }
	}
	switch (tc->what) {
	case eTextChangedRange:
	    ChangedAt(fl, tc->from, TRUE, tl);
	    if (updateSelection && AnySelection())
		PrivSetSelection(tc->from, tc->to, TRUE);
	    break;

	case eTextDeleted:
	    marks->Cut(tc->from, tc->to - tc->from);
	    d = DeleteLines (fl, tl);
	    ChangedAt(fl, tc->from, d > 0);
	    if (updateSelection && AnySelection())
		PrivSetSelection(tc->from, tc->from, TRUE);
	    break;

	case eTextReplaced:
	    if (tc->from != tc->to) {
		marks->Cut(tc->from, tc->to - tc->from);
		d1= DeleteLines(fl, tl);
	    }
	    marks->Paste(tc->from, tc->size); 
	    d = FormatInsertedText(fl, tc->from, tc->size);
	    ChangedAt(fl+d, tc->from, d > 0 || d1 > 0);
	    if (updateSelection && AnySelection()) {
		int p= tc->from+tc->size;
		PrivSetSelection(p, p, TRUE);   
	    }    
	    break;
	}
	ChangedWhat((void*)eText); // notify dependents of the textview   
    }
}

bool TextView::PrintOnWhenDependent(Object *from)
{
    return GetText() != from;
}

istream& TextView::ReadFrom(istream &s)
{
    StaticTextView::ReadFrom(s);
    GetText()->AddDependent(this);
    scratchText= text->GetScratchText(cMaxBatchedIns);
    return s;
}

void TextView::NormSelection ()
{
    if (start.ch > end.ch) 
	Swap(start, end);
}

void TextView::drawCaretImage (void *vp, void *vline, void *vhs)
{
    DrawCaretImage(*(Point*)vp, *(int*)vline, *(HighlightState*)vhs);
}

void TextView::DrawCaret(Point p, int line, HighlightState hs)
{
    ShowInAllFrames(&TextView::drawCaretImage, this, &p, &line, &hs);
}

void TextView::DrawCaretImage(Point p, int line, HighlightState)
{
    int bh= BaseHeight(line);
    p+= GetInnerOrigin();

    if (LineHeight(line) - bh >= cCaretHt) {
	Rectangle r(p.x-cCaretHt, p.y+bh, cCaretWd, cCaretHt);
	GrInvertBitMap(r, &CaretImage);
    }
    GrSetPenNormal();
    GrSetPenSize(1);
    GrInvertLine(Point(p.x-1, p.y), Point(p.x-1, p.y+bh-1));
}

int TextView::FormatInsertedText(int at, int startCh, int n)
{
    int nl, end, upto, d;
    int atCh= StartLine(at);
    LineMark *m;

    interruptable= FALSE;
    // temporary replacement of the array with the line marks
    ObjArray *saved= lines;
    MarkList *savedMarks= marks;
    marks= new MarkList;
    lines= new ObjArray(32, at);

    upto= startCh + n;
    if (wrap)  // format the line structure into the temporary line array
	end= Format(at, atCh, nl, upto);
    else 
	end= NoWrapFormat(at, atCh, nl, upto);
    d= max(0, end - at);
    swap(&saved, &lines);
    swap(&savedMarks, &marks);
    if (d > 0) {
	for (int i= nLines-1; i > at; i--) { // make room for new lines
	    m= MarkAtLine(i);
	    MarkLine(i+d, m->pos, m->pos+m->len, &m->ld);
	} 
	for (i= at; i <= at + d; i++) {  // insert the new formatted lines
	    m= (LineMark*)(*saved)[i];
	    MarkLine(i, m->pos, m->pos+m->len, &m->ld);
	}
	MarkLineAsChanged(i-1);           // mark the last line as changed
    }
    nLines+= d; 
    savedMarks->FreeAll();
    delete savedMarks;
    delete saved; 
    InvalidateRange(at, at+d);
    interruptable= TRUE;
    return d;
}

int TextView::DeleteLines(int from, int to)
{
    int d= to - from;
    if (d > 0) {
	for (int i= from + 1; i < nLines-d; i++) {
	    LineMark *m= MarkAtLine(i+d);
	    MarkLine(i, m->pos, m->pos+m->len, &m->ld);
	} 
	nLines-= d;
	MarkLineAsChanged(from);
    }
    return d;
}
 
void TextView::Dump()
{
    int i;
    StaticTextView::Dump();
    cerr << "Selection: start = " << start.ch << "," << start.line;
    cerr << " end = " << end.ch << "," << end.line NL;
    for (i = start.ch; i < end.ch; i++)
	cerr.put((*text)[i]);
    cerr NL;
    if (caretState == On)
	cerr << "caretState = On\n";
    else
	cerr << "caretState = Off\n";
}

//---- menu related commands -----------------------------------------------

void TextView::DoCreateMenu(Menu *mp)
{
    View::DoCreateMenu(mp);
    if (!TestFlag(eTextNoFind)) 
	mp->InsertItemsAfter(cLASTEDIT, "select all",     cSELECTALL,
					"find/change...", cFIND,
					0);
    if (gDebug)
	mp->InsertItemAfter(cLASTEDIT, "inspect", cINSPECT);
}

bool TextView::HasSelection()
{
    if (! Caret())
	return TRUE;
}

void TextView::SelectionToClipboard(char *type, ostream &os)
{
    Text *t= SelectionAsText();
    if (t) {
	if (strcmp(type, cClipEt) == 0)
	    os << t SP;
	else if (strcmp(type, cClipAscii) == 0)
	    t->PrintOnAsPureText(os);
	delete t;
    }
}

Command *TextView::PasteData(char *type, istream &s)
{
    if (strcmp(type, cClipEt) == 0) {
	Object *op= 0;
	s >> op;
	if (op && op->IsKindOf(Text))
	    return new PasteCommand(this, (Text*)op);
    } else if (strcmp(type, cClipAscii) == 0)
	return new PasteCommand(this, new GapText(s.bufptr()));
    return gNoChanges;
}

void TextView::DoSetupMenu(Menu *mp)
{
    View::DoSetupMenu(mp);
    if (!GetReadOnly()) {
	if (!Caret())
	    mp->EnableItem(cCUT);
	if (AnySelection())
	    mp->EnableItem(cPASTE);
    }
    if (AnySelection())
	mp->EnableItems(cSELECTALL, cFIND, 0);
    if (gDebug)
	mp->EnableItem(cINSPECT);
}

Command *TextView::DoMenuCommand(int cmd)
{
    switch (cmd) {
    case cCUT:
	View::DoMenuCommand(cmd);
	return new CutCopyCommand(this, cCUT);
    case cCOPY:
	View::DoMenuCommand(cmd);
	return new CutCopyCommand(this, cCOPY);
    case cSELECTALL:
	SelectAll();
	break;
    case cINSPECT:
	GetText()->Inspect();
	break;
    case cFIND:
	if (findChange == 0) {
	    findChange= new FindDialog("Find/change", this);
	    GetDocument()->AddWindow(findChange->GetWindow());
	}
	findChange->ShowOnWindow(GetWindow());
	return gNoChanges;
    default:
	return View::DoMenuCommand(cmd);
    }
    return gNoChanges;
}

bool TextView::SelectRegExpr(RegularExp *rex, bool forward)
{
    int selStart, selEnd, pos, matched;

    GetSelection(&selStart, &selEnd);
    if (forward)
	pos= GetText()->Search(rex, &matched, selEnd);
    else
	pos= GetText()->Search(rex, &matched, max(0,selStart-1), cMaxInt, FALSE);

    if (pos != -1) {
	SetSelection(pos, pos+matched);
	return TRUE;
    }
    return FALSE;
}

void TextView::Home()
{
    SetSelection(0, 0);
}

void TextView::Bottom()
{
    SetSelection(cMaxInt, cMaxInt);
}

void TextView::SelectAll()
{
    SetSelection(0, text->Size());
}

void TextView::Enable(bool b, bool redraw)
{
    StaticTextView::Enable(b, redraw);
    if (! Enabled())
	SetNoSelection(TRUE);
} 
