#include "Storage.h"
#include "System.h"
#include "Error.h"
#include "FixedSizeStorage.h"

#include "MALLOC/storage.h"

const int cMaxSize= 300;
bool gMemStatistics= FALSE;
static bool init= FALSE;
static void *smallestAddress= 0;
static int maxA= 0;
static int ASizes[cMaxSize];
static int FSizes[cMaxSize];
static int totalA, totalF;
extern bool stinit;

static char *cUnreasonableAddress   = "unreasonable address (0x%x)",
	    *cSizeNull              = "size < 0",
	    *cPtrNull               = "ptr == 0";
	    
static void StorageInit()
{
    init= TRUE;
    storage_init();
    // smallestAddress= Malloc(1);
}

inline void InitStorage()
{
    if (!init)
	StorageInit();
}

static void EnterStat(int size)
{
    if (size >= cMaxSize)
	ASizes[cMaxSize-1]++;
    else
	ASizes[size]++;
    totalA+= size;
}

static void RemoveStat(int size)
{
    if (size >= cMaxSize)
	FSizes[cMaxSize-1]++;
    else
	FSizes[size]++;
    totalF+= size;
}

void* _new(long size)
{
    return Malloc(size);
}

void _delete(void *p)
{
    Free(p);
}

void *Malloc(unsigned size)
{
    InitStorage();
    if (size < 0)
	Fatal("Malloc", cSizeNull);
    maxA= max(maxA, size);
    EnterStat(size);
    return storage_malloc(size);
}

void *Alloc(unsigned size)
{
    InitStorage();
    if (size < 0)
	Fatal("Alloc", cSizeNull);
    maxA= max(maxA, size);
    EnterStat(size);
    return storage_malloc_atomic(size);
}

byte *NewBytes(unsigned size)
{
    InitStorage();
    if (size < 0)
	Fatal("NewBytes", cSizeNull);
    maxA= max(maxA, size);
    EnterStat(size);
    return (byte*) storage_malloc_atomic(size);
}

char *NewChars(unsigned size)
{
    InitStorage();
    if (size < 0)
	Fatal("NewChars", cSizeNull);
    maxA= max(maxA, size);
    EnterStat(size);
    return (char*) storage_malloc_atomic(size);
}

void Free(void *ptr)
{
    if (ptr) {
	if (ptr < smallestAddress)
	    Fatal("Free", cUnreasonableAddress, ptr);
	int size= storage_size(ptr);
	if (size < 0 || size > maxA)
	    Fatal("Free", "unreasonable size (%d)", size);
	RemoveStat(size);
	storage_free(ptr, size);
    }
}

void *Realloc(void **ptr, unsigned size)
{
    InitStorage();
    maxA= max(maxA, size);
    if (ptr) {
	if (*ptr == 0) {
	    EnterStat(size);
	    *ptr= storage_malloc(size);
	} else if (ptr < smallestAddress) {
	    Fatal("Realloc", cUnreasonableAddress, ptr);
	} else {
	    int oldsize= storage_size(*ptr);
	    if (size > oldsize) {
		RemoveStat(oldsize);
		EnterStat(size);
		*ptr= storage_realloc(*ptr, oldsize, size);
	    }
	}
    } else
	Fatal("Realloc", cPtrNull);
    return *ptr;
}

void *Expand(void **ptr, unsigned size)
{
    InitStorage();
    maxA= max(maxA, size);
    if (ptr) {
	if (*ptr == 0) {
	    EnterStat(size);
	    *ptr= storage_malloc(size);
	} else if (ptr < smallestAddress)
	    Fatal("Expand", cUnreasonableAddress, ptr);
	else {
	    int oldsize= storage_size(*ptr);
	    if (size > oldsize) {
		RemoveStat(oldsize);
		EnterStat(size);
		*ptr= storage_realloc(*ptr, oldsize, size);
	    }
	}
    } else
	Fatal("Realloc", cPtrNull);
    return *ptr;
}

void PrintStorageStatistics()
{
    if (! gMemStatistics)
	return;
    fprintf(stderr, "\n");
    fprintf(stderr, "  Size Alloc  Free  Diff\n");
    fprintf(stderr, "------------------------\n");
    for (int i= 0; i < cMaxSize; i++)
	if (ASizes[i] != FSizes[i])
	    fprintf(stderr, "%6d%6d%6d%6d\n", i, ASizes[i], FSizes[i],
							ASizes[i]-FSizes[i]);
    fprintf(stderr, "\n\ntotalA: %d  totalF: %d totalD: %d\n", totalA, totalF,
							totalA-totalF);
    fflush(stderr);
}

ONENTRY(Storage)
{
    InitStorage();
}

ONEXIT(Storage)
{
    /*
    if (gMemStatistics) {
	PrintStorageStatistics();
	PrintFixedStorageStatistics();
    }
    */
}
