//$StaticTextView,LineMark$
overload Swap;

#include "StaticTextView.h"
#include "String.h"
#include "View.h"
#include "BlankWin.h"
#include "FixedSizeStorage.h"

Rectangle gFitRect(cFit, cFit);
bool formInterrupted = FALSE;

//---- class LineMark ------------------------------------------------------

MetaImpl0(LineMark);

FIXED_STORAGE(LineMark, 400);

LineMark::LineMark(LineDesc ldesc,int pos = 0, int len= 0, eMarkState s= 0)
							    : (pos, len, s)
{
    FIXED_SIZE_ALLOC(LineMark);
    ld = ldesc; 
}

LineMark::~LineMark()
{
    FIXED_SIZE_DEALLOC(LineMark);
}

ostream& LineMark::DisplayOn (ostream&s)
{ 
    Mark::DisplayOn(s);
    return s << ld.lnAscent << "/" << ld.lnHeight; 
}

//---- global functions --------------------------------------------------

int TextViewLineHeight(FontPtr fd, eSpacing sp)
{ 
    return ((int) sp) * fd->Spacing() / 2;
}

//----- StaticTextView Methods --------------------------------------------------

MetaImpl(StaticTextView, (I_O(text), I_I(just), I_I(spacing), I_B(wrap),
    I_B(drawViewBorder), I_B(vertExtend), I_B(horExtend), I_P(border),
    I_I(dirtyAt), I_B(interruptable), I_I(nLines), I_O(lines), I_O(marks)));

StaticTextView::StaticTextView(EvtHandler *eh, Rectangle r, Text *t, eTextJust m, 
	  eSpacing sp, bool doWrap, TextViewFlags fl, Point b, int id) : (eh,r,id) 
{
    Init(r, t, m, sp, doWrap, fl, b);
}   

void StaticTextView::Init(Rectangle r, Text* t, eTextJust ej, eSpacing sp, 
			bool doWrap, TextViewFlags flags, Point itsBorder)
{
    lines = new ObjArray(32);
    contentRect = r;
    text = t;
    nextc = text->GetIterator();
    just = ej;
    spacing = sp;
    wrap = doWrap;
    drawViewBorder = FALSE;
    marks = new MarkList;
    LineDesc ld;
    lines->AtPut(0,new LineMark(ld)); //dummy
    marks->Add((*lines)[0]);    
    vertExtend = (contentRect.extent.y == cFit);
    horExtend = (contentRect.extent.x == cFit);
    border = itsBorder;
    if (horExtend)
	wrap = FALSE;
    nLines = 0;
    dirtyAt = -1;
    fixedSpacing.FromFont(text->GetFont());
    interruptable = FALSE;
    ChangedAt(0);
    interruptable = TRUE;

    if ((flags & eTextViewVisMode) == eTextViewVisMode)
	SetFlag(eTextViewVisMode);
    if ((flags & eTextViewReadOnly) == eTextViewReadOnly)
	SetFlag(eTextViewReadOnly);
    if ((flags & eTextFormPreempt) == eTextFormPreempt)
	SetFlag(eTextFormPreempt);
    if ((flags & eTextFixedOrigin) == eTextFixedOrigin)
	SetFlag(eTextFixedOrigin);
    if ((flags & eTextViewNoBatch) == eTextViewNoBatch)
	SetFlag(eTextViewNoBatch);
    if ((flags & eTextFixedSpacing) == eTextFixedSpacing)
	SetFlag(eTextFixedSpacing);
}

bool StaticTextView::IsJustified(int endOfLine)
{
    return (just == eJustified && endOfLine != text->Size() &&
	    (*text)[endOfLine-1] != '\n' &&
	    (*text)[endOfLine-1] != '\r');
}

StaticTextView::~StaticTextView()
{
    if (lines)
	lines->FreeAll();
    SafeDelete(marks);
    SafeDelete(lines);
    SafeDelete(nextc);
}

void StaticTextView::Draw(Rectangle r)
{
    int startAt = PointToLine(r.origin-GetInnerOrigin()); // r is in View coords
    short textpatid= port->textpatid;    // to be modified with an access functions

    if (startAt >= nLines)
	return;
    Point p = LineToPoint(startAt,TRUE) + GetInnerOrigin();
    Rectangle lineRect(GetInnerOrigin().x,p.y-BaseHeight(startAt),
				   GetInnerExtent().x,LineHeight(startAt));

    GrSetTextMode(eRopCopy);
    if (! Enabled())
	GrSetTextPattern(ePatGrey50);

    for (int i = startAt; ; i++ ) {
	if (!r.Intersects(lineRect))
	    break;
	DrawLine (p,i,lineRect,r);
	if (i >= nLines-1)
	    break;
	p.y+= (LineHeight(i)-BaseHeight(i)) + BaseHeight(i+1);
	lineRect.origin.y+= LineHeight(i); 

    }        
    if (drawViewBorder) {
	GrSetPenPattern(ePatGrey50);
	GrSetPenSize(2);
	GrStrokeRect (Rectangle (contentRect.origin,contentRect.extent));
    }
    GrSetTextPattern(textpatid);
}

void StaticTextView::DrawLine (Point p,int i,Rectangle, Rectangle rr)
{
    int start, end;

    start= StartLine(i);
    end= EndLine(i);
    bool justified = IsJustified(end);
    p.x+= FirstCharPos (start, end);
    if (Isspace((*text)[end-1]) && !TestFlag(eTextViewVisMode))     
	end = max(start,--end);
    GrTextMoveto(p);
    if (justified)
	text->DrawTextJust(start,end,GetInnerExtent().x,p,rr);
    else
	text->DrawText(start,end,rr);
}

void StaticTextView::SetFont(FontPtr fd)
{
    text->SetFont(fd);
    Reformat();
}

void StaticTextView::SetSpacing(eSpacing sp)
{
    spacing = sp;
    Reformat();
}

void StaticTextView::SetJust(eTextJust m)
{
    just = m;
    Reformat();
}

void StaticTextView::SetVisibleMode(bool m)
{
    SetFlag(eTextViewVisMode, m);
    Reformat();
}

bool StaticTextView::GetVisibleMode()
{ 
    return TestFlag(eTextViewVisMode); 
}

void StaticTextView::SetWordWrap(bool m)
{   
    wrap = m;
    Reformat();
}

bool StaticTextView::GetWordWrap()
{ 
    return wrap; 
}

void StaticTextView::SetReadOnly(bool m)
{
    SetFlag(eTextViewReadOnly, m);
}

bool StaticTextView::GetReadOnly()
{ 
    return TestFlag(eTextViewReadOnly); 
}

void StaticTextView::SetNoBatch(bool m)
{
    SetFlag(eTextViewNoBatch, m);
}

bool StaticTextView::GetNoBatch()
{ 
    return TestFlag(eTextViewNoBatch); 
}

Text *StaticTextView::SetText(Text *t)
{
    Text *old = text;
    text = t;   
    nextc = text->GetIterator();
    Reformat();  
    ChangedWhat((void*)eReplacedText);
    return old;
}

void StaticTextView::SetString(byte *str, int len)
{
    text->ReplaceWithStr(str, len);
    Reformat();  
    ChangedWhat((void*)eReplacedText);
}

int StaticTextView::NumberOfLines()
{
    return nLines;
}

//---- format updates an array of line marks -- does not touch any other field

int StaticTextView::Format (int fromLine, int start, int &numLines, int upto, int minUpto)
{
    int width= GetInnerExtent().x;
    int end, ch, nWords = 0, wx = 0,cx = 0;
    int startLine = fromLine;
    LineDesc maxld,ld;
    int lastPeek = startLine;

    nextc->Reset(text,start,upto);

    while (TRUE) {            
	// check whether formatting process should be preempted
	if (interruptable && 
	    TestFlag(eTextFormPreempt) && 
		fromLine > minUpto &&
		    fromLine - lastPeek > cMaxFormat) {
	    Token t= gWindow->ReadEvent(0);
	    if (t.Code != eEvtNone) {
		gWindow->PushBackEvent(t);
		// suspend formatting process
		if (dirtyAt != -1) 
		    dirtyAt= min (fromLine, dirtyAt);
		else 
		    dirtyAt= fromLine;
		formInterrupted = TRUE;
		MarkLineAsChanged(fromLine);        
		return fromLine-1;
	    } else
		lastPeek = fromLine;
	}

	ch = nextc->Token(&wx,&ld);
	// handle special case if one word has to be folded on several lines
	if (cx + wx > width && nWords == 0) {
	    end = nextc->GetPos();
	    BreakWord (width,&start,&fromLine,&end,&cx,&maxld);
	    wx = 0;
	}

	switch (ch) {
	case cEOT: 
	    maxld.Max(ld);
	    if (!MarkLine(fromLine,start,nextc->GetPos(),&maxld)
			    && fromLine >= minUpto)
		return fromLine;
	    numLines= fromLine+1;
	    return fromLine;

	case '\t':
	    wx = text->Tabulate(cx);  // ---> no break
	case ' ': case '\n': case '\r':            
	    maxld.Max(ld);
	    if (ch == '\n' || ch == '\r' || (cx + wx > width)) {
		end = nextc->GetPos();
		if (!MarkLine(fromLine,start,end,&maxld) && fromLine >= minUpto) 
		    return (fromLine-1);
		fromLine++;
		start = end;
		cx = 0;
		nWords = 0;
	    } 
	    else {
		cx += wx;
		nWords++;
	    }
	    break;

	default:
	    if (cx + wx > width) {  // wordwrap
		end = nextc->GetLastPos();
		if (!MarkLine(fromLine,start,end,&maxld) && fromLine >= minUpto)  // line did not change
		    return (fromLine-1);
		maxld = ld;
		fromLine++;
		start = end;
		nWords = 0;
		cx = wx;
	    } else {
		maxld.Max(ld);
		cx += wx;
		nWords++;
	    }
	    break;
	}
    }
}

void StaticTextView::ResumeFormat(bool isInterruptable)
{
    register int i;

    if (dirtyAt == -1)
	return;
    interruptable = isInterruptable;
    formInterrupted = FALSE;
    for (i = dirtyAt; i < nLines; i++)
	if(MarkAtLine(i)->state != eStateNone) {
	    ChangedAt(i, 0, FALSE,i);
	    if (formInterrupted)
		break;
	}
    interruptable = TRUE;
    if (formInterrupted)
	dirtyAt = i;
    else
	dirtyAt = -1;
}

void StaticTextView::Reformat()
{
    nLines = 0;
    interruptable = FALSE;
    ChangedAt(0);
    interruptable = TRUE;
}

int StaticTextView::NoWrapFormat (int fromLine, int start, int &numLines, int upto, int minUpto)
{
    int end;
    LineDesc ld,maxld;

    nextc->Reset(text,start,upto);

    while ((end = nextc->Line(&ld)) != cEOT) {
	maxld.Max(ld);
	if (!MarkLine(fromLine,start,end,&maxld) && fromLine >= minUpto)// line did not change
	    return (fromLine-1);
	fromLine++;
	start = end;
    }
    numLines= fromLine;
    return fromLine-1;
}

//---- mark a line and return whether line changed

bool StaticTextView::MarkLine (int line,int start,int end,LineDesc *ld)
{
    LineMark *m = 0;
    bool markChanged;

    if (TestFlag(eTextFixedSpacing)) {
	LineDesc tmp= fixedSpacing;
	fixedSpacing.Max(*ld);
	if (!tmp.IsEqual(fixedSpacing))
	    ForceRedraw();
	*ld= fixedSpacing;    
    }

    if (line < lines->Size())   
	m = MarkAtLine(line);

    if (m) {
	markChanged = m->HasChanged(start,end-start) || line >= nLines;
	m->ChangeMark(start,end-start,*ld);
	ld->Reset();
	return markChanged;
    }    
    m = new LineMark(*ld,start,end - start);
    ld->Reset();
    marks->Add(m);
    lines->AtPutAndExpand(line,m);
    return TRUE;
}

void StaticTextView::MarkLineAsChanged (int line)
{
    LineMark *m = MarkAtLine(max(0,min(line,nLines-1)));
    m->state = eStateChanged;   
}

void StaticTextView::SetExtent(Point p)
{
    if (p != contentRect.extent) {
	View::SetExtent(p);
	if (horExtend)
	    wrap= FALSE;
	Reformat();
    }
}

Metric StaticTextView::GetMinSize()
{
    Reformat();
    return Metric(GetExtent(), Base());
}

int StaticTextView::Base()
{
    return LineToPoint(0, TRUE).y+border.y;
}

void StaticTextView::Fit()
{
    Point newExtent(contentRect.extent);
    Point newOrigin(contentRect.origin);
    register int s, e;

    if (vertExtend) 
	newExtent.y = LineToPoint(nLines).y + 2*border.y;

    if (horExtend) {
	newExtent.x = 0;
	for (int i = 0; i < nLines; i++) {
	    s= StartLine(i);
	    e= EndLine(i);
	    newExtent.x = max(newExtent.x, text->TextWidth(s,e));
	    if (!TestFlag(eTextFixedOrigin)) {
		// adjust origin
		switch (just) {
		case eCenter:
		    newOrigin.x = contentRect.origin.x + 
				  (contentRect.extent.x-newExtent.x)/2;
		    break;
		case eRight:
		    newOrigin.x = contentRect.origin.x + 
				  (contentRect.extent.x-newExtent.x);
		    break;
		}
	    }
	}
	newExtent.x += 2*border.x;
    }

    // the height of the view is at least the height of one line
    newExtent.y = max(newExtent.y,
		      TextViewLineHeight(text->GetFont(), spacing)+2*border.y);

    // invalidate difference Rects if necessary
    if (newExtent.x < contentRect.extent.x || newExtent.y < contentRect.extent.y) {
	Rectangle r[4];
	// expanded by 4,0 to remove the caret too
	int n = Difference(r,contentRect.Expand(Point(4,0)),
					    Rectangle(newOrigin,newExtent));
	for (int i = 0; i < n; i++) 
	    InvalidateRect(r[i]);
    }   
    if (newOrigin != contentRect.origin) {
	View::SetOrigin(newOrigin);
	ChangedWhat((void*)eOrigin);
    }
    if (newExtent != contentRect.extent) {
	View::SetExtent(newExtent);
	if (TestFlag(eTextFixedOrigin)) 
	   InvalidateRect(contentRect);
	ChangedWhat((void*)eExtent);
    }    
}

Point StaticTextView::LineToPoint (int n,bool basePoint, bool relative)
{
    Point p;

    n = range(0,nLines,n);
    for (int i = 0; i < n; i++)
	p.y += LineHeight(i);

    if (basePoint)
	p.y += BaseHeight(n);
    if (!relative)
	p += GetInnerOrigin();
    return p;
}

int StaticTextView::PointToLine(Point p) // p is in coordinates relative to contentRect
{
    int line,y = 0;
    p.y = max(0,p.y);
    for (line = 0; line < nLines; line++) {
	if (y + LineHeight(line) > p.y)
	    break;
	y += LineHeight(line);
    }
    return (min(nLines,line));
}

//---- map a point in view coordinates to line and character number

void StaticTextView::PointToPos(Point p,Point *viewPos,int *lineNo,int *charNo)
{
    int start,end;
    int fx, cx;
    int l = *lineNo;

    fx= cx= 0;

    l = PointToLine(p);
    if (l >= nLines) {
	l = nLines-1;
	p.x = contentRect.extent.x;    // set to end of line
    }
    if (l < 0) {
	l = 0;
	p.x = 0;            // set to start of line
    }
    start  = StartLine(l);
    end    = EndLine(l);

    fx = FirstCharPos (start,end);

    bool justified = IsJustified(end);

    // make new lines at end of line not selectable
    if (end - start) {
	if ((*text)[end-1] == '\n' || (*text)[end-1] == '\r')
	    end--;
	if (wrap && (*text)[end-1] == ' ')
	    end--;
    }
    if (justified) 
	text->JustifiedMap(start, end, GetInnerExtent().x, end, p.x-fx, charNo, &cx); 
    else
	text->Map(start, end, end, p.x-fx, charNo, &cx);
    *lineNo = l;
    *viewPos = Point(cx+fx,LineToPoint(l).y);
}

int StaticTextView::CharToLine(int ch)
{
    int line;

    for (line = 0; line < nLines; line++) {
	if (EndLine(line) > ch) 
	    break;
    }
    return max(0,min(line,nLines-1));
}

void StaticTextView::CharToPos (int charNo,int *lineNo,Point *viewPos, bool relative)
{
    int line,ch;
    int start,end,x;
    Point p(0);

    ch = min(max(0,charNo),text->Size()); // establish 0 <= ch <= size
    line = CharToLine(ch);
    p.y = LineToPoint(line).y;
    if (line >= nLines && line > 0) { // beyound end of text
	p.y-= LineHeight(line);
	line = max(0,nLines-1);
    }
    start  = StartLine(line);
    end    = EndLine(line);

    //-- find character position    
    p.x = FirstCharPos (start,end);

    bool justified = IsJustified(end);
    if (justified) 
	text->JustifiedMap(start, end, GetInnerExtent().x, charNo, cMaxInt, 0, &x);
    else
	text->Map(start,end,charNo,cMaxInt,0,&x);
    viewPos->x = p.x + x;
    viewPos->y = p.y;
    if (!relative)
	*viewPos += GetInnerOrigin();
    *lineNo = line;
}

//---- highlight the characters in the given range -----------------------

void StaticTextView::highlight3(void *nv, void *rev)
{
    int n= *(int*)nv;
    Rectangle *rv= (Rectangle*)rev;
    for (int i= 0; i < n; i++)
	GrInvertRect(rv[i]);
}

int StaticTextView::SelectionRects(Rectangle *rv, int from, Point fp, int to, Point tp)
{
    if((from == to && fp.x > tp.x) || from > to) { // normalize range
	swap(from, to);
	Swap(fp, tp);   
    }

    if (from == to) {
	rv[0]= Rectangle (LineToPoint(from)+Point (fp.x,0),
		       Point(tp.x-fp.x,LineHeight(from)));
	rv[0] += GetInnerOrigin();
	return 1;
    } 
    rv[0]= Rectangle(LineToPoint(from)+ Point(fp.x,0),
		  Point(GetInnerExtent().x-fp.x,LineHeight(from)));
    rv[0] += GetInnerOrigin();
    rv[1]= Rectangle(LineToPoint(from + 1), Point(GetInnerExtent().x,
		  LineToPoint(to).y - LineToPoint(from + 1).y));
    rv[1] += GetInnerOrigin();
    rv[2]= Rectangle(LineToPoint(to), Point(tp.x,LineHeight(to)));
    rv[2] += GetInnerOrigin();
    return 3;
}

void StaticTextView::Invert (int from, Point fp, int to, Point tp)
{    
    Rectangle rv[3];    
    int n= SelectionRects(rv, from, fp, to, tp);
    ShowInAllFrames(&StaticTextView::highlight3, this, &n, (void*)&rv[0]);
}

int StaticTextView::FirstCharPos (int from,int to)
{
    int px = 0;

    //---- strip trailing whitespace, blank, tab or new line
    if (Isspace((*text)[to-1]))
	to = max(0,--to);

    switch (just) {
    case eCenter:
	px = (GetInnerExtent().x - text->TextWidth(from,to))/2;
	break;
    case eRight:
	px = (GetInnerExtent().x - text->TextWidth(from,to));
	break;
    }
    return px;
}

void StaticTextView::ChangedAt(int line, int, bool redrawAll, int minUpto)
{
    int start,to;
    LineMark::lineChanged = FALSE;
    if (wrap) {
	start = max(0,line-1);
	to = Format(start, StartLine(start), nLines, cMaxInt, minUpto);
	if (to < line && to != nLines-1)  // no backward propagation
	    to = Format(start= line, StartLine(line), nLines, cMaxInt, minUpto);
    }
    else {
	start = line;
	to = NoWrapFormat(line, StartLine(line), nLines, cMaxInt, minUpto);
    }
    if (vertExtend || horExtend)
	Fit(); 
    if (redrawAll || LineMark::lineChanged)
	to = nLines-1;
    InvalidateRange(start,max(to,start));
}

void StaticTextView::InvalidateRange(int from, int to)
{
    Point p,t;
    p = LineToPoint(from);
    t = Point (contentRect.extent.x,LineToPoint(to).y + LineHeight(to));
    Rectangle r = NormRect(p,t);
    r += GetInnerOrigin();
    if (from == 0)  { // consider border
	r.origin.y -= border.y;
	r.extent.y += border.y;
    }
    if (to == nLines-1 || to == 0) 
	r.extent.y += border.y;
    r.origin.x -= border.x;
    r.extent.x += 2*border.y;
    InvalidateRect(r.Expand(Point(4,0)));
}

void StaticTextView::BreakWord(int lWidth, int *start, int *line,int *end,int *width,
								LineDesc *maxld)
{
    int w,cw = 0; 
    AutoTextIter ti(text,*start,*end); // avoid nesting of nextc
    int l = 0;
    LineDesc ld;

    maxld->Reset();
    while (ti(&w,&ld) != cEOT) 
	if (cw + w > lWidth) {
	    cw = w;
	    *end = ti.GetPos() - 1;
	    MarkLine(*line,*start,*end,maxld); 
	    *maxld = ld;
	    (*line)++;
	    *start = *end;
	} 
	else {      
	   maxld->Max(ld);
	   cw += w;
	}
    *width = cw;
    *end = ti.GetPos();
} 

void StaticTextView::Dump()
{
    drawViewBorder = TRUE;    
    cerr << "Lines: " << nLines NL;
    cerr << "Contents:\n"<< *text NL;
    for (int i = 0; i < nLines; i++)
	cerr << i << "." << (*lines)[i]->ClassName(), (*lines)[i]->DisplayOn(cerr), cerr NL;;
}

char *StaticTextView::AsString()
{
    return text->AsString();
}

ostream& StaticTextView::PrintOn (ostream&s)
{
    View::PrintOn(s);
    return s << text SP << wrap SP << spacing SP << just SP << 
		vertExtend SP << horExtend SP << border SP << contentRect.extent;
}

istream& StaticTextView::ReadFrom(istream &s)
{
    Text *txt;
    int wrap, vExtend, hExtend;
    Point extent,itsBorder;
    int isp, ijust;
    eSpacing sp;
    eTextJust just;
    Rectangle r;

    SafeDelete(text);

    View::ReadFrom(s);
    s >> txt >> wrap >> isp >> ijust >> vExtend >> hExtend >> itsBorder >> extent;
    sp = isp;
    just = ijust;
    r= Rectangle(extent);
    if (vExtend)
	r.extent.y = cFit;
    if (hExtend)
	r.extent.x = cFit;

    Init(r, txt, just, sp, bool(wrap), 0, itsBorder);
    return s;
}

void StaticTextView::AddMark(Mark *m)
{
    text->AddMark(m);
}

Mark *StaticTextView::RemoveMark(Mark *m)
{
    return text->RemoveMark(m);
}

Iterator *StaticTextView::GetMarkIter()
{
    return text->GetMarkIter();
}

MarkList *StaticTextView::GetMarkList()
{
    return text->GetMarkList();
}

int StaticTextView::OutOfRangeLineMetric(bool lineHeight)
{
    if (lineHeight)
	return TextViewLineHeight(text->GetFont(),spacing);  
    else
	return text->GetFont()->Ascender();      
}

void StaticTextView::InspectorId(char *buf, int sz)
{
    if (text)
	text->InspectorId(buf, sz);
    else
	View::InspectorId(buf, sz);
}
	
