#include <pixrect/pixrect_hs.h>
#include <sunwindow/window_hs.h>
#include <math.h>
#include <fcntl.h>
#include <errno.h>
#include <sunwindow/win_ioctl.h>
#include <sys/ioctl.h>
#include <suntool/fullscreen.h>

#include "sungr.h"

#define PATTERNS "patterns.font"
#define CURSORS "cursors.font"

#define MAXPENS     30
#define MAXSLOTS    32
#define MAXPATTERNS 25

#define PI180 0.017453293

#define TAN(x) (sin((float)(x))/cos((float)(x)))
#define COT(x) (cos((float)(x))/sin((float)(x)))
#define ATAN2(x, y) (atan2((float)(x), (float)(y)))

#define PROX (port->fsorigin.p_x)
#define PROY (port->fsorigin.p_y)

#define lock(port, r)  if (! (port)->inbatch) pw_lock((port)->pw, (r))
#define unlock(port)   if (! (port)->inbatch) pw_unlock((port)->pw)

#define AlignX(x) (((x)-port->origin.p_x) & 0xf)
#define AlignY(y) (((y)-port->origin.p_y) & 0xf)

#define pat2ix(p) ((p) >= 3 ? 2 : (p)-1)

#define setmode(pat, mode) Rop= Tab[mode][pat2ix(pat)].rop

/*
 * always keep in sync with Port, WindowPort & SunWindowPort !!!
 * never touch the underscored fields
 */
 
struct SunWindowPort {    
/* Root: */
    int *_vtbl;
    
/* Port: */
    byte _type;
    
    /* graphic state */
    struct point origin;
    struct rect cliprect;
    BOOL _inclip;
    BOOL _pendingclip;
    short _mode;
    short _patid;
    
    /* pen state */
    struct point _penpos;
    short _pensize;
    short _penmode;
    short _penpatid;
    short _pencap;
    
    /* text state */
    short _textmode;
    struct point _textpos;
    short _textpatid;
    FontPtr _textfd;
    
/* WindowPort: */
    Token _pushbacktoken;
    BOOL _havepushbacktoken;
    BOOL _done;
    BOOL _block;
    BOOL overlay;
    GrCursor cursor;
    void *_privdata;
    void *_privdata2;
    int (*_ihf)();
    struct rect _windowRect;
    int _state;
    struct rect *_invalRects;
    struct rect _invalBounds;
    int _inval;
    
/* SunWindowPort: */
    void *next;
    void *handler;
    int fd;
    struct point lastpos, inpOffset, fsorigin;
    struct pixwin *pw, *opw, *rpw;
    struct pixrect *pr, *opr, *rpr;
    struct SunWindowPort *fatherport;
    Pw_pixel_cache *savepr;
    BOOL inbatch;
    struct rect myrect;
};

struct tab {
    char mode11, modeNN;
    int rop;
};

struct SunWindowPort *overlayport;
static int rootFd= -1;
static char parentName[WIN_NAMESIZE];
static int Rop;
static struct rect batchrect;
static bool fdgrab[FD_SETSIZE];
static struct SunWindowPort *allwindows;

/* free on exit */
static struct patterns {
    struct pixrect *pat, *pr;
} Patterns[MAXPATTERNS];

static struct pixfont *cursors, *patterns;
static struct fullscreen *FS;
static struct pixrect *batchpr;
static struct pixrect *pens[MAXPENS];
static struct pixrect *tmppr;

static struct cache {
    short type, k1, k2, k3;
    struct pixrect *pr;
} Cache[MAXSLOTS];


void SunWindowCleanup()
{
    register int i;
    
    for (i= 0; i < MAXSLOTS; i++)
	if (Cache[i].pr)
	    pr_destroy(Cache[i].pr);
    for (i= 0; i < MAXPENS; i++)
	if (pens[i])
	    pr_destroy(pens[i]);
    for (i= 0; i < MAXPATTERNS; i++)
	if (Patterns[i].pr)
	    pr_destroy(Patterns[i].pr);
    if (cursors)
	pf_close(cursors);
    if (patterns)
	pf_close(patterns);
    if (batchpr)
	pr_destroy(batchpr);
    if (tmppr)
	pr_destroy(tmppr);
    if (FS)
	fullscreen_destroy(FS);
}

BOOL sunwindowinit(lib)
char *lib;
{
    static char pathname[100];
    int i;
    
    if (we_getparentwindow(parentName) != 0)
	return TRUE;
	    
    rootFd= open(parentName, O_RDONLY);
    if (rootFd < 0) {
	Error("sunwindowinit", "can't open rootwindow\n");
	return TRUE;
    }
    
    sprintf(pathname, "./%s", PATTERNS);
    if ((patterns= pf_open(pathname)) == NULL) {
	sprintf(pathname, "%s/fonts/%s", lib, PATTERNS);
	if ((patterns= pf_open(pathname)) == NULL) {
	    Error("sunwindowinit", "can't open patternfont %s", pathname);
	    return TRUE;
	}
    }
    for (i= 0; i < MAXPATTERNS; i++)
	Patterns[i].pat= patterns->pf_char[i].pc_pr;
    
    sprintf(pathname, "./%s", CURSORS);
    if ((cursors= pf_open(pathname)) == NULL) {
	sprintf(pathname, "%s/fonts/%s", lib, CURSORS);
	if ((cursors= pf_open(pathname)) == NULL) {
	    Error("sunwindowinit", "can't open cursorfont %s", pathname);
	    return TRUE;
	}
    }
    return FALSE;
}

static double Phi (dx, dy)
int dx, dy;
{
    int r;
    
    r= (int) (ATAN2(dx, dy) * 180.0 / M_PI);
    r= (450-(r % 360)) % 360;
    return r * PI180;
}

static int NewPixrect(prp, w, h, exact, clear)
struct pixrect **prp;
int w, h, exact, clear;
{
    struct pixrect *pr= *prp;
    
    if (pr == 0 || pr->pr_size.x != w || pr->pr_size.y != h) {
	if (pr == 0 || pr->pr_size.x < w || pr->pr_size.y < h || exact) {
	    if (pr)
		pr_destroy(pr);
	    if ((*prp= mem_create(w, h, 1)) == 0) {
		Error("NewPixrect", "can't allocate pixrect");
		return 0;
	    }
	} else if (clear)
	    pr_rop(pr, 0, 0, w, h, CLR|NCLIP, 0, 0, 0);
	return 1;
    }
    return 0;
}

/*---- FillOval ---------------------------------------------------------------*/

#define setbit(a,i) ((a)[(i) >> 5])|= 1 << (31-((i) & 31))

static int gapybits[50];
static int shallowsteep[100];
static int steepshallow[100];

static struct pr_chain left1= { 0, { 0, 0 }, steepshallow },
		       leftgapy= { &left1, { 0, 0 }, gapybits },
		       left0= { &leftgapy, { 0, 0 }, shallowsteep },
		       right1= { 0, { 0, 0 }, steepshallow },
		       rightgapy= { &right1, { 0, 0 }, gapybits },
		       right0= { &rightgapy, { 0, 0 }, shallowsteep };
		
static struct pr_fall left_oct = {
    { 0, 0 },
    &left0
};
static struct pr_fall right_oct = {
    { 0, 0 },
    &right0
};
	       
static struct pr_trap oval = {
    &left_oct,
    &right_oct,
    0,
    0
};

/*---- Clipping ---------------------------------------------------------------*/

static struct tab Tab[3][3]= {
/*             white        black       other */
/* Copy  */    1,5,  CLR,   0,5,SET,    3,4,COPY,
/* Xor   */    -1,-1,NOP,   2,5,INV,    3,4,XOR,
};

static struct pixrect *GetPattern(pid, w, h)
register int w, h;
{
    register struct patterns *p;
    
    if (pid < 3)
	return 0;
    pid--;
    
    w= ((w >> 4)+2) << 4;
    h= ((h >> 4)+2) << 4;
    
    p= &Patterns[pid];
    if (p->pat == 0) {
	Error("GetPattern", "patid out of range (%d)", pid);
	return 0;
    }
    
    if (NewPixrect(&p->pr, w, h, FALSE, TRUE))
	pr_replrop(p->pr, 0, 0, w, h, OR|NCLIP, p->pat, 0, 0);
    return p->pr;
}

static void BitBlt(port, p, r, x, y, w, h)
register struct SunWindowPort *port;
register int x, y, w, h, p, r;
{
    if (port->pr) {
	if (p > 2)
	    pr_replrop(port->pr, x+PROX, y+PROY, w, h, r, Patterns[p-1].pat,
					x-port->origin.p_x, y-port->origin.p_y);
	else
	    pr_rop(port->pr, x+PROX, y+PROY, w, h, r, 0, 0, 0);
    } else {
	if (p > 2)
	    pw_replrop(port->pw, x, y, w, h, r, Patterns[p-1].pat, 
					x-port->origin.p_x, y-port->origin.p_y);
	else
	    pw_rop(port->pw, x, y, w, h, r, 0, 0, 0);
    }
}

static void Vector(port, r, x1, y1, x2, y2)
register struct SunWindowPort *port;
int r;
register int x1, y1, x2, y2;
{
    if (port->pr)
	pr_vector(port->pr, x1+PROX, y1+PROY, x2+PROX, y2+PROY, r, 1);
    else
	pw_vector(port->pw, x1, y1, x2, y2, r, 1);
}

static void Traprop(port, pat, rop, x, y, w, h, radx, oval)
register struct SunWindowPort *port;
int pat, rop, x, y, w, h, radx;
struct pr_trap oval;
{
    if (port->pr) {
	if (pat > 2)
	    pr_traprop(port->pr, x+radx+PROX, y+PROY, oval, rop,
			    GetPattern(pat,w,h), radx+AlignX(x), AlignY(y));
	else
	    pr_traprop(port->pr, x+radx+PROX, y+PROY, oval, rop, 0, 0, 0);
    } else {
	if (pat > 2)
	    pw_traprop(port->pw, x+radx, y, oval, rop, GetPattern(pat,w,h),
						radx+AlignX(x), AlignY(y));
	else
	    pw_traprop(port->pw, x+radx, y, oval, rop, 0, 0, 0);
    }
}

static void Polygon(port, pat, rop, n, pts, rp, pr)
register struct SunWindowPort *port;
int pat, rop, n;
register struct pr_pos *pts;
struct rect *rp;
struct pixrect *pr;
{
    register int i, l, t;
    struct pixrect *patpr;
    
    l= rp->r_left;
    t= rp->r_top;
	
    if (pr) {
	for (i= 0; i < n-1; i++)
	    pr_vector(pr, pts[i].x+l, pts[i].y+t,  pts[i+1].x+l, pts[i+1].y+t, NCLIP|SET, 1);
	pr_vector(pr, pts[i].x+l, pts[i].y+t,  pts[0].x+l, pts[0].y+t, NCLIP|SET, 1);
	pr_polygon_2(pr, l, t, 1, &n, pts, NCLIP|SET, 0, 0, 0);
	return;
    }
    
    if (pat > 2) {
	patpr= GetPattern(pat, rp->r_width, rp->r_height);
    } else {
	patpr= 0;
	/* urgh!! polygons are half open !! (wrong for patterns) */
	for (i= 0; i < n-1; i++)
	    Vector(port, rop, pts[i].x+l, pts[i].y+t, pts[i+1].x+l, pts[i+1].y+t);
	Vector(port, rop, pts[i].x+l, pts[i].y+t, pts[0].x+l, pts[0].y+t);
    }
    if (port->pr)
	pr_polygon_2(port->pr, l+PROX, t+PROY, 1, &n, pts, rop, patpr,
							    AlignX(l), AlignY(t));
    else
	pw_polygon_2(port->pw, l, t, 1, &n, pts, rop, patpr,
							    AlignX(l), AlignY(t));
}

static struct tab PolyTab[3][3]= {
/*             white        black       other */
/* Copy  */    1,3,CLR,     1,3,SET,    2,3,COPY,
/* Xor   */    0,0,NOP,     1,3,INV,    2,3,XOR,
};

static struct pixrect *MakePen(psz)
int psz;
{
    struct pixrect *pr= mem_create(psz, psz, 1);
    int gap, rad;
    
    rad= psz/2;
    gap= psz - 2*rad;
    MakeQuadrant(rad, rad, gap, gap, psz);
    pr_traprop(pr, rad, 0, oval, SET|NCLIP, 0, 0, 0);
    return pr;
}

static void PolyDot(port, pr, dx, dy, pts, n, pattern, mode, psz)
register struct SunWindowPort *port;
register struct pixrect *pr;
register int n; 
int dx, dy, pattern, mode, psz;
register struct pr_pos *pts;
{
    register int i, x, y;
    register struct pr_pos *p;
    register struct pixrect *patpr, *penpr= 0;
    int oox, ooy, rop, Mode;
    
    rop= PolyTab[mode][pat2ix(pattern)].rop;
    if (psz <= 1) 
	Mode= PolyTab[mode][pat2ix(pattern)].mode11;
    else {
	if (psz > 2 && psz < MAXPENS) {
	    if (pens[psz] == 0)
		pens[psz]= MakePen(psz);
	    penpr= pens[psz];
	}
	Mode= PolyTab[mode][pat2ix(pattern)].modeNN;
    }
    patpr= GetPattern(pattern);
    
    if (pr) {
	if (penpr)
	    rop= OR|NCLIP;
	else
	    rop= SET|NCLIP;
	patpr= 0;
	switch (Mode) {
	case 0:
	    break;
	case 1:
	    pr_polypoint(pr, dx, dy, n, pts, rop);
	    break;
	case 2:
	    for (p= pts, i= 0; i<n; i++, p++) {
		x= p->x+dx;
		y= p->y+dy;
		pr_vector(pr, x, y, x, y, rop,
				  pr_get(patpr, AlignX(x), AlignY(y)) ? 1 : 0 );
	    }
	    break;
	case 3:
	    for (p= pts, i= 0; i<n; i++, p++)
		pr_rop(pr, p->x+dx, p->y+dy, psz, psz, rop, penpr, 0, 0);
	    break;
	default:
	    Error("PolyDot", "wrong Mode(1)");
	    break;
	}
    } else if (port->pr) {
	switch (Mode) {
	case 0:
	    break;
	case 1:
	    pr_polypoint(port->pr, dx+PROX, dy+PROY, n, pts, rop);
	    break;
	case 2:
	    oox= PROX;
	    ooy= PROY;
	    for (p= pts, i= 0; i<n; i++, p++) {
		x= p->x + dx;
		y= p->y + dy;
		pr_vector(port->pr, x+oox, y+ooy, x+oox, y+ooy, rop,
			      pr_get(patpr, AlignX(x), AlignY(y)) ? 1 : 0 );
	    }
	    break;
	case 3:
	    oox= PROX+dx;
	    ooy= PROY+dy;
	    for (p= pts, i= 0; i<n; i++, p++)
		pr_rop(port->pr, p->x+oox, p->y+ooy, psz, psz, rop,
								    penpr, 0, 0);
	    break;
	default:
	    Error("PolyDot", "wrong Mode(2)");
	    break;
	}
    } else {
	switch (Mode) {
	case 0:
	    break;
	case 1:
	    pw_polypoint(port->pw, dx, dy, n, pts, rop);
	    break;
	case 2:
	    for (p= pts, i= 0; i<n; i++, p++) {
		x= p->x + dx;
		y= p->y + dy;
		pw_vector(port->pw, x, y, x, y, rop,
			      pr_get(patpr, AlignX(x), AlignY(y)) ? 1 : 0 );
	    }
	    break;
	case 3:
	    for (p= pts, i= 0; i<n; i++, p++)
		pw_rop(port->pw, p->x+dx, p->y+dy, psz, psz, rop, penpr, 0, 0);
	    break;
	default:
	    Error("PolyDot", "wrong Mode(3)");
	    break;
	}
    }
}

/*---- pixrect cache ----------------------------------------------------------*/

static int findpr(type, pr, wd, ht, k1, k2, k3)
char type;
register struct pixrect **pr;
register int wd, ht, k1, k2, k3;
{
    static int last= 0;
    int i, found= FALSE, emptyslot= -1;
    int stop;
    
    if (last)
	stop= last-1;
    else
	stop= MAXSLOTS-1;
    for (i= last; i != stop; i= (i+1) % MAXSLOTS) {
	if (Cache[i].pr == 0) {
	    emptyslot= i;
	    continue;
	}
	if (Cache[i].type == type
		&& Cache[i].pr->pr_size.x == wd && Cache[i].pr->pr_size.y == ht
		&& Cache[i].k1 == k1 && Cache[i].k2 == k2 && Cache[i].k3 == k3) {
	    found= TRUE;
	    break;
	}
    }
    if (! found) {
	if (emptyslot >= 0)
	    i= emptyslot;
	else
	    pr_destroy(Cache[i].pr);
	Cache[i].pr= mem_create(wd, ht, 1);
	Cache[i].type= type;
	Cache[i].k1= k1;
	Cache[i].k2= k2;
	Cache[i].k3= k3;
    }
    *pr= Cache[last= i].pr;
    return found;
}

struct ttab {
    int mode;
    int rop;
};

static struct ttab TTab[3][3]= {
/*              white      black       other */
/* Copy  */     0,ERASE,   0,OR,       1,COPY,
/* Xor   */     0,XOR,     0,XOR,      0,XOR,
};

static void Stencil(port, r, pr, pat, mode)
register struct SunWindowPort *port;
register struct rect *r;
int mode, pat;
struct pixrect *pr;
{
    int rop= TTab[mode][pat2ix(pat)].rop;
    
    lock(port, r);
    if (TTab[mode][pat2ix(pat)].mode) {
	if (port->pr)
	    pr_stencil(port->pr, r->r_left+PROX, r->r_top+PROY,
				r->r_width, r->r_height, rop, pr, 0, 0,
			      GetPattern(pat, r->r_width, r->r_height),
					AlignX(r->r_left), AlignY(r->r_top));
	else
	    pw_stencil(port->pw, r->r_left, r->r_top, r->r_width,
					    r->r_height, rop, pr, 0, 0,
			      GetPattern(pat, r->r_width, r->r_height),
					AlignX(r->r_left), AlignY(r->r_top));
    } else {
	if (port->pr)
	    pr_rop(port->pr, r->r_left+PROX, r->r_top+PROY, r->r_width,
						   r->r_height, rop, pr, 0, 0);
	else
	    pw_rop(port->pw, r->r_left, r->r_top, r->r_width, r->r_height,
								rop, pr, 0, 0);
    }
    unlock(port);
}

/*---- line -------------------------------------------------------------*/

static void Line(port, pr, fromX, fromY, X2, Y2, pat, mode, psz)     /* bresenham */
register struct SunWindowPort *port;
register struct pixrect *pr;
int fromX, fromY, X2, Y2, pat, mode, psz;
{
    register int i, j, k, p, q, d, dx, dy;
    struct pr_pos pts[2000], *pp= pts;

    i= (fromX < X2) ? 1 : -1;
    j= (fromY < Y2) ? 1 : -1;
    dx= abs(fromX - X2);
    dy= abs(fromY - Y2);
    if (dx >= dy) { /* flat line */
	d= p= dy+dy;
	q= 2*(dy-dx);
	for(k= 0; k <= dx; fromX+= i, k++) {    
	    pp->x= fromX;
	    pp++->y= fromY;
	    if(d <= dx)
		d+= p;
	    else {
		d+= q;
		fromY+= j;
	    }
	}
    } else {          /* steep line */
	d= p= dx+dx;
	q= 2*(dx-dy);
	for(k= 0; k <= dy; fromY+= j, k++) {
	    pp->x= fromX;
	    pp++->y= fromY;
	    if (d <= dy)
		d+= p;
	    else {
		d+= q;
		fromX+= i;
	    }
	}
    }
    PolyDot(port, pr, 0, 0, pts, k, pat, mode, psz);
}

#define prline(port, pr, psz, x0, y0, x1, y1)           \
if (psz <= 1)                                           \
    pr_vector(pr, x0, y0, x1, y1, SET|NCLIP, 1);        \
else                                                    \
    Line(port, pr, x0, y0, x1, y1, Black, Copy, psz)

#define line(port, pat, mode, psz, x0, y0, x1, y1, r)   \
if (psz <= 1 && pat <= 2)                               \
    Vector(port, Rop, x0, y0, x1, y1);                  \
else                                                    \
    _line(port, pat, mode, psz, x0, y0, x1, y1, r)

static void _line(port, pat, mode, psz, x0, y0, x1, y1, r)
register struct SunWindowPort *port;
register int pat, mode, psz, x0, y0, x1, y1;
struct rect *r;
{
    int w= psz/2;
    
    if (x0 == x1) {
	if (y0 <= y1)
	    BitBlt(port, pat, Rop, x0-w, y0, psz, y1-y0);
	else
	    BitBlt(port, pat, Rop, x1-w, y1, psz, y0-y1);
    } else if (y0 == y1) {
	if (x0 <= x1)
	    BitBlt(port, pat, Rop, x0, y0-w, x1-x0, psz);
	else
	    BitBlt(port, pat, Rop, x1, y1-w, x0-x1, psz);
    } else if (psz <= 1)
	Line(port, 0, x0, y0, x1, y1, pat, mode, psz);
    else {
	int dx, dy;
	float a;
	struct pr_pos pts[4];
	a= Phi(x1-x0, y1-y0);
	dx= (int)((float)psz/2*sin(a)+0.5);
	dy= (int)((float)psz/2*cos(a)+0.5);
	
	pts[0].x= x0 - dx;
	pts[0].y= y0 + dy;
	pts[1].x= x0 + dx;
	pts[1].y= y0 - dy;
	pts[2].x= x1 + dx;
	pts[2].y= y1 - dy;
	pts[3].x= x1 - dx;
	pts[3].y= y1 + dy;
	r->r_left= r->r_top= 0;
	Polygon(port, pat, Rop, 4, pts, r, 0);
    }
}

void _SunWindowPort_DevStrokeLine2(port, pat, mode, psz, r, cap, p1, p2)
register struct SunWindowPort *port;
struct rect *r;
struct point p1, p2;
int pat, mode, cap, psz;
{
    setmode(pat, mode);
    if (psz <= 0)
	psz= 1;
    lock(port, r);
    line(port, pat, mode, psz, p1.p_x, p1.p_y, p2.p_x, p2.p_y, r);
    unlock(port);
}

/*---- FillRect ---------------------------------------------------------------*/

void _SunWindowPort_DevFillRect(port, pat, mode, r)
register struct SunWindowPort *port;
register struct rect *r;
int pat, mode;
{
    setmode(pat, mode);
    lock(port, r);
    BitBlt(port, pat, Rop, r->r_left, r->r_top, r->r_width, r->r_height);
    unlock(port);
}

/*---- Show and Scale Bitmap --------------------------------------------------*/

void _SunWindowPort_DevShowBitmap(port, pat, mode, r, bmp)
register struct SunWindowPort *port;
struct rect *r;
GrBitMap *bmp;
int pat, mode;
{
    register int src, dst, x, y, sx, sy, d;
    register float scale;
    struct pixrect *pr, *pr1;

    x= bmp->size.p_x;
    y= bmp->size.p_y;
    sx= r->r_width;
    sy= r->r_height;
    
    if (sx <= 0 || sy <= 0) {
	Error("SunWindowPort::DevShowBitmap", "sx <= 0 || sy <= 0");
	return;
    }
    if (x <= 0 || y <= 0) {
	Error("SunWindowPort::DevShowBitmap", "x <= 0 || y <= 0");
	return;
    }
    
    if (sx == x && sy == y)
	pr= bmp->pr;        /* use unscaled bitmap */
    else  if (bmp->spr && bmp->spr->pr_size.x == sx && bmp->spr->pr_size.y == sy)
	pr= bmp->spr;       /* use last scaled bitmap */
    else {
	if (bmp->spr)
	    pr_destroy(bmp->spr);
	
	if (sx == x)        /* same width */
	    pr= bmp->pr;
	else if (sx < x) {  /* x-shrink */
	    pr= mem_create(sx, y, 1);
	    scale= (float)x/(float)sx;
	    for (dst= 0; dst < sx; dst++)
		pr_rop(pr, dst, 0, 1, y, COPY|NCLIP, bmp->pr,
						    (int)(dst*scale+0.5), 0);
	} else {            /* x-magnify */
	    pr= mem_create(sx, y, 1);
	    d= sx/x;
	    scale= (float)sx/(float)x;
	    for (src= 0; src < x; src++)    /* slice */
		pr_rop(pr, (int)(src*scale+0.5), 0, 1, y, COPY|NCLIP, bmp->pr, 
									src, 0);
	    for (src= 0; src < d; src++)  /* smear */
		pr_rop(pr, 1, 0, sx-1, y, OR|NCLIP, pr, 0, 0);
	}
	
	if (sy < y) {           /* y-shrink */
	    pr1= mem_create(sx, sy, 1);
	    scale= (float)y/(float)sy;
	    for (dst= 0; dst < sy; dst++)
		pr_rop(pr1, 0, dst, sx, 1, COPY|NCLIP, pr, 0,
							(int)(dst*scale+0.5));
	    if (sx != x)
		pr_destroy(pr);
	    pr= pr1;
	} else if (sy > y) {    /* y-magnify */
	    pr1= mem_create(sx, sy, 1);
	    d= sy/y;
	    scale= (float)sy/(float)y;
	    for (src= 0; src < y; src++)    /* slice */
		pr_rop(pr1, 0, (int)(src*scale+0.5), sx, 1, COPY|NCLIP, pr, 0,
									    src);
	    for (src= 0; src < d; src++)  /* smear */
		pr_rop(pr1, 0, 1, sx, sy-1, OR|NCLIP, pr1, 0, 0);
	    if (sx != x)
		pr_destroy(pr);
	    pr= pr1;
	}
	bmp->spr= pr;   
    }
    Stencil(port, r, pr, pat, mode);
}

/*---- StrokeRect -------------------------------------------------------------*/

void _SunWindowPort_DevStrokeRect2(port, pat, mode, psz, r)
register struct SunWindowPort *port;
register struct rect *r;
int pat, mode, psz;
{
    register int l= r->r_left, t= r->r_top, wd= r->r_width, ht= r->r_height;
    
    setmode(pat, mode);
    if (psz <= 0)
	psz= 1;
    lock(port, r); 
    BitBlt(port, pat, Rop, l+psz, t, wd-psz, psz);
    BitBlt(port, pat, Rop, l+wd-psz, t+psz, psz, ht-psz);
    BitBlt(port, pat, Rop, l, t+ht-psz, wd-psz, psz);
    BitBlt(port, pat, Rop, l, t, psz, ht-psz);
    unlock(port); 
}

/*---- Polygon ----------------------------------------------------------------*/

void _SunWindowPort_DevFillPolygon2(port, r, pat, mode, pts, npts, type)
register struct SunWindowPort *port;
int pat, mode;
struct point *pts;
struct rect *r;
int npts;
GrPolyType type;
{
    struct pr_pos p[1000];
    register int i;
    
    for (i= 0; i < npts; i++) {
	p[i].x= pts[i].p_x;
	p[i].y= pts[i].p_y;
    }
    
    setmode(pat, mode);
    lock(port, r);
    Polygon(port, pat, Rop, npts, p, r, 0);
    unlock(port);
}

/*---- StrokePolygon ----------------------------------------------------------*/    

void _SunWindowPort_DevStrokePolygon2(port, r, pat, mode, p, n, type, psz, cap)
register struct SunWindowPort *port;
struct rect *r;
struct point *p;
int pat, mode, n, cap, psz;
GrPolyType type;
{
    register int i, dx, dy, x, y, x0, y0, x1, y1;
    bool xormode= (mode == (int)Xor);
    
    setmode(pat, mode);
	
    if (psz <= 0)
	psz= 1;
	
    xormode= 1;
    psz= 1;
    pat= Black;

    if (psz <= 1 && xormode && pat == (int) Black) {
	x0= p[0].p_x + r->r_left;
	y0= p[0].p_y + r->r_top;
	lock(port, r);
	for (i= 1; i<n; i++) {
	    x= x1= p[i].p_x + r->r_left;
	    y= y1= p[i].p_y + r->r_top;
	    
	    if (xormode && i < n-1) {
		dx= x1-x0;
		dy= y1-y0;
		if (dx >= abs(dy))
		    x--;
		else if (-dx >= abs(dy))
		    x++;
		if (dy >= abs(dx))
		    y--;
		else if (-dy >= abs(dx))
		    y++;
	    }
	    line(port, pat, mode, psz, x0, y0, x, y, r);
	    x0= x1;
	    y0= y1;
	}
	unlock(port);
    } else {
	/*
	struct rect rr;
	if (NewPixrect(&poly->strokepr, r->r_width+psz, r->r_height+psz, TRUE, TRUE)) {
	    for (i= 0; i<n-1; i++)
		prline(port, poly->strokepr, psz, list[i].x, list[i].y, list[i+1].x, list[i+1].y);
	    prline(port, poly->strokepr, psz, list[i].x, list[i].y, list[0].x, list[0].y);
	}
	rr= *r;
	rr.r_left-= psz/2;
	rr.r_top-= psz/2;
	rr.r_width+= psz;
	rr.r_height+= psz;
	Stencil(port, &rr, poly->strokepr, pat, mode);
	*/
    }
}

/*---- FillOval ---------------------------------------------------------------*/

static int MakeQuadrant(radx, rady, gapx, gapy, h)
int radx, rady, gapx, gapy, h;
{
    static int lastradx, lastrady, lastgapx, lastgapy;
    register int d, m, n, t6, t7;
    int i, t1, t2, t3, t4, t5, t8;

    if (lastradx==radx && lastrady==rady && lastgapx==gapx && lastgapy==gapy)
	return FALSE;
    lastradx= radx; lastrady= rady;
    lastgapx= gapx; lastgapy= gapy;

    left0.size.x= right1.size.x= -radx;
    right0.size.x= left1.size.x= radx;
    left0.size.y= right0.size.y= left1.size.y= right1.size.y= rady;
    leftgapy.size.y= rightgapy.size.y= gapy;
    right_oct.pos.x= gapx;
    oval.y0= 0;
    oval.y1= h;
	
    radx--;
    rady--;
    
    t1= radx * radx; t2= t1 + t1;
    t3= rady * rady; t4= t3 + t3;
    t5= radx * t3; t6= t5 + t5; t7= 0;
    t8= radx + rady;
    d= (t1+t3)/4-t5;
    
    for (i= 0; i < 100; i++)
	shallowsteep[i]= steepshallow[i]= 0;
    setbit(shallowsteep, 0);
    for (m= 1, n= t8; m <= t8; m++, n--) {
	if (d < 0) {
	    t7+= t2;
	    d+= t7;
	} else {
	    setbit(steepshallow, m);
	    setbit(shallowsteep, n);
	    t6-= t4;
	    d-= t6;
	}
    }
    return TRUE;
}

void _SunWindowPort_DevFillRRect2(port, pat, mode, r, dia)
register struct SunWindowPort *port;
struct rect *r;
struct point dia;
int pat, mode;
{
    int gapx, gapy, radx= dia.p_x/2, rady= dia.p_y/2;
    
    setmode(pat, mode);
	
    gapx= r->r_width - 2*radx;
    gapy= r->r_height - 2*rady;
    MakeQuadrant(radx, rady, gapx, gapy, r->r_height);
    lock(port, r);
    Traprop(port, pat, Rop, r->r_left, r->r_top, r->r_width, r->r_height, radx, oval);
    unlock(port);
}

/*---- StrokeOval -------------------------------------------------------------*/
       
static void PolyEllipse(port, pr, x0, y0, x1, x2, y1, y2, a, b, pat, mode, psz)
register struct SunWindowPort *port;
register struct pixrect *pr;
int x0, y0, x1, x2, y1, y2, a, b, pat, mode, psz;
{
    static struct pr_pos q1[1000], q2[1000], q3[1000], q4[1000];
    static int lasta= -1, lastb= -1, lastn= -1;
    
    register struct pr_pos *q1p= q1, *q2p= q2, *q3p= q3, *q4p= q4;
    register int x, y;
    int t1, t2, t3, t4, t5, t6, t7, t8, i, d;
    
    if (lasta != a || lastb != b) {
	lasta= x= a;
	lastb= b;
	y= 0;
	t1= a*a; t2= t1 + t1;
	t3= b*b; t4= t3 + t3;
	t5= a * t3; t6= t5 + t5; t7= 0;
	t8= a + b;
	d= (t1+t3)/4-t5;
	
	for (i= 0; i < t8; i++) {
	    q1p->x= x;  q1p++->y= -y;
	    q2p->x= x;  q2p++->y= y;
	    q3p->x= -x; q3p++->y= y;
	    q4p->x= -x; q4p++->y= -y;
	    if (d < 0) {
		y++;
		t7+= t2;
		d+= t7;
	    } else {
		x--;
		t6-= t4;
		d-= t6;
	    }
	}
	lastn= i;
    }
    
    x1+= x0; x2+= x0;
    y1+= y0; y2+= y0;
    
    PolyDot(port, pr, x1, y1, q1, lastn, pat, mode, psz);
    PolyDot(port, pr, x1, y2, q2, lastn, pat, mode, psz);
    PolyDot(port, pr, x2, y2, q3, lastn, pat, mode, psz);
    PolyDot(port, pr, x2, y1, q4, lastn, pat, mode, psz);
}

void _SunWindowPort_DevStrokeRRect2(port, pat, mode, psz, r, dia)
register struct SunWindowPort *port;
struct rect *r;
struct point dia;
int pat, mode, psz;
{
    int ww, hh, cx, cy, w, h, rad1, rad2, l, t, r1, r2;
    struct pixrect *cpr= 0;
	
    setmode(pat, mode);    
    
    if (psz <= 0)
	psz= 1;
	
    ww= r->r_width;
    hh= r->r_height;
    
    w= ww - psz;
    h= hh - psz;
    
    rad1= (dia.p_x - psz)/2;
    rad2= (dia.p_y - psz)/2;
    rad1= max(1, rad1);
    rad2= max(1, rad2);
    
    r1= dia.p_x/2 -1;
    r2= dia.p_y/2;
    
    cx= ww-r1-r1;
    cy= hh-r2-r2;
    
    if (psz > 1 && pat > 0) {
	if (! findpr('a', &cpr, ww, hh, rad1, rad2, psz)) {
	    PolyEllipse(port, cpr, 0, 0, w-rad1, rad1, rad2, h-rad2, rad1, rad2, pat, mode, psz);
	    if (cx) {
		pr_rop(cpr, r1, 0, cx, psz, SET|NCLIP, 0, 0, 0);
		pr_rop(cpr, r1, h, cx, psz, SET|NCLIP, 0, 0, 0);
	    }
	    if (cy) {
		pr_rop(cpr, 0, r2, psz, cy, SET|NCLIP, 0, 0, 0);
		pr_rop(cpr, w, r2, psz, cy, SET|NCLIP, 0, 0, 0);
	    }
	}
	Stencil(port, r, cpr, pat, mode);
    } else {
	lock(port, r);
	l= r->r_left;
	t= r->r_top;
	PolyEllipse(port, 0, l, t, w-rad1, rad1, rad2, h-rad2, rad1, rad2, pat, mode, psz);
	if (cx) {
	    BitBlt(port, pat, Rop, l+r1, t,   cx, psz); 
	    BitBlt(port, pat, Rop, l+r1, t+h, cx, psz); 
	}
	if (cy) {
	    BitBlt(port, pat, Rop, l,   t+r2, psz, cy);
	    BitBlt(port, pat, Rop, l+w, t+r2, psz, cy);
	}
	unlock(port);
    }
}

/*---- wedges -----------------------------------------------------------------*/

static void ang2pt(a, w, h, xp, yp)
int a, w, h, *xp, *yp;
{
    int rad1= w/2, rad2= h/2, b;
    int x, y;
    float d;
    
    if (a < 0)
	b= (abs(a % 360) + 90) % 360;
    else
	b= (450-(a % 360)) % 360;
    d= PI180 * (float)b;
    
    if (a > 45 && a <= 135) {
	x= rad1;
	y= (int)(TAN(d)*rad2);
    } else if (a > 135 && a <= 225) {
	x= -(int)(COT(d)*rad1);
	y= -rad2;
    } else if (a > 225 && a <= 315) {
	x= -rad1;
	y= -(int)(TAN(d)*rad2);
    } else {
	x= (int)(COT(d)*rad1);
	y= rad2;
    }
    *xp= rad1 + x;
    *yp= rad2 - y;
}

static struct pixrect *makewedgepoly(w, h, start, end)
int w, h, start, end;
{
    struct pixrect *wpr;
    
    w++;
    h++;
    if (! findpr('w', &wpr, w, h, start, end, 0)) {
	struct pr_pos list[10];
	int a, k;
	
	list[0].x= w/2; list[0].y= h/2; 
	ang2pt(end, w, h, &list[1].x, &list[1].y);
	
	for (a= end, k= 2; a != start; a=(a+1)%360) {
	    switch(a) {
	    case 45:
		list[k].x= w+1;
		list[k++].y= 0;
		break;
	    case 135:
		list[k].x= w+1;
		list[k++].y= h+1;
		break;
	    case 225:
		list[k].x= 0;
		list[k++].y= h+1;
		break;
	    case 315:
		list[k].x= 0;
		list[k++].y= 0;
		break;
	    default:
		continue;
	    }
	}
	ang2pt(start, w, h, &list[k].x, &list[k].y);
	k++;
	
	pr_polygon_2(wpr, 0, 0, 1, &k, list, SET, 0, 0, 0);
    }
    return wpr;
}

void _SunWindowPort_DevFillWedge2(port, pat, mode, r, start, end)
register struct SunWindowPort *port;
struct rect *r;
int start, end, pat, mode;
{
    struct pixrect *pr;
    register int w= r->r_width, h= r->r_height;
       
    setmode(pat, mode);
    if (! findpr('c', &pr, w, h, start, end, 0)) {
	int radx= w/2, rady= h/2;
	
	MakeQuadrant(radx, rady, w - 2*radx, h - 2*rady, h);
	pr_traprop(pr, radx, 0, oval, ICOPY, makewedgepoly(w,h,start,end), radx, 0);
    }
    Stencil(port, r, pr, pat, mode);
}

void _SunWindowPort_DevStrokeWedge2(port, pat, mode, psz, cap, r, start, end)
register struct SunWindowPort *port;
register struct rect *r;
int start, end, pat, mode, cap, psz;
{
    register int ww= r->r_width, hh= r->r_height;
    struct pixrect *pr, *pr1, *pr2;
	
    setmode(pat, mode);
    
    if (psz <= 0)
	psz= 1;
	
    if (! findpr('b', &pr, ww, hh, start, end, psz)) {
	int rad1, rad2;
    
	rad1= (ww - psz)/2;
	rad2= (hh - psz)/2;
	rad1= max(1, rad1);
	rad2= max(1, rad2);
	
	if (! findpr('a', &pr1, ww, hh, rad1, rad2, psz)) {
	    int w, h, r1, r2, cx, cy;
	    
	    w= ww - psz;
	    h= hh - psz;
	    
	    PolyEllipse(port, pr1, 0, 0, w-rad1, rad1, rad2, h-rad2, rad1, rad2, pat, mode, psz);
    
	    r1= ww/2 -1;
	    r2= hh/2;
	
	    cx= ww-r1-r1;
	    cy= hh-r2-r2;
	    
	    if (cx) {
		pr_rop(pr1, r1, 0, cx, psz, SET|NCLIP, 0, 0, 0);
		pr_rop(pr1, r1, h, cx, psz, SET|NCLIP, 0, 0, 0);
	    }
	    if (cy) {
		pr_rop(pr1, 0, r2, psz, cy, SET|NCLIP, 0, 0, 0);
		pr_rop(pr1, w, r2, psz, cy, SET|NCLIP, 0, 0, 0);
	    }
	}
	pr2= makewedgepoly(ww, hh, start, end);
	pr_rop(pr, 0, 0, ww, hh, COPY|NCLIP, pr1, 0, 0);
	pr_rop(pr, 0, 0, ww, hh, ERASE, pr2, 0, 0);
    }
    Stencil(port, r, pr, pat, mode);            
}

/*---- text -------------------------------------------------------------------*/

static struct pr_prpos bat[MaxTextBatchCnt], *bp;

bool _SunWindowPort_DevShowChar(port, fdp, dd, c, isnew, pos)
struct SunWindowPort *port;
register FontPtr fdp;
struct point dd, pos;
byte c;
bool isnew;
{
    static int lastdx, lastdy;
    register struct pixchar *pcc;
    register int dx, dy;
    
    if (isnew) {
	bp= bat;
	lastdx= lastdy= 0;
    } else if (bp-bat >= MaxTextBatchCnt)
	return TRUE;
	
    pcc= &fdp->fd_pf->pf_char[c];
    if (pcc->pc_pr == 0)
	_Font_CheckChar(fdp, c);
    bp->pr= pcc->pc_pr;
    dx= pos.p_x + pcc->pc_home.x;
    dy= pos.p_y + pcc->pc_home.y;
    bp->pos.x= dx - lastdx;
    bp++->pos.y= dy - lastdy;
    lastdx= dx;
    lastdy= dy;
    
    return FALSE;
}

void _SunWindowPort_DevShowTextBatch(port, pat, mode, r, pos)
struct SunWindowPort *port;
int pat, mode;
struct rect *r;
struct point pos;
{
    int n= bp-bat;

    if (n <= 0)
	return;

    bat[0].pos.x-= r->r_left;
    bat[0].pos.y-= r->r_top;
    
    if (TTab[mode][pat2ix(pat)].mode) {
	NewPixrect(&tmppr, r->r_width, r->r_height, TRUE, FALSE);
	pr_batchrop(tmppr, 0, 0, OR|NCLIP, bat, n);
	Stencil(port, r, tmppr, pat, mode);
    } else {
	int rop= TTab[mode][pat2ix(pat)].rop;
	lock(port, r);
	if (port->pr)
	    pr_batchrop(port->pr, r->r_left+PROX, r->r_top+PROY, rop, bat, n);
	else
	    pw_batchrop(port->pw, r->r_left, r->r_top, rop, bat, n);
	unlock(port);
    }
}

/*-----------------------------------------------------------------------------*/

void _SunWindowPort_DevFullscreen(port, b)
register struct SunWindowPort *port;
bool b;
{
    if (port->overlay)
	return;
    if (b) {
	if ((FS= fullscreen_init(port->fd)) == NULL)
	    Fatal("SunWindowPort::Fullscreen", "can't get fullscreen access");
	port->pw= FS->fs_pixwin;
    } else {
	fullscreen_destroy(FS);
	port->pw= port->opw;
	FS= 0;
    }
}

/*---- Scrolling --------------------------------------------------------------*/

void _SunWindowPort_DevScrollRect(port, r, p)
register struct SunWindowPort *port;
struct rect r;
struct point p;
{
    register int l= r.r_left, t= r.r_top;
    bool hard;
    
    if (hard= (p.p_x != 0 && p.p_y != 0))
	NewPixrect(&tmppr, r.r_width, r.r_height, FALSE, FALSE);
    
    lock(port, &port->cliprect);
    if (port->overlay) {
	l+= PROX;
	t+= PROY;
	if (hard) {
	    pr_rop(tmppr, 0, 0, r.r_width, r.r_height, COPY,
					      port->pr, l-p.p_x, t-p.p_y);
	    pr_rop(port->pr, l, t, r.r_width, r.r_height, COPY, tmppr, 0, 0);
	} else
	    pr_rop(port->pr, l, t, r.r_width, r.r_height, COPY,
					      port->pr, l-p.p_x, t-p.p_y);
    } else {
	register struct rectnode *rlp;
	struct rect rl[20];
	int i, j;
	
	if (hard)
	    pw_read(tmppr, 0, 0, r.r_width, r.r_height, COPY, port->pw,
							    l-p.p_x, t-p.p_y);
	else
	    pw_copy(port->pw, l, t, r.r_width, r.r_height, COPY,
						port->pw, l-p.p_x, t-p.p_y);

	for(i= 0, rlp= port->pw->pw_fixup.rl_head; rlp; rlp= rlp->rn_next, i++) {
	    rl[i]= rlp->rn_rect;
	    rl[i].r_left+= l;
	    rl[i].r_top+= t;
	    if (p.p_x > 0)
		rl[i].r_left-= p.p_x;
	    if (p.p_y > 0)
		rl[i].r_top-= p.p_y;
	}
	if (hard)
	    pw_rop(port->pw, l, t, r.r_width, r.r_height, COPY, tmppr, 0, 0);
	for(j= 0; j<i; j++)
	    _WindowPort_Damage(port, eFlgDamage4, &rl[j]);
    }
    unlock(port);
}

/*---- Cursor state -----------------------------------------------------------*/

void _SunWindowPort_DevSetCursor(port, c)
register struct SunWindowPort *port;
GrCursor c;
{
    struct cursor curs;

    curs.cur_function= XOR;
    if (c <= 0) {
	curs.cur_shape= 0;
	curs.cur_yhot= curs.cur_xhot= 0;
    } else {
	curs.cur_xhot= -cursors->pf_char[c-1].pc_home.x;
	curs.cur_yhot= -cursors->pf_char[c-1].pc_home.y;
	curs.cur_shape= cursors->pf_char[c-1].pc_pr;
    }
    curs.flags= 0;
    ioctl(port->fd, WINSETCURSOR, &curs);
}

/*---- init ------------------------------------------------------------------*/

void _SunWindowPort_SunWindowPortInit(port, size)
register struct SunWindowPort *port;
int size;
{
    if (sizeof(struct SunWindowPort) != size)
	Fatal("SunWindowPort::SunWindowPortInit", "size mismatch: %d != %d", sizeof(struct SunWindowPort), size);
    port->rpr= port->pr= port->opr= 0;
    port->rpw= port->pw= port->opw= 0;
    if (!port->overlay)
	port->fd= win_getnewwindow();
}

/*---- clipping --------------------------------------------------------------*/

void _SunWindowPort_DevResetClip(port)
register struct SunWindowPort *port;
{
    if (port->pr) {
	port->pr= port->opr;
	port->fsorigin.p_x= port->fsorigin.p_y= 0;
    }
    if (! port->overlay)
	port->pw= port->opw;
}

void _SunWindowPort_DevClip(port, r, p)
register struct SunWindowPort *port;
struct rect r;
struct point p;
{
    port->cliprect= r;
    if (port->pr) {
	if (port->rpr)
	    pr_close(port->rpr);
	port->pr= port->rpr= pr_region(port->opr, r.r_left, r.r_top, r.r_width,
								    r.r_height);
	/* simulate missing function pr_set_xy_offset() */
	port->fsorigin.p_x= -r.r_left;
	port->fsorigin.p_y= -r.r_top;
    }
    if (! port->overlay) {
	if (port->rpw)
	    pw_set_region_rect(port->rpw, &r, 1);
	else
	    port->rpw= pw_region(port->opw, r.r_left, r.r_top, r.r_width, r.r_height);
	port->pw= port->rpw;
	pw_set_xy_offset(port->pw, r.r_left, r.r_top);
    }
}

/*---- Batching ---------------------------------------------------------------*/

void _SunWindowPort_DevGiveHint(port, code, len, vp)
register struct SunWindowPort *port;
int code, len;
void *vp;
{    
    struct rect *r= (struct rect*) vp;
    
    switch (code) {
    case eHintLock:
	if (!gDebug && !port->inbatch)
	    pw_lock(port->pw, r);
	break;
	
    case eHintUnlock:
	if (!gDebug && !port->inbatch)
	    pw_unlock(port->pw);
	break;
	
    case eHintBatch:
	if (port->inbatch || port->overlay)
	    return;
	port->inbatch= TRUE;
	batchrect= *r;
	NewPixrect(&batchpr, r->r_left+r->r_width, r->r_top+r->r_height, FALSE, FALSE);
	port->pr= port->opr= batchpr;
	port->fsorigin.p_x= port->fsorigin.p_y= 0;
	break;
	
    case eHintUnbatch:
	if (! port->inbatch || port->overlay)
	    return;
	port->inbatch= FALSE;
	pw_rop(port->pw, batchrect.r_left, batchrect.r_top,
			batchrect.r_width, batchrect.r_height, COPY,
				    batchpr, batchrect.r_left, batchrect.r_top);
	port->pr= port->opr= 0;
	port->fsorigin.p_x= port->fsorigin.p_y= 0;
	break;
	
    default:
	break;
    }
}
    
void _SunWindowPort_DevHide(p)
register struct SunWindowPort *p;
{
    if (p->overlay) {
	struct rect rr;
	struct SunWindowPort *fp= p->fatherport;
	
	/* restore the pixels */
	rr= p->savepr->r;
	rr.r_left= rr.r_top= 0;
	pw_lock(p->pw, &rr);
	pw_restore_pixels(p->pw, p->savepr);
	if (fp->overlay && (fp->cursor != p->cursor))
	    _SunWindowPort_DevSetCursor(p, fp->cursor);
	pw_unlock(p->pw);
	
	if (! fp->overlay) {
	    fullscreen_destroy(FS);
	    FS= 0;
	}

	if (p->rpw)
	    pw_close(p->rpw);
	if (p->opw)
	    pw_close(p->opw);
	p->rpw= p->pw= p->opw= 0;
	if (p->rpr)
	    pr_close(p->rpr);
	if (p->opr)
	    pr_close(p->opr);
	p->rpr= p->pr= p->opr= 0;
	
    } else {
	if (allwindows == p)
	    allwindows= p->next;
	else {
	    register struct SunWindowPort *wp;
	    for (wp= allwindows; wp; wp= wp->next) {
		if (wp->next == p) {
		    wp->next= p->next;
		    break;
		}
	    }
	}
	p->next= 0;
	win_remove(p->fd);
    }
    overlayport= 0;
}

static void SetMask(port)
register struct SunWindowPort *port;
{
    register short i;
    struct inputmask Mask;

    input_imnull(&Mask);
    Mask.im_flags= IM_ASCII | IM_NEGEVENT | IM_META;

    for(i = KEY_LEFTFIRST; i <= VKEY_LASTFUNC; i++)
	win_setinputcodebit(&Mask,i);
    for(i = BUT_FIRST; i <= BUT_LAST; i++)
	win_setinputcodebit(&Mask,i);
    win_setinputcodebit(&Mask, LOC_WINENTER);
    win_setinputcodebit(&Mask, LOC_WINEXIT);
    win_setinputcodebit(&Mask, LOC_MOVE);
    win_setinputcodebit(&Mask, LOC_MOVEWHILEBUTDOWN);
    win_setinputmask(port->fd, &Mask, (struct inputmask*) NULL, WIN_NULLLINK);
}

void _SunWindowPort_DevShow(p, fp, r)
register struct SunWindowPort *p, *fp;
struct rect r;
{
    struct rect fsr, rr;
    int fd;
    
    fd= p->fd;
    p->fatherport= fp;
    
    overlayport= 0;
    if (! p->overlay) {
	int parentNo, topmostNo;
    
	p->next= allwindows;
	allwindows= p;
	
	p->myrect.r_left= p->myrect.r_top= p->myrect.r_width=
							p->myrect.r_height= -1;
	win_setrect(fd, &r);    
	
	parentNo= win_nametonumber(parentName);
	topmostNo= win_getlink(rootFd, WL_TOPCHILD);
    
	/*
	 * reset the links
	 */
	win_setlink(fd, WL_COVERING, WIN_NULLLINK); 
	win_setlink(fd, WL_OLDESTCHILD, WIN_NULLLINK); 
	win_setlink(fd, WL_YOUNGESTCHILD, WIN_NULLLINK); 
	win_setlink(fd, WL_PARENT, parentNo);
	win_setlink(fd, WL_COVERED, topmostNo);
	win_insert(fd);
	
	p->opw= p->pw= pw_open(fd);
	_SunWindowPort_DevSetCursor(p, BoldArrow);
	SetMask(p);
	return;
    }
    if (fp == 0)
	Fatal("SunWindowPort::DevShow", "fatal error (fp == 0)");
    overlayport= p;
    p->fd= fp->fd;
    p->cursor= fp->cursor;
    
    if (FS == 0 || (p->fd != FS->fs_windowfd))
	if ((FS= fullscreen_init(p->fd)) == NULL)
	    Fatal("SunWindowPort::DevShow", "can't get fullscreen access");

    p->pr= p->opr= pr_region(FS->fs_pixwin->pw_pixrect, r.r_left, r.r_top, r.r_width, r.r_height);
    p->fsorigin.p_x= p->fsorigin.p_y= 0;
    
    rr= r;
    r.r_left+= FS->fs_screenrect.r_left;
    r.r_top+= FS->fs_screenrect.r_top;
    fsr= r;
    
    p->opw= p->pw= pw_region(FS->fs_pixwin, r.r_left, r.r_top, r.r_width, r.r_height);
    p->inpOffset.p_x= -r.r_left;
    p->inpOffset.p_y= -r.r_top;
    
    fsr.r_left= fsr.r_top= 0;

    pw_lock(p->pw, &fsr);
    p->savepr= pw_save_pixels(fp->pw, &r);  /* now save the pixels */
    /* reset cursor (in lock -> cursor is invisible) */
    _WindowPort_Damage(p, eFlgDamage5, &rr);
    /*
    rr.r_left= rr.r_top= 0;
    _WindowPort_Damage(p, eFlgDamage2, &rr);
    */
    _SunWindowPort_DevSetCursor(p, BoldArrow);
    pw_unlock(p->pw);
}

/*
 * bring window to top of window stack
 */
    
void _SunWindowPort_DevTop(port, top)
register struct SunWindowPort *port;
BOOL top;
{
    if (! port->overlay) {
	if (top)
	    wmgr_top(port->fd, rootFd);
	else
	    wmgr_bottom(port->fd, rootFd);
    }
}

void _SunWindowPort_DevGrab(port, m)
register struct SunWindowPort *port;
BOOL m;
{
    if (port->overlay)
	return;
    
    if ((fdgrab[port->fd] != 0) != (m != 0)) {
	if (m) {
	    fdgrab[port->fd]= TRUE;
	    win_grabio(port->fd);
	} else {
	    fdgrab[port->fd]= FALSE;
	    win_releaseio(port->fd);
	}
    }
}

void _SunWindowPort_DevMoveMousePos(port, d)
register struct SunWindowPort *port;
struct point d;
{
    d.p_x+= win_get_vuid_value(port->fd, LOC_X_ABSOLUTE);
    d.p_y+= win_get_vuid_value(port->fd, LOC_Y_ABSOLUTE);
    win_setmouseposition(port->fd, d.p_x, d.p_y);
}

void _SunWindowPort_DevSetMousePos(port, p)
register struct SunWindowPort *port;
struct point p;
{
    win_setmouseposition(port->fd, p.p_x, p.p_y);
}

void _SunWindowPort_DevBell(port, wait)
register struct SunWindowPort *port;
long wait;
{
    struct timeval t;                 
    
    t.tv_sec= wait / 1000;
    t.tv_usec= wait % 1000;
    win_bell(port->fd, t, port->pw); 
}

void TestDamage()
{
    register struct SunWindowPort *p;
    struct rect s;
    
    for (p= allwindows; p; p= p->next) {
	win_getrect(p->fd, &s);
	pw_damaged(p->opw);
	if (s.r_width != p->myrect.r_width || s.r_height != p->myrect.r_height) {
	    pw_donedamaged(p->opw);
	    p->myrect= s;
	    _WindowPort_Damage(p, eFlgDamage5, &s);
	} else {                        
	    s= p->opw->pw_clipdata->pwcd_clipping.rl_bound;
	    if (s.r_width > 0 && s.r_height > 0)
		_WindowPort_Damage(p, eFlgDamage2, &s);
	    pw_donedamaged(p->opw);
	}
    }
}

void _SunWindowPort_DevSetRect(port, r)
register struct SunWindowPort *port;
struct rect *r;
{
    struct rect o;
	
    if (port->overlay)
	return;
    win_getrect(port->fd, &o);
    if ((r->r_left != o.r_left || r->r_top != o.r_top) &&
	    (r->r_width == o.r_width && r->r_height == o.r_height)) {
	_WindowPort_Damage(port, eFlgDamage3, r);
	port->myrect= *r;
    }
    if (wmgr_iswindowopen(port->fd))
	wmgr_completechangerect(port->fd, r, &o, 0, 0);
    else
	win_setrect(port->fd, r);
};

void _SunWindowPort_FreeResources(port)
register struct SunWindowPort *port;
{
    if (port->overlay)
	return;
    win_remove(port->fd);
    if (port->rpw)
	pw_close(port->rpw);
    if (port->opw)
	pw_close(port->opw);
    if (port->rpr)
	pr_close(port->rpr);
    if (port->pr)
	pr_close(port->pr);
    close(port->fd);
}

static bool myread(fd, millisec, ie)
int fd, millisec;
struct inputevent *ie;
{
    fd_set readready;
    struct timeval t;
    int rval;
		   
    if (millisec < 0) { /* block */
	if (ie)
	    input_readevent(fd, ie);
	return FALSE;
    }
    if (millisec == 0)
	t.tv_sec= t.tv_usec= 0L;
    else {
	t.tv_sec= (long) (millisec / 1000);
	t.tv_usec= (long) (millisec % 1000);
    }
    for (;;) {
	FD_ZERO(&readready);
	FD_SET(fd, &readready);
	if ((rval= select(fd+1, &readready, 0, 0, &t)) < 0) {
	    if (errno != EINTR)
		perror("select");
	    continue;
	}
	if (rval == 0) /* timeout */
	    return TRUE;
	if (ie) {
	    if (FD_ISSET(fd, &readready))
		input_readevent(fd, ie);
	    return FALSE;
	}
    }
}

/*
 * get the next input event and map it to the token structure
 */
 
void _SunWindowPort_DevGetEvent(port, t, timeout, overread)
register struct SunWindowPort *port;
register Token *t;
int timeout;
BOOL overread;
{
    struct inputevent ie; /* buffer to read an event */
    bool first;

    t->Code= eEvtNone;
    t->Flags= 0;
    t->Pos= port->lastpos;
    
    if (overread) {
	first= TRUE;
	do {
	    if (myread(port->fd, timeout, &ie))
		break;
	    first= FALSE;
	} while (event_id(&ie) == eEvtLocMove || event_id(&ie) == eEvtLocMoveBut);
	if (first)
	    return;    /* no events in queue */
    } else if (myread(port->fd, timeout, &ie))
	return;    /* no events in queue */
    
    t->Code= event_id(&ie);
    t->At= event_time(&ie).tv_sec * 1000 + event_time(&ie).tv_usec / 1000;
    t->Pos.p_x= event_x(&ie) + port->inpOffset.p_x;
    t->Pos.p_y= event_y(&ie) + port->inpOffset.p_y;
    port->lastpos= t->Pos;

    if (event_shift_is_down(&ie))
	t->Flags |= (int)eFlgShiftKey;
    if (event_ctrl_is_down(&ie))
	t->Flags |= (int)eFlgCntlKey;
    if (win_inputnegevent(&ie)) 
	t->Flags |= (int)eFlgButDown;
    if (event_meta_is_down(&ie)) 
	t->Flags |= (int)eFlgMetaKey;
       
    /* map cursor key escape sequences to one input token
     * (cannot map the keyboard, because mapping affects all windows !!)
     * check whether a cursor sequence (ESC '[' ( 'A' | 'B' | 'C' | 'D' ) ) occured
     */
    if (t->Code == 033) {
Start: 
	for(;;) {
	    if (myread(port->fd, 0, &ie))
		break;
	    if (ie.ie_code != 033)
		break;              /* sometimes more ESC are received */
	}
	if (ie.ie_code != '[') {
	    t->Code= ie.ie_code;
	    return;
	}
    
	for(;;) {
	    if (myread(port->fd, 0, &ie))
		break;
	    if (ie.ie_code == 033)
		goto Start; /* the sequence restarts */
	}
	switch ((char)ie.ie_code) {
	case 'A': 
	    t->Code= (int) eEvtCursorUp;
	    break;
	case 'B': 
	    t->Code= (int) eEvtCursorDown;
	    break;
	case 'C': 
	    t->Code= (int) eEvtCursorRight;
	    break;
	case 'D': 
	    t->Code= (int) eEvtCursorLeft;
	    break;
	default : 
	    t->Code= ie.ie_code;
	    break;
	}
    }
}
