#ifndef Port_First
#define Port_First

#include "Root.h"
#include "Types.h"
#include "Point.h"
#include "Metric.h"
#include "Rectangle.h"
#include "Font.h"
#include "Bitmap.h"

const int MaxTextBatchCnt= 400;

enum PortType {
    ePortOutput = 1,
    ePortWindow = 2,
    ePortPrinter= 4
};

enum GrHints {
    eHintLock,
    eHintUnlock,
    eHintBatch,
    eHintUnbatch,
    eHintTextBatch,
    eHintTextUnbatch,
    eHintPostScript,
    eHintPic,
    eHintFlush
};

enum GrMode { 
    eRopCopy = 0,
    eRopXor  = 1
};

enum GrPattern {
    ePatNone  = 0,
    ePatWhite = 1,
    ePatBlack = 2,
    ePatGrey87= 3, ePatGrey75= 4, ePatGrey60= 5, ePatGrey50= 6, ePatGrey40= 7,
    ePatGrey25= 8, ePatGrey12= 9,
    ePat00 = 10, ePat01 = 11, ePat02 = 12, ePat03 = 13, ePat04 = 14,
    ePat05 = 15, ePat06 = 16, ePat07 = 17, ePat08 = 18, ePat09 = 19,
    ePat10 = 20, ePat11 = 21, ePat12 = 22, ePat13 = 23, ePat14 = 24,
    ePat15 = 25
};

enum GrLineCap {
    eDefaultCap= 0,
    eStartArrow= 1,
    eEndArrow  = 2
};

enum GrPolyType {
    ePolyBezier  = BIT(0),
    ePolySpline2 = BIT(1),
    ePolySpline3 = BIT(2),
    ePolyClose   = BIT(3),
    ePolyPartial = BIT(4),
    
    ePolyDefault = 0
};

enum GrCursor {
    eCrsNone       =  0, 
    eCrsArrow      =  1,
    eCrsBoldArrow  =  2,
    eCrsCross      =  3,
    eCrsBoldCross  =  4,
    eCrsUpArrow    =  5,
    eCrsDownArrow  =  6,
    eCrsRightArrow =  7,
    eCrsLeftArrow  =  8,
    eCrsHourglass  =  9,
    eCrsPen        = 10,
    eCrsMoveHand   = 11,
    eCrsHand       = 12,
    eCrsCrossHair  = 13,
    eCrsMoveStretch= 14,
    eCrsIBeam      = 15,
    eCrsLeftRightArrow  = 18,
    eCrsUpDownArrow= 19
};

//---- editing  characters of tty
extern char gRubout;
extern char gBackspace;

#include "Token.h"

typedef void (*InpHandlerFun)(void*, void*, Token*);     
    // event handler function

extern class Port *tbport;

class Port : public Root {    
protected:
public:
    byte type;

    /* graphic state */
    Point origin;
    Rectangle cliprect;
    bool inclip, pendingclip;
    short mode;
    short patid;
    
    /* pen state */
    Point penpos;
    short pensize;
    short penmode;
    short penpatid;
    short pencap;
    
    /* text state */
    short textmode;
    Point textpos;
    short textpatid;
    FontPtr textfd;
	  
    void flushtext();

protected:
    Port(int code= ePortOutput);
    
    bool Visible(GrPattern pat, Rectangle &r)
	{ return (pat != ePatNone && (!inclip || cliprect.Intersects(r))); }

    void FlushClip()
	 { if (pendingclip) { DevClip(cliprect, origin); pendingclip= FALSE; } }

    void FlushMyText()
	{ if (tbport == this) flushtext(); }
		
    void FlushAnyText()
	{ if (tbport) tbport->flushtext(); }

    virtual void DrawObject(char, GrPattern, GrMode, Rectangle*, Point, int, GrLineCap);
    virtual void DrawPolygon(char, GrPattern, GrMode, Rectangle*, Point*, int, GrPolyType, int, GrLineCap);

public:
    ~Port();
    void PortInit();
    int Type()
	{ return type; }

    //---- device dependent primitives ------------------------------------------
    virtual void DevDestroy();
    virtual void DevClip(Rectangle, Point);
    virtual void DevResetClip();
    virtual void DevStrokeLine(GrPattern, GrMode, int, Rectangle*, GrLineCap, Point, Point);
    virtual void DevStrokeRect(GrPattern, GrMode, int, Rectangle*);
    virtual void DevFillRect(GrPattern, GrMode, Rectangle*);
    virtual void DevStrokeOval(GrPattern, GrMode, int, Rectangle*);
    virtual void DevFillOval(GrPattern, GrMode, Rectangle*);
    virtual void DevStrokeRRect(GrPattern, GrMode, int, Rectangle*, Point);
    virtual void DevFillRRect(GrPattern, GrMode, Rectangle*, Point);
    virtual void DevStrokeWedge(GrPattern, GrMode, int, GrLineCap, Rectangle*, int, int);
    virtual void DevFillWedge(GrPattern, GrMode, Rectangle*, int, int);
    virtual void DevStrokePolygon(Rectangle*, GrPattern, GrMode, Point*, int, GrPolyType t,
								int, GrLineCap);
    virtual void DevFillPolygon(Rectangle*, GrPattern, GrMode, Point*, int, GrPolyType t);
    virtual void DevShowBitmap(GrPattern, GrMode, Rectangle*, struct Bitmap*);
    virtual bool DevShowChar(FontPtr, Point, byte, bool, Point);
    virtual void DevShowTextBatch(GrPattern, GrMode, Rectangle*, Point);
    virtual void DevShowPicture(Rectangle *r, class Picture *pic);
    virtual void DevGiveHint(int code, int len, void *vp);
    //---- end of device dependent primitives ----------------------------------


    //---- device independent primitives ---------------------------------------
    virtual GrCursor SetCursor(GrCursor c);
    virtual GrCursor SetWaitCursor(unsigned int d= 400, GrCursor c= eCrsHourglass);
    virtual GrCursor GetCursor();

    void Destroy()
	{ DevDestroy(); }
    void Clip(Rectangle, Point);
    void ResetClip();
    void StrokeLine(GrPattern, GrMode, int, GrLineCap, Point, Point);
    void StrokeRect(GrPattern, GrMode, int, Rectangle*);
    void FillRect(GrPattern, GrMode, Rectangle*);
    void StrokeOval(GrPattern, GrMode, int, Rectangle*);
    void FillOval(GrPattern, GrMode, Rectangle*);
    void StrokeRRect(GrPattern, GrMode, int, Rectangle*, Point);
    void FillRRect(GrPattern, GrMode, Rectangle*, Point);
    void StrokeWedge(GrPattern, GrMode, int, GrLineCap, Rectangle*, int, int);
    void FillWedge(GrPattern, GrMode, Rectangle*, int, int);
    void StrokePolygon(Point, GrPattern, GrMode, Point*, int, GrPolyType, int, GrLineCap);
    void FillPolygon(Point, GrPattern, GrMode, Point*, int, GrPolyType);
    void ShowTextBatch(GrPattern, GrMode, FontPtr, Rectangle*, Point, int, byte*);
    void ShowString(FontPtr, GrPattern, GrMode, Point, byte*, int);
    void ShowBitmap(GrPattern, GrMode, Rectangle*, struct Bitmap*);
    void GiveHint(int code, int len, void *vp);
    void ShowPicture(Rectangle*, Picture*);
    //---- end of device independent primitives --------------------------------

    //---- clipping ------------------------------------------------------------
    Point GetOrigin()
	{ return origin; }
    Rectangle GetCliprect()
	{ return cliprect; }

    //---- Fill -----------------------------------------------------------------
    void SetNormal()
	{ patid= ePatBlack; mode= eRopCopy; }
    void SetPattern(GrPattern p)
	{ patid= p; }
    void SetMode(GrMode m)
	{ mode= m; }

    //---- Pen and Lines --------------------------------------------------------
    void SetPenNormal();
    void SetPenPattern(GrPattern p)
	{ penpatid= p; }
    void SetLineCap(GrLineCap lc)
	{ pencap= lc; }
    void SetPenMode(GrMode m)
	{ penmode= m; }
    void SetPenSize(int w)
	{ pensize= w; }
    void Moveto(Point p)
	{ penpos= p; }
    void Moveby(Point p)
	{ penpos+= p; }
    void Line(Point p1, Point p2)
	{ StrokeLine(penpatid, penmode, pensize, pencap, p1, p2); }
    void PaintLine(GrPattern pat, GrLineCap cap, Point p1, Point p2)
	{ StrokeLine(pat, eRopCopy, pensize, cap, p1, p2); }
    void Lineto(Point);

    //---- Text -----------------------------------------------------------------
    void SetTextNormal();
    void DrawChar(byte);
    void DrawString(byte*, int);
    void SetFamily(GrFont);
    void SetFont(FontPtr fd)
	{ textfd= fd; }
    FontPtr GetFont()
	{ return textfd; }
    void SetSize(int);
    void SetFace(GrFace);
    void SetTextMode(GrMode m)
	{ textmode= m; }
    void SetTextPattern(GrPattern p)
	{ textpatid= p; }
    Point GetTextPos()   
	{ return textpos; }
    void TextMoveto(Point p)
	{ textpos= p; }
    void TextMoveby(Point p)
	{ textpos+= p; }
    void TextHMoveby(int h)
	{ textpos.x+= h; }
    void TextVMoveby(int v)
	{ textpos.y+= v; }
};

typedef Port *PortDesc;
extern PortDesc port;

//---- Port syntactic sugar ----------------------------------------------------

inline void GrSetPort(PortDesc newport)
    { port= newport; }
inline PortDesc GrGetPort()
    { return port; }

//---- cursor -------------------------------------------------------------------
inline GrCursor GrSetCursor(GrCursor c)
    { return port->SetCursor(c); }
inline GrCursor GrGetCursor()
    { return port->GetCursor(); }
inline GrCursor GrSetWaitCursor(unsigned int d= 400, GrCursor c= eCrsHourglass)
    { return port->SetWaitCursor(d, c); }

//---- clipping -----------------------------------------------------------------

inline void GrSetClip(Rectangle r, Point o)
    { port->Clip(r, o); }

//---- implementation hints -----------------------------------------------------
inline void GrGiveHint(int n, int l= 0, void *v= 0)
    { port->GiveHint(n, l, v); }

//---- pens ---------------------------------------------------------------------
inline void GrSetPenSize(int w)
    { port->SetPenSize(w); }
inline void GrSetPenMode(GrMode m)
    { port->SetPenMode(m); }
inline void GrSetLineCap(GrLineCap lc)
    { port->SetLineCap(lc); }
inline void GrSetPenPattern(GrPattern p)
    { port->SetPenPattern(p); }
inline void GrSetPenNormal()
    { port->SetPenNormal(); }

//---- lines --------------------------------------------------------------------
inline void GrMoveto(Point p)
    { port->Moveto(p); }
inline void GrLineto(Point p)
    { port->Lineto(p); }
inline void GrLine(Point p1, Point p2)
    { port->StrokeLine(port->penpatid, port->penmode, port->pensize, port->pencap, p1, p2); }
inline void GrPaintLine(GrPattern pat, GrMode m, int psz, GrLineCap cap, Point p1, Point p2)
    { port->StrokeLine(pat, m, psz, cap, p1, p2); }
inline void GrStrokeLine(GrPattern pat, GrMode m, GrLineCap cap, Point p1, Point p2)
    { port->StrokeLine(pat, m, port->pensize, cap, p1, p2); }
inline void GrInvertLine(Point p1, Point p2)
    { port->StrokeLine(ePatBlack, eRopXor, port->pensize, port->pencap, p1, p2); }
inline void GrEraseLine(Point p1, Point p2)
    { port->StrokeLine(ePatWhite, eRopCopy, port->pensize, port->pencap, p1, p2); }

//---- fill ---------------------------------------------------------------------
inline void GrSetMode(GrMode m)
    { port->SetMode(m); }
inline void GrSetPattern(GrPattern p)
    { port->SetPattern(p); }

//---- rectangle ----------------------------------------------------------------
inline void GrFillRect(Rectangle r)
    { port->FillRect(port->patid, port->mode, &r); }

inline void GrStrokeRect(Rectangle r)
    { port->StrokeRect(port->penpatid, port->penmode, port->pensize, &r); }

inline void GrEraseRect(Rectangle r)
    { port->FillRect(ePatWhite, eRopCopy, &r); }

inline void GrInvertRect(Rectangle r)
    { port->FillRect(ePatBlack, eRopXor, &r); }

inline void GrPaintRect(Rectangle r, GrPattern pat)
    { port->FillRect(pat, eRopCopy, &r); }

//---- round rectangles --------------------------------------------------------

inline void GrStrokeRoundRect(Rectangle r, Point dia)
    { port->StrokeRRect(port->penpatid, port->penmode, port->pensize, &r, dia); }

inline void GrFillRoundRect(Rectangle r, Point dia)
    { port->FillRRect(port->patid, port->mode, &r, dia); }

inline void GrEraseRoundRect(Rectangle r, Point dia)
    { port->FillRRect(ePatWhite, eRopCopy, &r, dia); }

inline void GrInvertRoundRect(Rectangle r, Point dia)
    { port->FillRRect(ePatBlack, eRopXor, &r, dia); }

inline void GrPaintRoundRect(Rectangle r, Point dia, GrPattern pat)
    { port->FillRRect(pat, eRopCopy, &r, dia); }

//----- ovals ------------------------------------------------------------------
inline void GrStrokeOval(Rectangle r)
    { port->StrokeOval(port->penpatid, port->penmode, port->pensize, &r); }

inline void GrFillOval(Rectangle r)
    { port->FillOval(port->patid, port->mode, &r); }

inline void GrEraseOval(Rectangle r)
    { port->FillOval(ePatWhite, eRopCopy, &r); }

inline void GrInvertOval(Rectangle r)
    { port->FillOval(ePatBlack, eRopXor, &r); }

inline void GrPaintOval(Rectangle r, GrPattern pat)
    { port->FillOval(pat, eRopCopy, &r); }

//----- wedges -----------------------------------------------------------------

inline void GrStrokeWedge(Rectangle r, int s, int e)
    { port->StrokeWedge(port->penpatid, port->penmode, port->pensize, port->pencap, &r, s, e); }

inline void GrFillWedge(Rectangle r, int s, int e)
    { port->FillWedge(port->patid, port->mode, &r, s, e); }

inline void GrEraseWedge(Rectangle r, int s, int e)
    { port->FillWedge(ePatWhite, eRopCopy, &r, s, e); }

inline void GrInvertWedge(Rectangle r, int s, int e)
    { port->FillWedge(ePatBlack, eRopXor, &r, s, e); }

inline void GrPaintWedge(Rectangle r, int s, int e, GrPattern pat)
    { port->FillWedge(pat, eRopCopy, &r, s, e); }

//---- bitmaps -----------------------------------------------------------------

inline void GrShowBitMap(Rectangle r, Bitmap *bmp)
    { port->ShowBitmap(port->patid, port->mode, &r, bmp); }

inline void GrPaintBitMap(Rectangle r, Bitmap *bmp, GrPattern pat)
    { port->ShowBitmap(pat, eRopCopy, &r, bmp); }

inline void GrInvertBitMap(Rectangle r, Bitmap *bmp)
    { port->ShowBitmap(ePatBlack, eRopXor, &r, bmp); }

//--- Polygon ------------------------------------------------------------------

inline void GrStrokePolygon(Point at, Point *pts, int npts, GrPolyType t)
    { port->StrokePolygon(at, port->penpatid, port->penmode, pts, npts, t,
					    port->pensize, port->pencap); }

inline void GrFillPolygon(Point at, Point *pts, int npts, GrPolyType t)
    { port->FillPolygon(at, port->patid, port->mode, pts, npts, t); }

inline void GrErasePolygon(Point at, Point *pts, int npts, GrPolyType t)
    { port->FillPolygon(at, ePatWhite, eRopCopy, pts, npts, t); }

inline void GrInvertPolygon(Point at, Point *pts, int npts, GrPolyType t)
    { port->FillPolygon(at, ePatBlack, eRopXor, pts, npts, t); }

inline void GrPaintPolygon(Point at, GrPattern pat, Point *pts, int npts, GrPolyType t)
    { port->FillPolygon(at, pat, eRopCopy, pts, npts, t); }


//---- text --------------------------------------------------------------------
inline void GrSetFont(FontPtr fd)
    { port->SetFont(fd); }
inline FontPtr GrGetFont()
    { return port->GetFont(); }
inline void GrSetFamily(GrFont f)
    { port->SetFamily(f); }
inline void GrSetSize(int s)
    { port->SetSize(s); }
inline void GrSetFace(GrFace f)
    { port->SetFace(f); }
inline void GrDrawChar(byte c)
    { port->DrawChar(c); }
inline void GrDrawString(byte *s, int l= -1)
    { port->DrawString(s, l); }
inline void GrShowString(FontPtr fd, GrPattern pat, GrMode mode, Point pos,
							 byte *text, int l= -1)
    { port->ShowString(fd, pat, mode, pos, text, l); }
inline void GrSetTextMode(GrMode mode)
    { port->SetTextMode(mode); }
inline void GrSetTextPattern(GrPattern patid)
    { port->SetTextPattern(patid); }
inline void GrTextMoveto(Point p)
    { port->TextMoveto(p); }
inline void GrTextAdvance(int h)
    { port->TextHMoveby(h); }
inline Point GrGetTextPos()
    { return port->GetTextPos(); }

//---- pictures ----------------------------------------------------------------
inline void GrShowPicture(Rectangle *r, Picture *pic)
    { port->ShowPicture(r, pic); }

#endif Port_First
