//$CellSelector,CollectionView$
#include "CollectionView.h"
#include "Menu.h"
#include "String.h"
#include "Collection.h"
#include "VObject.h"
#include "CmdNo.h"
#include "Error.h"
#include <math.h>

//---- CollectionView ----------------------------------------------------------

MetaImpl(CollectionView, (I_O(matrix), I_R(selection), I_P(gap), I_P(minExtent),
			    I_S(rows), I_S(cols), I_O(defaultItem), I_O(menu)));

CollectionView::CollectionView(EvtHandler *eh, Collection *m, CollViewOptions o,
						    int r, int c) : (eh, gRect0)
{
    SetFlag(o);
    defaultItem= new VObject();
    rows= r;
    cols= c;
    SetCollection(m);
}

CollectionView::~CollectionView()
{
    if (matrix) {
	matrix->FreeAll();
	SafeDelete(matrix);
    }
    SafeDelete(xPos);
    SafeDelete(yPos);
    SafeDelete(defaultItem);
}

void CollectionView::SetCollection(class Collection* m, bool freeold)
{
    if (matrix)
	matrix->RemoveDependent(this);
    if (freeold && matrix) {
	matrix->FreeAll();
	delete matrix;
    }
    if (matrix= m) {
	matrix->AssertClass(VObject);
	matrix->AddDependent(this);
    }
    selection= gRect0;
    Modified();
    RevealAlign(Rectangle(1,1));
}

void CollectionView::SetDefaultItem(class VObject *d)
{ 
    if (defaultItem)
	delete defaultItem;
    defaultItem= d; 
}

void CollectionView::Modified()
{
    SetFlag(eCVModified);
    if (IsOpen())
	Update();
}

void CollectionView::SetMinExtent(Point e)
{
    if (minExtent != e) {
	minExtent= e;
	Modified();
    }
}

VObject *CollectionView::GetItem(int x, int y)
{
    VObject *gop= (VObject*)matrix->At(x*rows+y);
    return gop ? gop : defaultItem;
}

Rectangle CollectionView::ItemRect(int x, int y)
{
    x= range(0, cols, x);
    y= range(0, rows, y);
    Rectangle r(xPos[x], yPos[y], xPos[x+1]-xPos[x], yPos[y+1]-yPos[y]);
    if (TestFlag(eCVGrid))
	r.extent-= gPoint1;
    r.origin+= GetOrigin();
    return r;
}

void CollectionView::ConstrainScroll(Point *p)
{
    p->y= ItemRect(0, PointToItem(*p).y).origin.y;
}

void CollectionView::DoUpdate(Object *op, void*)
{
    if (op == matrix)
	Modified();
}

Metric CollectionView::GetMinSize()
{
    Update();
    return GetExtent();
}

void CollectionView::Update()
{
    register VObject *gop;
    register int x, y, ww, hh, w, h;
    int xpos= 0, ypos= 0, sz;
    Rectangle r;
    Point g;

    if (!TestFlag(eCVModified))
	return;
    ResetFlag(eCVModified);
    if (matrix == 0)
	return;
    sz= matrix->Size();
    if (sz == 0) {
	ForceRedraw();
	return;
    }    
    if (rows <= 0)
	SetFlag(eCVExpandRows);
    if (cols <= 0)
	SetFlag(eCVExpandCols);
    
    if (TestFlag(eCVExpandRows) && TestFlag(eCVExpandCols)) {
	cols= (int) (sqrt((float)sz)+0.5);
	rows= (sz+cols-1) / cols;
    } else if (TestFlag(eCVExpandRows))
	rows= (sz+cols-1) / cols;
    else if (TestFlag(eCVExpandCols))
	cols= (sz+rows-1) / rows;

    Expand(&yPos, (rows+1) * sizeof(short));
    Expand(&xPos, (cols+1) * sizeof(short));

    xpos= ypos= 0;
    g= 2*gap;
    if (TestFlag(eCVGrid))
	g+= gPoint1;
    
    for (x= 0; x < cols; x++) {
	xPos[x]= xpos;
	ww= minExtent.x;
	for (y= 0; y < rows; y++) {
	    gop= GetItem(x, y);
	    gop->SetView(this);
	    gop->SetContainer(this);
	    gop->CalcExtent();
	    w= gop->Width();
	    ww= max(ww, w);
	}
	xpos+= ww+g.x;
    }
    xPos[x]= xpos;

    for (y= 0; y < rows; y++) {
	yPos[y]= ypos;
	hh= minExtent.y;
	for (x= 0; x < cols; x++) {
	    h= GetItem(x, y)->Height();
	    hh= max(hh, h);
	}
	ypos+= hh+g.y;
    }
    yPos[y]= ypos;

    for (x= 0; x < cols; x++)
	for (y= 0; y < rows; y++)
	    GetItem(x, y)->SetContentRect(ItemRect(x, y).Inset(gap), FALSE);

    if (TestFlag(eCVGrid)) {
	xpos--;
	ypos--;
    }
    SetExtent(Point(xpos, ypos));
    ForceRedraw();
}

void CollectionView::SetOrigin(Point at)
{
    register int x, y;
    
    View::SetOrigin(at);
    at+= gap;
    for (x= 0; x < cols; x++)
	for (y= 0; y < rows; y++)
	    GetItem(x, y)->SetOrigin(Point(xPos[x]+at.x, yPos[y]+at.y));
}

Point CollectionView::ItemPos(VObjPtr g)
{
    VObjPtr gop;
    Iter next(matrix);

    for (int i= 0; gop= (VObject*) next(); i++)
	if (gop == g)
	    break;
    if (i >= matrix->Size())
	return gPoint_1;
    return Point(i/rows, i%rows);
}

void CollectionView::DoSelect(Rectangle r)
{
    if (r.IsNotEmpty()) {
	int partcode= clickCount >= 2 ? cPartCollDoubleSelect: cPartCollSelect;
	Control(GetId(), partcode, (void*) r.origin.y);
	if (TestFlag(eCVClearSelection))
	    SetNoSelection();
    }
}

int CollectionView::SetSelection(Rectangle newsel)
{
    if (selection != newsel) {
	if (IsOpen()) {
	    InvalidateRect(ItemRect(selection));
	    InvalidateRect(ItemRect(newsel));
	}
	selection= newsel;
    }
    return 0;
}

Point CollectionView::PointToItem(Point p, bool *outside)
{
    register int x, y;
    register bool out= FALSE;

    if (matrix == 0) {
	*outside= TRUE;
	return gPoint0;
    }
    p-= GetOrigin();
    if (p.x < xPos[0]) {
	out= TRUE;
	x= 0;
    } else if (p.x >= xPos[cols]) {
	out= TRUE;
	x= cols-1;
    } else {
	for (x= 0; x < cols; x++)
	    if (p.x >= xPos[x] && p.x < xPos[x+1])
		break;
    }
    if (p.y < yPos[0]) {
	out= TRUE;
	y= 0;
    } else if (p.y >= yPos[rows]) {
	out= TRUE;
	y= rows-1;
    } else {
	for (y= 0; y < rows; y++)
	    if (p.y >= yPos[y] && p.y < yPos[y+1])
		break;
    }
    if (outside)
	*outside= out;
    return Point(x, y);
}

VObject *CollectionView::PointToItemPtr(Point p, Point *ip)
{
    bool outside;
    *ip= PointToItem(p, &outside);
    if (outside && TestFlag(eCVDontStuckToBorder))
	return 0;
    return GetItem(ip->x, ip->y);
}

void CollectionView::DoCreateMenu(Menu *m)
{
    if (!menu)
	View::DoCreateMenu(m);
}

Command *CollectionView::DoLeftButtonDownCommand(Point, Token, int c)
{
    clickCount= c;
    if (matrix && matrix->Size() > 0)
	return new CellSelector(this);
    return gNoChanges;
}

extern Clipper *gClipper;   // dirty hack

Command *CollectionView::DoKeyCommand(int ch, Point, Token)
{
    register VObject *gop;
    register int i;
    int start= 0;
    
    if (gClipper) {
	Rectangle r= gClipper->GetViewedRect();
	start= r.origin.y+r.extent.y;
	if (start > GetExtent().y)
	    start= 0;
	start= PointToItem(Point(0,start)).y;
    }

    for (i= start; i != start-1; i= (i+1) % rows) {
	gop= GetItem(0, i);
	if (gop && (gop->AsString()[0] == ch))
	    break;
    }

    if (gop) {
	if (gClipper)
	    gClipper->RevealAlign(ItemRect(Rectangle(ItemPos(gop), gPoint1)));
	else
	    RevealAlign(ItemRect(Rectangle(ItemPos(gop), gPoint1)));
    }
    return gNoChanges;
}

void CollectionView::Draw(Rectangle r)
{
    register int x, y;
    Point p1, p2;

    if (matrix == 0 || matrix->Size() <= 0)
	return;
    p1= PointToItem(r.origin);
    p2= PointToItem(r.origin + r.extent);
    if (TestFlag(eCVGrid)) {
	Point o= GetOrigin();
	GrSetPenNormal();
	for (x= p1.x; x <= p2.x; x++)
	    GrLine(Point(xPos[x]-1, yPos[p1.y])+o, 
					    Point(xPos[x]-1, yPos[p2.y+1]-1)+o);
	for (y= p1.y; y <= p2.y; y++)
	    GrLine(Point(xPos[p1.x], yPos[y]-1)+o, 
					    Point(xPos[p2.x+1]-1, yPos[y]-1)+o);
    }
    for (x= p1.x; x <= p2.x; x++)
	for (y= p1.y; y <= p2.y; y++)
	    GetItem(x, y)->DrawAll(r);
}

void CollectionView::DoHighlightSelection(HighlightState hst)
{
    register int x, y;
    register VObject *gop;

    for (x= selection.origin.x; x < selection.origin.x+selection.extent.x; x++) {
	for (y= selection.origin.y; y < selection.origin.y+selection.extent.y; y++) {
	    gop= GetItem(x, y);
	    if (gop->Enabled())
		gop->Highlight(hst);
	}
    }
}

Rectangle CollectionView::ItemRect(Rectangle r)
{
    Rectangle rr;

    if (r.IsEmpty())
	return gRect0;
    Update();
    rr.origin.x= xPos[r.origin.x];
    rr.origin.y= yPos[r.origin.y];
    rr.extent.x= xPos[r.origin.x+r.extent.x] - rr.origin.x;
    rr.extent.y= yPos[r.origin.y+r.extent.y] - rr.origin.y;
    return rr+GetOrigin();
}

ostream& CollectionView::PrintOn(ostream &s)
{
    Object::PrintOn(s);
    return s << gap SP << minExtent SP << rows SP << cols SP << defaultItem SP << matrix SP;
}

bool CollectionView::PrintOnWhenDependent(Object *from)
{
    return from != matrix;
}

istream& CollectionView::ReadFrom(istream &s)
{
    Collection *m;
    VObject *di;

    Object::ReadFrom(s);
    s >> gap >> minExtent >> rows >> cols >> di >> m;
    SetCollection(m);
    SetDefaultItem(di);
    return s;
}

void CollectionView::InspectorId(char *buf, int sz)
{
    if (matrix && matrix->Size() > 0) 
	matrix->At(0)->InspectorId(buf, sz);    
    else
	View::InspectorId(buf, sz);
}

void CollectionView::DoOnItem(int m, VObject *gop, Point p)
{
}

//---- CellSelector ------------------------------------------------------------

CellSelector::CellSelector(CollectionView* v)
{ 
    lvp= v;
    if (lvp->selection.IsNotEmpty())
	newcell= lvp->GetItem(lvp->selection.origin.x, lvp->selection.origin.y);
}

void CellSelector::TrackFeedback(Point, Point pp, bool on)
{
    if (on) {
	int code= 2;
	if (newcell != lastcell) {
	    code= 1;
	    if (lastcell && lastcell->Enabled())
		lastcell->Highlight(Off);
	    if (newcell && newcell->Enabled())
		newcell->Highlight(On);
	    if (newcell == 0)
		code= 0;
	}
	lvp->DoOnItem(code, newcell, pp);
    }
}

Command *CellSelector::TrackMouse(TrackPhase atp, Point, Point, Point np)
{
    Point item;

    lastcell= newcell;
    newcell= lvp->PointToItemPtr(np, &item);
    if (newcell)
	lvp->selection= Rectangle(item, gPoint1);
    else
	lvp->selection= gRect0;
    switch (atp) {
    case eTrackRelease:
	lvp->DoSelect(lvp->selection);
	return gNoChanges;
    case eTrackExit:
	return gNoChanges;
    default:
	break;
    }
    return this;
}
