//$CodeTextView$

#include "CodeTextView.h"
#include "StyledText.h"

Rectangle gCodeTextViewRect(Point(1000, cFit));

MetaImpl(CodeTextView, (I_B(autoIndent), I_I(cursorPos)));

CodeTextView::CodeTextView(EvtHandler *eh, Rectangle r, Text *t, eTextJust m, eSpacing sp, 
		       TextViewFlags fl, Point b, int id) : 
					(eh, r, t, m, sp, FALSE, fl, b, id) 
{
    SetStopChars("\n\r");
    autoIndent= TRUE;
    Font *fd1, *fd2, *fd3, *fd4;
    fd4= t->GetFont();
    int fid= fd4->Fid();
    int size= fd4->Size();
    fd1= new Font(fid, size, eFacePlain);
    fd2= new Font(fid, size, eFaceItalic);
    fd3= new Font(fid, size, eFaceBold);

    commentStyle= new Style(fd2);
    functionStyle= new Style(fd3);
    classDeclStyle= new Style(fd3);
    plainStyle= new Style(fd1);
    ExitCursorMode();
}   

Command *CodeTextView::DoKeyCommand(int ch, Point lp, Token token)
{
    Command *cmd;

    ExitCursorMode();
    if (ch == '\r' || ch == '\n' && autoIndent) { // auto indent
	int from, to, i;
	Text *t= GetText();
	GetSelection(&from,&to);
	scratchText->Empty();
	scratchText->Append('\n'); 
	for (i= StartLine(CharToLine(from)); i < t->Size() ; i++) {
	    ch= (*t)[i];
	    if (ch == ' ' || ch == '\t')
		scratchText->Append(ch);
	    else
		break;
	}
	cmd= InsertText(scratchText);
	scratchText->Empty();
	RevealSelection();
	return cmd;
    }
    return FixedLineTextView::DoKeyCommand(ch, lp, token);
}

void CodeTextView::SetAutoIndent(bool b)
{
    autoIndent= b;
}
    
bool CodeTextView::GetAutoIndent()
{
    return autoIndent;
}

void CodeTextView::ExitCursorMode()
{
    cursorPos= gPoint_1;
}

Command *CodeTextView::DoLeftButtonDownCommand(Point p, Token t, int cl)
{
    static char *obrackets = "({[\"";
    static char *cbrackets = ")}]\"";

    Point sp= p;
    ExitCursorMode();
    if (!Enabled())
	return gNoChanges;
    if (cl >= 2) {
	int line, cpos;
	char *br;
	Point pp;
	Text *text= GetText();

	p-= GetInnerOrigin();
	PointToPos(p, &pp, &line, &cpos);
	
	if (cpos > 0 && (br= index(obrackets,(*text)[cpos-1]))) {
	    MatchBracketForward(cpos, *br, cbrackets[br-obrackets]);
	    return gNoChanges;
	}
	if (cpos < text->Size() && (br= index(cbrackets,(*text)[cpos]))) {
	    MatchBracketBackward(cpos-1, obrackets[br-cbrackets], *br);
	    return gNoChanges;
	}
    }
    return FixedLineTextView::DoLeftButtonDownCommand(sp, t, cl);
}

Command *CodeTextView::DoMenuCommand(int m)
{
    ExitCursorMode();
    return TextView::DoMenuCommand(m);
}
    
Command *CodeTextView::DoOtherEventCommand(Point p, Token t)
{
    ExitCursorMode();
    return TextView::DoOtherEventCommand(p, t);
}

void CodeTextView::MatchBracketForward(int from, int obracket, int cbracket)
{
    Text *text= GetText();
    int ch, stack= 0;

    for (int i= from; i < text->Size(); i++) {
	ch= (*text)[i];
	if (ch == cbracket) {
	    if (stack-- == 0) 
		break;
	} else if (ch == obracket)
	    stack++;
    }    
    SetSelection(from, i, TRUE);
}

void CodeTextView::MatchBracketBackward(int from, int obracket, int cbracket)
{
    Text *text= GetText();
    int ch, stack= 0;

    for (int i= from; i >= 0; i--) {
	ch= (*text)[i];
	if (ch == obracket) { 
	    if (stack-- == 0) 
		break;
	} else if (ch == cbracket)
	    stack++;
    }
    SetSelection(i+1, from+1, TRUE);
}

Command *CodeTextView::DoCursorKeyCommand(EvtCursorDir cd, Point p, Token t)
{
    switch (cd) {
    case eCrsLeft:
    case eCrsRight:
	ExitCursorMode();
	break;
    default:
	break;
    }
    return TextView::DoCursorKeyCommand(cd, p, t);
}

int CodeTextView::CursorPos(int at, int line, EvtCursorDir d, Point p)
{    
    int charNo;
    Point basePoint;
    
    if (cursorPos == gPoint_1) 
	CharToPos (at,&line,&cursorPos);

    if (d == eCrsDown)
	line= min(nLines-1, line+1);
    else
	line= max(0, line-1);        
    basePoint= LineToPoint(line, TRUE) + Point(cursorPos.x, 0);
    PointToPos(basePoint, &p, &line, &charNo);
    return charNo;
}

void CodeTextView::FormatCode()
{
    if (!text->IsKindOf(StyledText)) 
	return;
    bool inDefine= FALSE;
    int lastComment= 0, prevCh= 0, c= '\n'; 
    int braceStack= 0, inString= 0, escape= FALSE, inClass= FALSE;
    int canBeClass= -1, canBeFunction= -1;

    StyledText *stext= (StyledText*)text;
    TextRunArray *st= new TextRunArray(stext);
    st->Insert(plainStyle, 0, 0, stext->Size());
       
    AutoTextIter next(stext, 0, stext->Size());
	while ((c= next()) != cEOT) {
	if (escape)
	    escape=FALSE;
	else
	    switch (c) {
	    case '#':
		if (prevCh == '\n')
		    inDefine= TRUE;
		break;
	    case '\\':
		escape= TRUE;
		break;
	    case '\n':
		inDefine= FALSE;
		break;
	    case '*':
		if (prevCh == '/' && inString == 0) {
		    if (canBeFunction == -1 && canBeClass == -1)
			lastComment= next.GetPos(); 
		    Comment(&next, st);
		    prevCh= 0;
		    continue;
		}
		break;
	    case '/':
		if (prevCh == '/' && inString == 0) {
		    if (canBeFunction == -1 && canBeClass == -1)
			lastComment= next.GetPos(); 
		    LineEndComment(&next, st);
		    prevCh= 0;
		    continue;
		}
		break;
	    case ';':
		canBeFunction= canBeClass= -1;
		break;
	    case '{':
		if (canBeFunction != -1) {
		    FunctionOrMethod(canBeFunction, lastComment, st);
		    canBeFunction= -1;
		}
		if (canBeClass != -1) {
		    ClassDecl(canBeClass,st);
		    canBeClass= -1;
		}
		if (inString == 0)
		    braceStack++;
		break;
	    case '}':
		if (inString == 0)
		    braceStack--;
		break;
	    case '(':
		if (!inDefine && inString == 0) {
		    if (braceStack == 0 && canBeFunction == -1)
			canBeFunction= next.GetPos()-1;
		    braceStack++;
		}
		break;
	    case ')':
		if (!inDefine && inString == 0)
		    braceStack--;
		break;
	    case '\'':
	    case '\"':
		if (inString == 0) {
		    if (c == '\"')
			inString= '\"';
		    else
			inString= '\'';
		} else {
		    if ((inString == '\"' && c == '\"') || (inString == '\'' 
							 && c == '\''))
			inString= 0;
		}
		break;
	    default:
		if (c == 'c' && !inDefine && inString == 0 
		    && !inClass && canBeClass == -1) {
		    if (braceStack == 0)
			if (inClass= IsClassDecl(next.GetPos()))
			    next.SetPos(next.GetPos()+4);
		} else if (inClass && Isinword(c)) {
		    inClass= FALSE;
		    canBeClass= next.GetPos();
		}
		break;
	}
	prevCh= c;
    }
    delete stext->SetStyles(st);
}

void CodeTextView::SetDefaultStyle()
{
    if (!text->IsKindOf(StyledText)) 
	return;
    StyledText *stext= (StyledText*)text;
    TextRunArray *st= new TextRunArray(stext);
    st->Insert(plainStyle, 0, 0, stext->Size());
    delete stext->SetStyles(st);
}

void CodeTextView::Comment(AutoTextIter *next, TextRunArray *st)
{
    int start= next->GetPos()-2;
    int end, c, prevCh= 0;
    StyledText *stext= (StyledText*)text;

    while ((c= (*next)()) != cEOT) {
	if (c == '/' && prevCh == '*') {
	    end= next->GetPos();
	    st->Insert(commentStyle, start, end, end - start);
	    break;
	}
	prevCh= c;
    }
}

void CodeTextView::LineEndComment(AutoTextIter *next, TextRunArray *st)
{
    int start= next->GetPos()-2;
    int end, c;
    StyledText *stext= (StyledText*)text;

    while ((c= (*next)()) != cEOT) 
	if (c == '\n') {
	    end= next->GetPos()-1;
	    st->Insert(commentStyle, start, end, end - start);
	    break;
	}
}

void CodeTextView::FunctionOrMethod(int at, int lastComment, TextRunArray *st)
{
    StyledText *stext= (StyledText*)text;
    byte c;
    int pos= at, len= 0;
    while (--pos >= lastComment) {
	c= (*text)[pos];
	if (!Isspace(c) && !index("[]*+-=%&><|:^%()~/",c)) // overloaded operators
	    break;
    }
    while (pos >= lastComment) {
	c= (*text)[pos];
	if (!(Isinword(c) || c == ':' || c == '~' ))
	    break;
	if (pos-1 < 0)
	    break;
	pos--;
	len++;
    }
    
    if (len) 
	st->Insert(functionStyle, pos, pos+len+1, len+1);
}

static char *match= "lass"; 

bool CodeTextView::IsClassDecl(int at)
{
    char *p= match;
    int c;
    AutoTextIter next(text, at, text->Size());
    while ((c= next()) != cEOT && *p) {
	if (c != *p++) 
	    return FALSE;
    }
    return TRUE;
}

void CodeTextView::ClassDecl(int at, TextRunArray *st)
{
    int start= at-1;
    AutoTextIter next(text, start, text->Size());
    int end, c;
    StyledText *stext= (StyledText*)text;

    while ((c= next()) != cEOT) 
	if (!Isinword(c)) 
	    break;
    end= next.GetPos()-1;
    if (start != end)
	st->Insert(classDeclStyle, start, end, end-start);
}
