//$Clipper$
#include "Clipper.h"
#include "View.h"
#include "Error.h"
#include "Token.h"
#include "BlankWin.h"
#include "ObjList.h"
#include "String.h"

extern bool gInUpdate;

MetaImpl(Clipper, (I_P(minExtent), I_P(relOrigin), I_O(vop), I_P(offset))); 

Clipper::Clipper(VObject *vp, Point minsize, int id) : (minsize, id)
{
    vop= vp;
    offset= relOrigin= gPoint0;
    SetMinExtent(minsize);
    ResetFlag(eVObjOpen);
}

Clipper::~Clipper()
{
}

void Clipper::Open(bool mode)
{
    if (IsOpen() == mode)
	return;
    VObject::Open(mode);
    if (vop) {
	if (mode) {
	    vop->AddToClipper(this);
	    ViewSizeChanged(gPoint0, vop->GetExtent());
	} else {
	    vop->RemoveFromClipper(this);
	}
	vop->Open(mode);
    }
}

void Clipper::Enable(bool b, bool redraw)
{
    VObject::Enable(b, redraw);
    if (vop)
	vop->Enable(b, redraw);
}

Rectangle Clipper::GetViewSize()
{
    if (vop)
	return vop->GetExtent();
    return GetExtent();
}

Rectangle Clipper::GetViewedRect()
{
    return Rectangle(relOrigin, contentRect.extent);
}

VObject *Clipper::GetViewedVObject()
{
    return vop;
}

Metric Clipper::GetMinSize()
{
    Point me(minExtent);
    if (vop && (minExtent.x == -1 || minExtent.y == -1)) {
	Point e= vop->GetMinSize().extent;
	if (minExtent.x == -1)
	    me.x= e.x;
	if (minExtent.y == -1)
	    me.y= e.y;
    }
    return me;
}

void Clipper::SetMinExtent(Point e)
{
    minExtent= e;
    if (e.x == -1)
	SetFlag(eVObjHFixed);
    if (e.y == -1)
	SetFlag(eVObjVFixed);
}

void Clipper::ShowsView(VObject *vp)
{
    RemoveView(vop);
    vop= vp;
    if (vop) {
	vop->AddToClipper(this);
	ViewSizeChanged(gPoint0, vop->GetExtent());
    }
}

void Clipper::RemoveView(VObject *vp)
{
    if (vop && vop == vp) {
	vop->RemoveFromClipper(this);
	vop= 0;
    }
}

void Clipper::ViewSizeChanged(Point oldsize, Point newsize)
{
    if (vop && newsize != oldsize) {
	Rectangle r[4], oldViewSize= oldsize, viewSize= newsize;
	int nr= Difference(r, oldViewSize, viewSize);
	for (int i= 0; i < nr; i++)
	    InvalidateViewRect(r[i]);
	Control(GetId(), cPartViewSize, &newsize);
    }
}

void Clipper::SetOrigin(Point at)
{
    VObject::SetOrigin(at);
    offset= contentRect.origin - relOrigin;
}

void Clipper::SetExtent(Point e)
{
    Rectangle viewSize= GetViewSize();
    
    VObject::SetExtent(e);
    if (viewSize.IsNotEmpty()) {
	Point oldrelOrigin= relOrigin;
	
	relOrigin= Min(relOrigin, viewSize.extent - e);
	if (viewSize.extent.x < e.x)
	    relOrigin.x= 0;
	if (viewSize.extent.y < e.y)
	    relOrigin.y= 0;
	if (relOrigin != oldrelOrigin)
	    Control(GetId(), cPartScrollPos, &relOrigin);
    }
}

void Clipper::InvalidateRect(Rectangle r)       // r is in Clipper coordinates ??
{
    if (IsOpen() && GetContainer()) {
	r+= offset;
	if (r.Clip(contentRect))
	    GetContainer()->InvalidateRect(r);
    }
}

void Clipper::InvalidateViewRect(Rectangle r)   // r is in View coordinates ??
{
    if (IsOpen() && GetContainer()) {
	r+= offset;
	if (r.Clip(contentRect))
	    GetContainer()->InvalidateRect(r);
    }
}

void Clipper::SetFocus(Rectangle r, Point o)
{
    GetContainer()->SetFocus(Inter(contentRect, r+offset), o+offset);
}

void Clipper::DrawAll(Rectangle r)
{
    if (IsOpen() && r.Clip(contentRect)) {
	Rectangle oldcliprect= port->GetCliprect();
	Point oldoffset= port->GetOrigin();
	GrSetClip(r+oldoffset, offset+oldoffset);
	
	r.origin-= offset;
	DrawBackground(r);
	Draw(r);
	DrawForeground(r);
	
	GrSetClip(oldcliprect, oldoffset);
    }
}

void Clipper::DrawBackground(Rectangle r)
{
    GrEraseRect(r);
}

void Clipper::Draw(Rectangle r)
{
    if (vop)
	vop->DrawAll(r);
}

Command *Clipper::DispatchEvents(Point lp, Token t, Clipper *vf)
{
    Command *cmd;
    if (vop)
	cmd= vop->Input(lp-offset, t, this);
    else
	cmd= VObject::DispatchEvents(lp-offset, t, vf);
    PerformCommand(cmd);
    return gNoChanges;
}

Point Clipper::ContainerPoint(Point p)
{
    return p+offset;
}

//---- Scrolling ----------------------------------------------------------------

Point Clipper::AutoScroll(Point p)
{
    Point scroll, newp;
    
    scroll= newp= Rectangle(contentRect.extent).Constrain(p);
    scroll-= p;
    if (scroll != gPoint0) 
	Control(GetId(), cPartScrollRel, &scroll);
    return newp;
}

void Clipper::RevealRect(Rectangle r, Point minToSee)
{
    Rectangle vr= GetViewedRect();
    Point scroll= 0, rUL= r.NW(), rLR= r.SE(), vrUL= vr.NW(), vrLR= vr.SE();
    int v;
    
    for (v= 0; v <= 1; v++) {
	if (rUL[v] < vrUL[v] && rLR[v] > vrLR[v])
	    continue;
	if (rLR[v] - vrUL[v] < minToSee[v])
	    scroll[v]= vrUL[v] - rUL[v];
	if (vrLR[v] - rUL[v] < minToSee[v])
	    scroll[v]= vrLR[v] - rLR[v];
    }
    if (scroll != gPoint0)
	Control(GetId(), cPartScrollRel, &scroll);
}

void Clipper::RevealAlign(Rectangle r, VObjAlign al)
{
    Rectangle vr= GetViewedRect();
    Point scroll;

    switch (al & eVObjH) {
    case eVObjHCenter:
	scroll.x= (vr.Center().x - r.Center().x)/2;
	break;
    case eVObjHRight:
	scroll.x= vr.NE().x - r.NE().x;
	break;
    default:
	scroll.x= vr.NW().x - r.NW().x;
	break;
    }
    switch (al & eVObjV) {
    case eVObjVCenter:
	scroll.y= (vr.Center().y - r.Center().y)/2;
	break;
    case eVObjVBottom:
	scroll.y= vr.SW().y - r.SW().y;
	break;
    default:
	scroll.y= vr.NW().y - r.NW().y;
	break;
    }
    if (scroll != gPoint0)
	Control(GetId(), cPartScrollRel, &scroll);
}

void Clipper::Control(int id, int part, void *val)
{
    if (part == cPartScrollRel)
	Scroll(part, *(Point*)val);
    else
	VObject::Control(id, part, val);
}

void Clipper::DownControl(int id, int part, void *val)
{
    switch (part) {
    case cPartScrollPage:
    case cPartScrollAbs:
    case cPartScrollHAbs:
    case cPartScrollVAbs:
    case cPartScrollRel:
    case cPartScrollStep:
	Scroll(part, *((Point*)val));
	break;
    case cPartIncr:
	Scroll(cPartScrollStep, Point(0, -1));
	break;
    case cPartDecr:
	Scroll(cPartScrollStep, Point(0, 1));
	break;
    default:
	VObject::DownControl(id, part, val);
	break;
    }
}

void Clipper::Scroll(int mode, Point scroll, bool redraw)
{
    Point leftCorner, newOrigin, delta;
    Rectangle r;
    
    switch(mode) {
    case cPartScrollStep:
	delta= 32 * -scroll;
	break;
    case cPartScrollPage:
	delta= contentRect.extent * -scroll;
	break;
    case cPartScrollAbs:
	delta= relOrigin - scroll;
	break;
    case cPartScrollHAbs:
	delta.x= relOrigin.x - scroll.x;
	break;
    case cPartScrollVAbs:
	delta.y= relOrigin.y - scroll.y;
	break;
    case cPartScrollRel:
	delta= scroll;
	break;
    }
    
    leftCorner= Max(gPoint0, GetViewSize().extent - contentRect.extent);
    newOrigin= relOrigin-delta;
    
    if (vop && vop->IsKindOf(View) && vop->Overridden(View, ConstrainScroll))
	((View*)vop)->ConstrainScroll(&newOrigin);
    
    delta= relOrigin - Min(Max(gPoint0, newOrigin), leftCorner);
    
    if (delta == gPoint0)
	return;
	
    if (! IsOpen())
	redraw= FALSE;
	
    if (redraw) {
	UpdateEvent();
	Focus();
	r= GetViewedRect();
    }
    
    relOrigin-= delta;
    offset+= delta;
    
    if (redraw) {
	GetWindow()->ScrollRect(r, delta);
	if (abs(delta.x) > Width()/3 || abs(delta.y) > Height()/3)
	    UpdateEvent(FALSE); // no batch
	else
	    UpdateEvent();
    }
    Control(GetId(), cPartScrollPos, &relOrigin);
    if (redraw)
	UpdateEvent();
}

// returns TRUE on exit loop
static bool TrackOnce(Command** tracker, TrackPhase atp, Point ap, Point pp, Point lp)
{
    Command *newTracker;

    if (*tracker == 0) {
	*tracker= gNoChanges;
	return TRUE;
    }

    newTracker= (*tracker)->TrackMouse(atp, ap, pp, lp);
    if (newTracker != *tracker) {
	if (*tracker && *tracker != gNoChanges)
	    delete *tracker;
	*tracker= newTracker;
	return (newTracker == gNoChanges);
    }
    return FALSE;
}

Clipper *aClipper= 0;

void Clipper::FeedbackOnce(Command *tracker, Point ap, Point pp, bool turniton)
{
    if (vop && !tracker->TestFlag(eCmdNoReplFeedback) && vop->IsKindOf(View)
					&& ((View*)vop)->GetFrameList()) {
	Iter next(((View*)vop)->GetFrameList());
	Clipper *fp;

	gInUpdate= TRUE;
	while (fp= (Clipper*) next()) {
	    if (fp->IsOpen()) {
		fp->Focus();
		aClipper= fp;
		GrSetPattern(ePatBlack);
		GrSetPenPattern(ePatBlack);
		GrSetPenMode(eRopXor);
		GrSetMode(eRopXor);
		GrSetPenSize(1);
		tracker->TrackFeedback(ap, pp, turniton);
		aClipper= 0;
	    }
	}
	gInUpdate= FALSE;
    } else {
	Focus();
	GrSetPattern(ePatBlack);
	GrSetPenPattern(ePatBlack);
	GrSetPenMode(eRopXor);
	GrSetMode(eRopXor);
	GrSetPenSize(1);
	tracker->TrackFeedback(ap, pp, turniton);
    }
}

static Token Overread(BlankWin *bwin, bool autoscroll, bool *haveSeenUp,
					    bool idle, int &cont, bool &isidle)
{
    Token t, tt;
    
    isidle= FALSE;
    for (;;) {
	t= bwin->ReadEvent(isidle ? 0 : 20, TRUE);
	
	switch (t.Code) {
	
	case eEvtNone:
	    if (autoscroll) {
		t.Code= eEvtLocMove;        // simulate a move event
		return t;
	    }
	    if (idle) {
		if (cont == 0) {
		    cont= 1;
		    break;
		}
		if (cont > 10) {
		    isidle= TRUE;
		    return t;               // return an idle event
		}
		cont++;
	    }
	    break;
	    
	case eEvtRightButton:
	case eEvtMiddleButton:
	case eEvtLeftButton:
	    if (t.Flags & eFlgButDown)
		*haveSeenUp= TRUE;
	    return t;                       // return button event immediately
	    
	case eEvtLocMove:
	case eEvtLocMoveBut:
	    return t;
	    
	case eEvtIdle:                      // ignore idle events
	default:
	    break;
	}
    }
}

Command *Clipper::TrackInContent(Point lp, Token token, Command* tracker)
{
    static int level= 0;
    static bool haveSeenUp= FALSE;
    
    bool autoscroll= FALSE, fullscreen, havereset, isidle;
    Point newlp, ap, pp, dp, delta, oldrelorigin;
    Token token1;
    BlankWin *bwin= GetWindow();
    int cont= 0;
      
    dp= token.Pos - lp;
    level++;
    if (level == 1)
	haveSeenUp= FALSE;
    Focus();
    
    if (fullscreen= tracker->TestFlag(eCmdFullScreen))
	bwin->Fullscreen(TRUE);
    else
	bwin->Grab(TRUE);
    havereset= FALSE;        
    ap= pp= lp;
	
    if (tracker->Overridden(Command, TrackConstrain)) {
	tracker->TrackConstrain(ap, pp, &lp);
	ap= pp= lp;
    }

    if (TrackOnce(&tracker, eTrackPress, ap, pp, lp))
	goto out;
    FeedbackOnce(tracker, ap, pp, TRUE);

restart:
    while (IsOpen()) {
	if (level > 0 && haveSeenUp && !tracker->TestFlag(eCmdMoveEvents)) {
	    break;
	}
	    
	gToken= token= Overread(bwin, autoscroll, &haveSeenUp,
		    tracker->TestFlag(eCmdIdleEvents), cont, isidle);
		
	lp= token.Pos - dp;
	
	if (haveSeenUp)
	    break;
	      
	FeedbackOnce(tracker, ap, pp, FALSE);   // clear previous feedback
	if (tracker->Overridden(Command, TrackConstrain))
	    tracker->TrackConstrain(ap, pp, &lp);
	oldrelorigin= relOrigin;
	if (! tracker->TestFlag(eCmdFullScreen)) {
	    newlp= AutoScroll(lp-relOrigin);
	    delta= relOrigin - oldrelorigin;
	    if (delta != gPoint0) {
		autoscroll= TRUE;
		lp= newlp + relOrigin;
		dp= dp - delta;
		Focus();
	    } else
		autoscroll= FALSE;
	}
	if (TrackOnce(&tracker, isidle ? eTrackIdle : eTrackMove, ap, pp, lp))
	    goto out;
	pp= lp;
	FeedbackOnce(tracker, ap, pp, TRUE);    // new feedback
    }
    FeedbackOnce(tracker, ap, pp, FALSE);   // clear last feedback
    if (tracker->Overridden(Command, TrackConstrain))
	tracker->TrackConstrain(ap, pp, &lp);
	
    if (! tracker->TestFlag(eCmdMoveEvents)) {
	if (fullscreen)
	    bwin->Fullscreen(FALSE);
	else
	    bwin->Grab(FALSE);
	havereset= TRUE;
	level--;
    }
		
    TrackOnce(&tracker, IsOpen() ? eTrackRelease : eTrackExit, ap, pp, lp);

    if (tracker->TestFlag(eCmdMoveEvents)) {
	haveSeenUp= FALSE;
	goto restart;
    }
out:
    Focus();
    if (! havereset) {
	if (fullscreen)
	    bwin->Fullscreen(FALSE);
	else
	    bwin->Grab(FALSE);
	level--;
    }
    return tracker;
}

ostream& Clipper::PrintOn(ostream &s)
{
    VObject::PrintOn(s);
    return s << minExtent SP << vop SP;
}

istream& Clipper::ReadFrom(istream &s)
{
    VObject::ReadFrom(s);
    s >> minExtent >> vop;
    ShowsView(vop);
    SetContentRect(contentRect, TRUE);
    return s;
}

void Clipper::InspectorId(char *buf, int sz)
{
    if (vop)
	vop->InspectorId(buf, sz);  
    else
	VObject::InspectorId(buf, sz);      
}
