/* Interface.h - abstract class implementation to point-repulsion simulation
 *
 * $Id: Interface.c,v 1.3 96/02/11 21:27:14 leech Exp $
 *
 * Copyright (C) 1996, Jonathan P. Leech
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is". Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Log:	Interface.c,v $
 * Revision 1.3  96/02/11  21:27:14  leech
 * New -vrml option, better closest-pair stopping criterion, change
 * some option names.
 * 
 * Revision 1.2  92/10/27  22:49:53  leech
 * *** empty log message ***
 *
 * Revision 1.1  92/04/27  04:25:08  leech
 * Initial revision
 *
 */

#include "Interface.h"
#include "SphereState.h"
#include "vrml.h"

#include <stream.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

// Lookup table for format character codes
int Interface::FmtMaskTable[UCHAR_MAX+1];

// Does the metric with specified index need to be computed, either
//  for tracing or as a stop condition?
boolean Interface::NeedFmt(int index) const {
    int tracemask = trace & (1 << index);
    return tracemask != 0 || stopType == index;
}

// Return mask of all trace codes in fmt
int Interface::FmtMask(const char *fmt) {
    int mask;
    for (mask = 0; *fmt != '\0'; fmt++) {
	int m = FmtType(*fmt);
	if (m == 0) {
	    cerr << "\nUnrecognized format character '"
		 << *fmt << "' in trace format '"
		 << traceFmt << "'" << endl;
	    exit(1);
	} else
	    mask |= m;
    }

    return mask;
}

Interface::Interface(int interactive, int &ac, char **av)
  : model(NULL), debug(false), log(false), trace(0), running(true),
    traceFmt(NULL), maxStep(10000000L), loadFile(NULL), saveFile(NULL),
    vrmlFile(NULL), stopType(kUnset), stopVal(1.0)
{
    InitFmt();

    // Default for interactive interfaces is not to terminate except
    //	by user input.
    if (interactive)
	stopType = kNeverStop;

    model = new SphereState(4);

    boolean setDt = false;  // If dt is set by user, don't change it
    int guiargs = 1;	    // # of arguments left over for GUI

    for (int i = 1; i < ac; i++) {
	// Options affecting model parameters
	if (!strcmp(av[i], "-nbody"))
	    model->SetNumBodies(atoi(av[++i]));
	else if (!strcmp(av[i], "-critical"))
	    model->SetCriticalDamping(1);
	else if (!strcmp(av[i], "-damping"))
	    model->SetDamping(atof(av[++i]));
	else if (!strcmp(av[i], "-dt")) {
	    setDt = true;
	    model->SetTimestep(atof(av[++i]));
	} else if (!strcmp(av[i], "-softening"))
	    model->SetSoftening(atof(av[++i]));
	else if (!strcmp(av[i], "-k"))
	    model->SetKForce(atof(av[++i]));

	// Options affecting how to run and terminate simulation
	else if (!strcmp(av[i], "-kestop"))
	    stopType = kKineticEnergy, stopVal = atof(av[++i]);
	else if (!strcmp(av[i], "-pestop"))
	    stopType = kPotentialEnergy, stopVal = atof(av[++i]);
	else if (!strcmp(av[i], "-vstop"))
	    stopType = kMaxVelocity, stopVal = atof(av[++i]);
	else if (!strcmp(av[i], "-rstop"))
	    stopType = kDistanceRatio, stopVal = atof(av[++i]);
	else if (!strcmp(av[i], "-neverstop"))
	    stopType = kNeverStop;
	else if (!strcmp(av[i], "-steps"))
	    maxStep = atol(av[++i]);

	else if (!strcmp(av[i], "-d"))
	    debug = true;
	else if (!strcmp(av[i], "-trace")) {
	    if (av[i+1] && av[i+1][0] != '-')
		traceFmt = av[++i];
	    else
		traceFmt = "KR";
	    trace = FmtMask(traceFmt);
	} else if (!strcmp(av[i], "-load"))
	    loadFile = av[++i];
	else if (!strcmp(av[i], "-save"))
	    saveFile = av[++i];
	else if (!strcmp(av[i], "-vrml"))
	    vrmlFile = av[++i];

	else if (!strcmp(av[i], "-stop"))
	    RunStop(false);

	else if (!strcmp(av[i], "-help"))
	    Help();
	else {
	    cerr << "Unrecognized option " << av[i] << endl;
	    Help();
	}
    }

    // Update argument count for GUI
    ac = guiargs;

    if (loadFile)
	model->ReadBodies(loadFile);

    if (!setDt)
	model->SetTimestep(model->DefaultTimestep());

    // If user didn't define a stop type, pick a reasonable one
    // How about when kinetic energy dies down to a reasonable level?
    if (stopType == kUnset) {
	stopType = kKineticEnergy;
	stopVal = 1e-5 * model->GetNumBodies();
    }

    // Header for logging KE levels
    if (debug)
	model->SetDebug(1);
}

Interface::~Interface() { }

SphereState *Interface::GetModel() const {
    return model;
}

void Interface::Run() {
    int done = 0;

    step = 0;
    while (step < maxStep && !done && !Done()) {
	if (!HandleEvent() && Running()) {
	    step++;

	    model->Timestep();

	    UpdatePoints();

	    // Values for termination condition
	    double
		condition,
		ke, pe, vmax, minpair, ratio;

	    if (NeedFmt(kKineticEnergy)) {
		ke = model->KineticEnergy();
		if (stopType == kKineticEnergy)
		    condition = ke;
	    }
	    if (NeedFmt(kPotentialEnergy)) {
		pe = model->PotentialEnergy();
		if (stopType == kPotentialEnergy)
		    condition = pe;
	    }
	    if (NeedFmt(kMaxVelocity)) {
		vmax = model->MaxVelocity();
		if (stopType == kMaxVelocity)
		    condition = vmax;
	    }
	    if (NeedFmt(kClosestPair)) {
		minpair = model->ClosestPair();
		if (stopType == kClosestPair)
		    condition = minpair;
	    }
	    if (NeedFmt(kDistanceRatio)) {
		double
		    min = model->ClosestPair(),
		    ideal = model->IdealPair();
		if (min > 0)
		    ratio = ideal / min;
		else
		    ratio = 1e10;
		if (stopType == kDistanceRatio)
		    condition = ratio;
	    }

	    if (trace != 0) {
		cout << step << '\t';

		// Parse the format string on the fly
		for (const char *fmt = traceFmt; *fmt != '\0'; fmt++) {
		    switch (FmtType(*fmt)) {
			case mKineticEnergy:	cout << ke; break;
			case mPotentialEnergy:	cout << pe; break;
			case mMaxVelocity:	cout << vmax; break;
			case mClosestPair:	cout << minpair; break;
			case mDistanceRatio:	cout << ratio; break;
		    }
		    if (fmt[1] != '\0')
			cout << '\t';
		}
		cout << endl;
	    }

	    // Stopping condition
	    if (stopType != kNeverStop && (condition < stopVal))
		done = 1;
	}
    }
}

void Interface::Cleanup(long &iterations, TerminationCause &why) {
    iterations = step;
    if (stopType == kNeverStop)
	why = kQuit;
    else if (step >= maxStep)
	why = kStopped;
    else
	why = kConverged;

    if (saveFile)
	model->WriteBodies(saveFile);

    if (vrmlFile)
	write_vrml(vrmlFile, model);
}

char *Interface::HelpText[] = {
    "    Model Parameters\n",
    "\t-nbody #\t\tInitial # of randomly placed points\n",
    "\t-critical\tUse critical damping\n",
    "\t-damping #\tDamping constant (1)\n",
    "\t-dt #\t\tTimestep (.01)\n",
    "\t-softening #\t\tSoftening parameter (.05)\n",
    "\t-k #\t\tForce coefficient (1)\n",
    "    Stopping Criteria\n",
    "\t-kestop #\tWhen system kinetic energy < #\n",
    "\t-pestop #\tWhen system potential energy < #\n",
    "\t-vstop #\tWhen maximum particle velocity < #\n",
    "\t-rstop #\tWhen max/min nearest neighbor ratio < # (1.6)\n",
    "\t-neverstop\tDon't apply any stopping criteria\n",
    "\t-steps #\tAfter maximum of # steps even if criteria not satisfied\n",
    "    Simulation Options\n",
    "\t-load file\tRead initial particle state from file\n",
    "\t-save file\tWrite final particle state to file (stdout)\n",
    "\t-help\t\tDisplay this help\n",
    "    Other options are passed to the GUI (if any)\n"
    // "\t-d\t\tEnable debugging\n",
    // "\t-trace [fmt]\tEnable tracing\n",
};

// Build a lookup table for translating format string into numeric codes.
void Interface::InitFmt() {
    static boolean init = 0;

    if (init)
	return;

    init = true;

    for (int i = 0; i <= UCHAR_MAX; i++)
	FmtMaskTable[i] = 0;
    FmtMaskTable['k'] = FmtMaskTable['K'] = mKineticEnergy;
    FmtMaskTable['p'] = FmtMaskTable['P'] = mPotentialEnergy;
    FmtMaskTable['v'] = FmtMaskTable['V'] = mMaxVelocity;
    FmtMaskTable['c'] = FmtMaskTable['C'] = mClosestPair;
    FmtMaskTable['r'] = FmtMaskTable['R'] = mDistanceRatio;
}

void Interface::Help(const char *program) {
    cerr << "Usage: " << program << " [options]\n";
    cerr << "Options are:\n";
    for (int i = 0; i < sizeof(HelpText) / sizeof(HelpText[0]); i++)
	cerr << HelpText[i];
    cerr.flush();
    exit(1);
}
