/* SphereState.c - implementation of point repulsion simulation class
 *
 * $Id: SphereState.c,v 1.3 96/02/11 21:29:09 leech Exp $
 *
 * Copyright (C) 1996, Jonathan P. Leech
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is". Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Log:	SphereState.c,v $
 * Revision 1.3  96/02/11  21:29:09  leech
 * Fix potential energy computation and nearest-neighbor computation.
 * Many minor tweaks and reorganization of code.
 * 
 * Revision 1.2  92/10/27  22:50:06  leech
 * *** empty log message ***
 *
 * Revision 1.1  92/04/27  04:25:10  leech
 * Initial revision
 *
 */

#include <stream.h>
#include "SphereState.h"
#include <stdlib.h>

// Place a point at rest on the unit sphere
void MassPoint::randomize(double m) {
    mass = m;

    // Place a point randomly on the unit sphere
    r = unit_random();

    // Compute a velocity for it (initially zero)
    v = dvector::zero;
}

SphereState::SphereState(unsigned n)
  : current(NULL), next(NULL), maxdotp(NULL), force(NULL),
    debug(0),
    forceLaw(1, 0.05, 1)	    // k, epsilon, r0
{
    InvalidateCache();

    // Simulation parameters; these are reasonable for small n
    dt	    = 0.1;		// default timestep
    lambda  = 1;		// default damping
    SetCutoffRadius(M_PI);	// compute cutoff parameters
    critical_damping = true;	// enable critical damping

    nbody   = n;		// # of bodies
    interactions = 0;		// pairwise interactions

    AllocateBuffers(nbody);

    ComputeIdealPair();

    // Randomize initial positions
    for (int i = 0; i < nbody; i++)
	current[i].randomize();
}

SphereState::~SphereState() {
    FreeBuffers();
}

// Return position of i'th body at current timestep.
dvector SphereState::GetPosition(unsigned i) const {
    if (i >= nbody)
	return dvector::zero;
    else
	return current[i].r;
}

// Set position of i'th body; force onto the unit sphere
void SphereState::SetPosition(unsigned i, const dvector &r) {
    if (i >= nbody)
	return;

    current[i].r = r;
    current[i].r.normalize();
    InvalidateCache();
}

// Return velocity of i'th body at current timestep.
dvector SphereState::GetVelocity(unsigned i) const {
    if (i >= nbody)
	return dvector::zero;
    else
	return current[i].v;
}

// Set velocity of i'th body
void SphereState::SetVelocity(unsigned i, const dvector &v) {
    if (i >= nbody)
	return;

    current[i].v = v;
    InvalidateCache();
}

double SphereState::GetSoftening() const {
    return forceLaw.GetSoftening();
}

void SphereState::SetSoftening(double e) {
    forceLaw.SetSoftening(e);
}

double SphereState::GetKForce() const {
    return forceLaw.GetKForce();
}

void SphereState::SetKForce(double f) {
    forceLaw.SetKForce(f);
}

double SphereState::GetR0() const {
    return forceLaw.GetR0();
}

void SphereState::SetR0(double r0) {
    forceLaw.SetR0(r0);
}

double SphereState::GetDamping() const {
    return lambda;
}

void SphereState::SetDamping(double l) {
    lambda = l;
}

double SphereState::GetTimestep() const {
    return dt;
}

void SphereState::SetTimestep(double t) {
    dt = t;
}

void SphereState::SetCriticalDamping(int flag) {
    critical_damping = (flag != 0);
}

int SphereState::GetCriticalDamping() const {
    return critical_damping;
}

// Change number of bodies
void SphereState::SetNumBodies(int newN) {
    if (newN == nbody || newN < 1)
	return;

    // Tag computed values as bad
    InvalidateCache();

    // Save current body state
    MassPoint *saved = current;
    current = NULL;

    // Reallocate storage space
    AllocateBuffers(newN);

    if (newN > nbody) {
	// If bodies added, copy all the old and randomize the new

	for (int i = 0; i < nbody; i++)
	    current[i] = saved[i];
	for (i = nbody; i < newN; i++)
	    current[i].randomize();
    } else {
	// If bodies removed, copy some of the old

	for (int i = 0; i < newN; i++)
	    current[i] = saved[i];
    }

    // Change body count
    nbody = newN;

    // Find new ideal particle distances
    ComputeIdealPair();

    // Get rid of saved state
    delete [] saved;
}

int SphereState::GetNumBodies() const {
    return nbody;
}

double SphereState::GetCutoffRadius() const {
    return cutoffRadius;
}

// Change the cutoff radius and trivial-reject cos(theta) value
void SphereState::SetCutoffRadius(double c) {
    if (c < 0)
	c = 0.0;
    else if (c > M_PI)
	c = M_PI;

    cutoffRadius = c;
    cutoffDotprod = cos(c);
}

// Timestep the simulation
boolean SphereState::Timestep()
{
    InvalidateCache();

    // Update velocities
    StepVelocity();

    // Update positions and rotate velocities to be tangent to positions
    boolean ok = StepPosition();

    // Copy new to old bodies (could just swap pointers)
    for (int i = 0; i < nbody; i++)
	current[i] = next[i];

    return ok;
}

ForceLaw *SphereState::GetForceLaw() {
    return &forceLaw;
}

// Return total KE of system [= sum(v^2)]
double SphereState::KineticEnergy() {
    if (!keComputed) {
	keComputed = true;

	ke = 0;
	for (int i = 0; i < nbody; i++)
	    ke += current[i].v * current[i].v;
    }

    return ke;
}

// Return total "potential energy" of system [= sum(i!=j) 1/Rij]
// This is not scaled to the same units as kinetic energy, and
//  it does not take into account the different force laws,
//  so it's hopeless to compute a Virial quantity.
double SphereState::PotentialEnergy() {
    if (!peComputed) {
	peComputed = true;

	pe = 0;
	for (int i = 0; i < nbody; i++)
	    for (int j = i+1; j < nbody; j++) {
		double len = forceLaw.Distance(current[i].r, current[j].r);
		if (len != 0)
		    pe += 1.0 / len;
	    }
    }

    return pe;
}

// Return maximum velocity of any body
double SphereState::MaxVelocity() {
    if (!vmaxComputed) {
	vmaxComputed = true;

	double vmaxsqr = 0.0;
	int i;

	for (i = 0; i < nbody; i++) {
	    double vsqr = current[i].v * current[i].v;
	    if (vsqr >= vmaxsqr)
		vmaxsqr = vsqr;
	}

	vmax = sqrt(vmaxsqr);
    }

    return vmax;
}

// Return smallest near-neighbor distance between points.
double SphereState::ClosestPair()
{
    ComputePairDistances();
    return min_pairdist;
}

// Return largest near-neighbor distance between points.
double SphereState::FurthestPair()
{
    ComputePairDistances();
    return max_pairdist;
}

// Return ideal near-neighbor distance between points on a hexagonal grid.
double SphereState::IdealPair()
{
    return ideal_pairdist;
}

// Return # of pairwise interactions in most recent timestep
int SphereState::Interactions()
{
    return interactions;
}

void SphereState::SetDebug(int d) {
    debug = d;
}

int SphereState::GetDebug() const {
    return debug;
}

// Load bodies from specified file
// Format: n x1 y1 z1 .. xn yn zn (whitespace ignored)
boolean SphereState::ReadBodies(const char *file) {
    FILE *fp = fopen(file, "r");
    if (fp == NULL) {
	fprintf(stderr, "SphereState::ReadBodies: can't open file %s\n", file);
	return false;
    }

    InvalidateCache();

    int n;
    fscanf(fp, "%d", &n);
    SetNumBodies(n);

    dvector r;
    for (int i = 0; i < n; i++) {
	if (fscanf(fp, "%lf %lf %lf", &r[0], &r[1], &r[2]) != 3) {
	    fprintf(stderr, "SphereState::ReadBodies: bad point %d\n", i);
	    return false;
	}
	SetPosition(i, r);
	SetVelocity(i, dvector::zero);
    }
    fclose(fp);
    return true;
}

boolean SphereState::WriteBodies(const char *file) const {
    FILE *fp = fopen(file, "w");
    if (fp == NULL) {
	fprintf(stderr, "SphereState::WriteBodies: can't open file %s\n", file);
	return false;
    }

    fprintf(fp, "%d\n", nbody);

    for (int i = 0; i < nbody; i++) {
	dvector r(GetPosition(i));
	fprintf(fp, "%lf %lf %lf\n", r(0), r(1), r(2));
    }
    fclose(fp);
    return true;
}

// Compute a reasonable timestep, assuming critical damping is enabled.
double SphereState::DefaultTimestep() const {
    const double
	a = 0.09,	// Suitable coefficients for
	b = -0.004;	//  dt = a exp(b N)

    return a * exp(b * nbody);
}

//////////////////////////////////////////////////////////////////////////////
// Protected methods
//////////////////////////////////////////////////////////////////////////////

// Update the velocities of bodies current[] by summing all the forces
//  using the current force law,
//  projecting the force into the tangent plane, and damping by a term
//  proportional to current velocity.
// Store new body velocities in bnew[].v
void SphereState::StepVelocity()
{
    // Compute new velocity:
    //	v' = v + dt dv/dt
    // Then rotate v' into the local frame of r'
    //
    // dv    /1		     normal\
    // -- = | - sum(i != j) F	    | - lambda v
    // dt    \m		     ji    /
    //
    // We assume all masses are equal, so the mass term drops out of
    //	the equation

    interactions = 0;

    if (debug)
	printf("accelerate_bodies:\n");

    // Summed forces on each point
    for (int i = 0; i < nbody; i++)
	force[i] = dvector::zero;

    // This loop computes each interaction i<->j only once, relying
    //	on the geometric symmetry of the force law.
    for (i = 0; i < nbody; i++) {
	for (int j = i+1; j < nbody; j++) {
	    dvector f_ij;

	    // Don't compute if ri dot rj <= cos(cutoffRadius)
	    if (current[i].r * current[j].r <= cutoffDotprod)
		continue;

	    f_ij = forceLaw.FindForce(current[i].r, current[j].r);

	    force[i] += f_ij;
	    force[j] -= f_ij;

	    if (debug)
		printf("\tf%d,%d = %g %g %g\n", i, j, f_ij(0), f_ij(1), f_ij(2));
	    interactions++;
	}
    }

    for (i = 0; i < nbody; i++) {
	// Component of force vector in normal plane to r_i
	dvector f_i(normal_component(force[i], current[i].r));

	// Compute new velocity from current velocity, force, and
	//  simulation parameters
	if (critical_damping) {
	    // Critical damping completely annihilates previous
	    //	velocity in one timestep.
	    next[i].v = dt * f_i;
	} else {
	    // Usually subtract term proportional to current velocity
	    // Velocity will be overdamped if (lambda * dt) > 1,
	    //	which is undesirable.
	    next[i].v = dt * f_i + (1 - lambda * dt) * current[i].v;
	}

	if (debug)
	    printf("\tF%d = %g %g %g\tnew V%d = %g %g %g\n",
		i, f_i(0), f_i(1), f_i(2),
		i, next[i].v(0), next[i].v(1), next[i].v(2));
    }
}

// Move bodies current[] along the great circle defined by their
//  current position and updated velocity by an amount
//  proportional to velocity and timestep dt. Put updated
//  positions into next[]. After moving the body, rotate its
//  updated velocity to be tangent to the new position.
// Returns true unless there's a numeric problem; in that
//  case returns false and prints a description of the problem
//  to stderr.
boolean SphereState::StepPosition() {
    boolean ok = true;
    const double
	minVelocity = 1e-10,	// Ignore velocities smaller than this
	velocityError = 1e-4;	// Make sure velocities are tangent within this distance-squared error

    // Maximum allowed particle velocity (covers average distance
    //	between two evenly distributed points in one timestep).
    double maxv = ideal_pairdist / (2 * dt);

    // Compute new position:
    //	theta = dt |v'|
    //	r' = r cos(theta) + (v' / |v'|) sin(theta)
    for (int i = 0; i < nbody; i++) {
	double vmag = next[i].v.magnitude();  // Magnitude of velocity

	if (vmag > minVelocity) {
	    // Unit vector tangent to sphere at r in direction of
	    //	velocity.
	    dvector vnorm(next[i].v / vmag);
	    double dotp = vnorm * current[i].r;

	    // Sanity check; clamp velocity to 0 if this happens
	    if (dotp > velocityError) {
		fprintf(stderr, "StepPosition: velocity %d not tangent to surface\n", i);
		fprintf(stderr, "\tv/|v| * r = %g, |v| = %g, v = %g %g %g, r = %g %g %g\n",
		    dotp, vmag,
		    next[i].v(0), next[i].v(1), next[i].v(2),
		    current[i].r(0), current[i].r(1), current[i].r(2));

		next[i].v = dvector::zero;
		next[i].r = current[i].r;

		// flag that an error occurred
		ok = false;
	    }

	    // Clamp velocity
	    if (vmag > maxv)
		vmag = maxv;

	    double
		theta = dt * vmag,   // Angle through which body moves
		ct = cos(theta), st = sin(theta);

	    // (r,v/|v|) defines a coordinate frame for the great
	    //	circle in which the body is moving. theta is the angle
	    //	through which the body moves (assuming constant velocity
	    //	during the time interval, e.g. Euler approximation).
	    next[i].r = ct * current[i].r + st * vnorm;
	    next[i].v = vmag * (ct * vnorm - st * current[i].r);

	    // Force to surface of sphere to correct for roundoff.
	    next[i].r.normalize();
	} else {
	    // Velocity may be forced to zero if it's too small.
	    if (vmag > 0)
		next[i].v = dvector::zero;

	    // The point didn't move
	    next[i].r = current[i].r;
	}
    }

    return ok;
}

// Mark the computed values for KE, PE, etc. as invalid (usually
//  because a new timestep has been done or a point has been
//  specified externally to the algorithm).
void SphereState::InvalidateCache() {
    keComputed = peComputed = vmaxComputed = pairdistComputed = false;
}

// Compute the new ideal distance between points.
// For reasons I don't completely understand, this appears
//  to be very well fitted to Pi / sqrt(N)
//
// Alternative approach assuming hexagonal packing of N points:
//	hexagon area = 3 sqrt(3) R^2 / 2
//	sphere area = 4 Pi / N
//  equating and solving for r,
//	hexagon size = r = sqrt(8 Pi / (3 sqrt(3) N))
//  ideal distance = 2 * r
void SphereState::ComputeIdealPair() {
    static double coeff = M_PI;

    ideal_pairdist = (nbody > 0) ? (coeff / sqrt(nbody)) : M_PI;
}

// Compute & cache closest and furthest near-neighbor distances.
void SphereState::ComputePairDistances()
{
    if (!pairdistComputed) {
	pairdistComputed = true;

	// Find minimum distance from each point i to all other points j.
	// Instead of computing Cartesian distance, we use the more rapidly
	//  computed dot-product distance, since all points are known to
	//  lie on a sphere. Dot-product distance is greatest when Cartesian
	//  distance is smallest, and decreases with increasing Cartesian
	//  distance.

	for (int i = 0; i < nbody; i++)
	    maxdotp[i] = -1;

	for (i = 0; i < nbody; i++) {
	    for (int j = i+1; j < nbody; j++) {
		// Ri * Rj = cos(distance(i,j))
		double dotp = current[i].r * current[j].r;

		if (dotp > maxdotp[i]) {
		    if (debug) {
			printf("R%d * R%d = %g\t; max_dotp%d = max_dotp%d = %g\n",
			    i, j, i, j, dotp);
		    }
		    maxdotp[i] = dotp;
		    maxdotp[j] = dotp;
		}
	    }
	}

	// Now find the closest and furthest pairs
	double
	    min_maxdotp = maxdotp[0],
	    max_maxdotp = maxdotp[0];

	for (i = 1; i < nbody; i++) {
	    if (debug)
		printf("max_dotp%d = %g\n", i, maxdotp[i]);

	    if (maxdotp[i] < min_maxdotp)
		min_maxdotp = maxdotp[i];
	    if (maxdotp[i] > max_maxdotp)
		max_maxdotp = maxdotp[i];
	}

	// Cache the results
	min_pairdist = acos(max_maxdotp);
	max_pairdist = acos(min_maxdotp);

	if (debug) {
	    printf("min_maxdotp = %g -> max_pairdist = %g\n", min_maxdotp, max_pairdist);
	    printf("max_maxdotp = %g -> min_pairdist = %g\n", max_maxdotp, min_pairdist);
	}
    }
}

// Allocate the history list, a circular buffer
void SphereState::AllocateBuffers(int n)
{
    FreeBuffers();

    current = new MassPoint[n];
    next = new MassPoint[n];
    maxdotp = new double[n];
    force = new dvector[n];
}

// Delete the history list and NULL the pointer
void SphereState::FreeBuffers()
{
    if (current)
	delete [] current;
    if (next)
	delete [] next;
    if (maxdotp)
	delete [] maxdotp;
    if (force)
	delete [] force;
}
