#include "defs.h"
#include "integer.e"
#include "poly.h"
#include "error.e"

t_poly
poly_z_integer_div WITH_3_ARGS(
    t_handle,      pring,
    t_poly,     apoly,
    integer_big,   bint
)
/*
*        IPOLY_INTEGER_DIV: integral polynomial integer div.
*        apoly is an integral polynomial,bint is a non-zero integer which
*        divides apoly . returns apoly/bint.
*/
{
    t_handle         resph;     /* t_handle to result */
    t_handle         aph;       /* t_handle to apoly */
    t_poly     acoefft;
    t_int  termno;    /* loop counter */
    t_int  nterms;    /* no. terms in apoly */

    if ( bint == 0 )
    {
        error_internal ("divide by zero in poly_z_integer_div\n");
        return  0;
    }

    if ( m_poly_const( apoly ) )
    {
        return  integer_div( apoly, bint );
    }

    /* non-trivial polynomial */

    aph = m_poly_poly_to_handle( apoly );
    nterms = m_poly_nterms( aph );
    m_poly_create_empty(&resph, m_poly_princvar( aph ),
                                    m_poly_least_pvar( aph ), nterms );

    for ( termno = 0; termno < nterms; termno++ )
    {
        /* divide each term by bint, exponent remains constant */

        m_poly_expt( resph, termno ) = m_poly_expt( aph, termno );
        acoefft = m_poly_coefft( aph, termno );

        if ( m_poly_not_const( acoefft ) )
            m_poly_coefft (resph, termno) =
                               poly_z_integer_div (pring, acoefft,  bint);
        else
            m_poly_coefft( resph, termno ) = integer_div( acoefft, bint );
    }

    m_poly_nterms( resph ) = nterms;

    return  poly_z_clean( pring, m_poly_handle_to_poly( resph ) ); 

} /* poly_z_integer_div() */



t_poly
poly_z_div WITH_3_ARGS(
    t_handle,    pring,
    t_poly,   apoly,
    t_poly,   bpoly
)
/*
*        IPOLY_DIV : integral polynomial quotient, no remainder
*        apoly, bpoly are integral polynomials, bpoly is a non_zero
*        divisor of apoly. returns apoly/bpoly.
*/
{
    block_declarations;
    t_poly         rempoly;        /*  remainder           */
    t_poly         qpoly;          /*  quotient (result)   */

    if ( ( m_poly_const( apoly ) ) && ( m_poly_const( bpoly ) ) )
    {
        /* both apoly and bpoly constant */

        return  integer_div( apoly, bpoly );
    }

    if ( m_poly_const( bpoly ) )
    {
        /* bpoly constant, apoly not */

        error_internal("polys not lifted in poly_z_div\n");
        return  poly_z_integer_div( pring, apoly, bpoly );
    }

    if ( m_poly_const( apoly ) )
    {
        /* apoly constant, bpoly not */

        error_internal("polys not lifted in poly_z_div\n");
        return  poly_z_zero_poly (pring, apoly);
    }

    /* both apoly and bpoly non-trivial polynomials */

    poly_z_quot_rem( pring, apoly, bpoly, &qpoly, &rempoly );
    m_poly_z_delref( pring, rempoly );

    return  qpoly;

} /* poly_z_div() */

