#include "defs.h"
#include "mp.e"
#include "mp.h"

mp_float
mp_div_int	WITH_3_ARGS(
	mp_float,	x,
	mp_int,		iy,
	mp_float,	z
)
/*
Sets z = x / (int)iy and returns z.  This is much faster than division
by an mp number.  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), zp = mp_ptr(z);
    mp_int		j;
    mp_length		t, guard, total, i;
    mp_base_type	b;
    mp_sign_type	result_sign;
    mp_expt_type	result_expt;
    mp_int		c, limit;
    mp_acc_index	acc_index;
    mp_digit_ptr_type	x_dig_p, z_dig_p;

    if (!iy)
	mp_error("mp_div_int: division by zero");

    /*
    Check for compatible parameters.
    */

    mp_check_2("mp_div_int", xp, zp);

    DEBUG_BEGIN(DEBUG_DIVI);
    DEBUG_PRINTF_1("+div_int {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_2("iy = %d\n\n", iy);

    result_sign = mp_sign(xp) * int_sign(iy);
    DEBUG_PRINTF_2("rs = %d\n", result_sign);

    if (result_sign == 0)
    {
	mp_set_sign(zp, 0);

	DEBUG_PRINTF_1("-} z = 0\n");
	DEBUG_END();

	return z;
    }

    j = int_abs(iy);
    b = mp_b(zp);
    t = mp_t(zp);

    result_expt = mp_expt(xp);


    /*
    Check for division by +-1 or +-b.
    */

    if (j == 1 || j == b)
    {
	/*
	All we need do is decrement the exponent if j is b or copy it if j is 1.
	*/

	if (j == b)
	    result_expt--;

	mp_copy_ptr(xp, zp);
        mp_set_sign(zp, result_sign);
	mp_expt(zp) = result_expt;


	/*
	Check for underflow.
	*/

	if (result_expt <= MIN_EXPT)
	    mp_underflow(z);

	/*
	Update exponent statistics.
	*/

	mp_update(zp);


	DEBUG_1("-} right-shifted z =", zp);
	DEBUG_END();
	return z;
    }


    /*
    We have to genuine division here.  Compute number of guard digits required:
    simple rounding depends on iy, truncating needs 0, and directed rounding
    needs 1.  If necessary, allocate accumulator space and reset float pointers.
    */

    guard = (round == MP_RND)? (1 + mp_guard_digits(iy, b)): round != MP_TRUNC;
    total = t + guard;


    if (round == MP_TRUNC)
	z_dig_p = mp_digit_ptr(zp, 0);
    else
    {
	mp_change_up();

	mp_acc_digit_alloc(total, acc_index);
	z_dig_p = mp_acc_digit_ptr(acc_index);

	if (mp_has_changed())
	{
	    xp = mp_ptr(x);
	    zp = mp_ptr(z);
	}

	mp_change_down();
    }


    x_dig_p = mp_digit_ptr(xp, 0);

#define x_dig(i)	x_dig_p[i]
#define z_dig(i)	z_dig_p[i]
#define z_ptr(i)	(z_dig_p + (i))


    /*
    If j * b is not representable as an integer we must simulate long division.
    */

    if (j < (limit = MAX_INT / b))
    {
	mp_int		q0;
	mp_length	k, rnum;

	/*
	Look for the first non-zero digit in quotient.
	*/

	for (i = c = q0 = 0; !q0; i++)
	{
	    c *= b;

	    if (i < t)
		c += x_dig(i);

	    q0 = c / j;

	    /*
	    Here Brent did a non-portable check to see if overflow had
	    occurred by checking whether q0 went below zero.
	    */
	}


	/*
	Adjust exponent and get total digits in quotient.
	*/

	result_expt -= i - 1;
	z_dig(0) = q0;

	c -= j * q0;		/* i.e. c %= j */
	c *= b;

	if (i < t)
	{
	    rnum = t - i + 1;

	    for (k = 1; k < rnum; k++)
	    {
		c += x_dig(i++);
		z_dig(k) = c / j;
		c -= j * z_dig(k);		/* i.e. c %= j */
		c *= b;
	    }
	}
	else
	    rnum = 1;


	if (rnum > total && (round == MP_RND_UP || round == MP_RND_DOWN))
	    z_dig(total - 1) = 1;
	else
	{
	    for (k = rnum; k < total; k++)
	    {
		z_dig(k) = c / j;
		c -= j * z_dig(k);		/* i.e. c %= j */
		c *= b;
	    }

	    /*
	    Same unportable check: seeing if c went below zero.
	    */
	}

    }
    else
    {
	/*
	Here we need simulated double precision division.
	*/

	mp_int		j1 = j / b, j2 = j % b, c2, ir, iq;
	mp_length	k;


	DEBUG_PRINTF_3("large case: j1 = %d, j2 = %d\n", j1, j2);

	/*
	Look for the first non-zero digit in quotient.
	*/

	i = -1;
	c = c2 = 0;

	do
	{
	    i++;
	    c = b * c + c2;
	    c2 = (i < t)? x_dig(i): 0;
	}
	while (c < j1 || c == j1 && c2 < j2);


	result_expt -= i;

	for (k = 0; k < total; k++, i++)
	{
	    register mp_int	iq_on_j;

	    /*
	    Get an approximate quotient first.
	    */

	    ir = c / (j1 + 1);

	    /*
	    Now reduce so overflow does not occur.
	    */

	    iq = c - ir * j1;

	    if (iq >= limit)
	    {
		/*
		Here iq*b would possibly overflow so increase ir.
		*/

		ir++;
		iq -= j1;
	    }

	    iq *= b;
	    iq -= ir * j2;

	    if (iq < 0)
	    {
		/*
		Here iq is negative so ir was too large.
		*/

		ir--;
		iq += j;
	    }

	    if (i < t)
		iq += x_dig(i);

	    /*
	    z_dig(k) = quotient, c = remainder.
	    */

	    iq_on_j = iq / j;
	    z_dig(k) = iq_on_j + ir;
	    c = iq - j * iq_on_j;		/* i.e. c = iq % j */
	}
    }

    if (round == MP_TRUNC)
    {
	/*
	Normalization is not needed.
	*/

	mp_set_sign(zp, result_sign);
	mp_expt(zp) = result_expt;
    }
    else
    {
	/*
	Normalize and round or truncate result.  Restore stack pointer.
	*/

	mp_nzr(result_sign, result_expt, zp, z_ptr(0), guard);
	mp_acc_delete(acc_index);
    }

    DEBUG_1("-} z = ", zp);
    DEBUG_END();
    return z;
}
