#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_int
mp_cmp		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Compares the mp numbers x and y, returning:

	-1	if x <  y,
	 0	if x == y,
	+1	if x >  y.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_sign_type	xs;
    
    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+cmp {\n");
    DEBUG_1("x = ", xp);
    DEBUG_1("y = ", yp);

    /*
    Compare signs of x and y.
    */

    xs = mp_sign(xp);
    if (xs < mp_sign(yp))
    {
	DEBUG_PRINTF_1("-} result = -1\n");
	DEBUG_END();

	return -1;
    }

    if (xs > mp_sign(yp))
    {
	DEBUG_PRINTF_1("-} result = 1\n");
	DEBUG_END();

	return 1;
    }


    /*
    sign(x) == sign(y), see if they are both zero.
    */

    if (xs == 0)
    {
	DEBUG_PRINTF_1("-} result = 0\n");
	DEBUG_END();

	return 0;
    }


    /*
    Compare exponents then digits.
    */

    if (mp_expt(xp) < mp_expt(yp))
    {
	DEBUG_PRINTF_2("-} result = %d\n", -xs);
	DEBUG_END();

	return -xs;
    }

    if (mp_expt(xp) > mp_expt(yp))
    {
	DEBUG_PRINTF_2("-} result = %d\n", xs);
	DEBUG_END();

	return xs;
    }


    /*
    The following debugging statement may look inefficient, but when
    debugging is turned off the whole statement disappears.
    */

    DEBUG_PRINTF_2("-} result = %d\n", xs * mp_comp_digits(xp, yp));
    DEBUG_END();

    return xs * mp_comp_digits(xp, yp);
}



mp_int
mp_cmp_abs	WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Compares mp numbers int_abs(x) and int_abs(y), returning:

	-1	if int_abs(x) <  int_abs(y),
	 0	if int_abs(x) == int_abs(y),
	+1	if int_abs(x) >  int_abs(y).
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_sign_type	x_sign = mp_sign(xp), y_sign = mp_sign(yp);
    mp_int		result;


    mp_set_sign(xp, int_abs(x_sign));
    mp_set_sign(yp, int_abs(y_sign));

    result = mp_cmp(x, y);

    mp_set_sign(xp, x_sign);
    mp_set_sign(yp, y_sign);

    return result;
}


mp_int
mp_cmp_int	WITH_2_ARGS(
	mp_float,	x,
	mp_int,		i
)
/*
Compares the mp number x with the integer i, returning:

	-1	if x <  i,
	 0	if x == i,
	+1	if x >  i.

Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_acc_float	temp;
    mp_int		result;

    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+cmp_int {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_2("i = %d\n", i);

    /*
    Convert i to mp temporary and compare.
    */

    mp_acc_float_alloc(mp_b(xp), mp_t(xp), temp);

    mp_int_to_mp(i, temp);
    result = mp_cmp(x, temp);

    mp_acc_float_delete(temp);

    DEBUG_PRINTF_2("-} result = %d\n", result);
    DEBUG_END();

    return result;
}


mp_int
mp_cmp_q	WITH_3_ARGS(
	mp_float,	x,
	mp_int,		i,
	mp_int,		j
)
/*
Compares the mp number x with the rational number i/j, returning:

	-1	if x <  i/j,
	 0	if x == i/j,
	+1	if x >  i/j.

Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_acc_float	temp;
    mp_base_type	b;
    mp_round_type	save_round;
    mp_int		result;


    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+cmp_q {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_2("i = %d\n", i);
    DEBUG_PRINTF_2("j = %d\n", j);

    if (j == 0)
	mp_error("mp_cmp_q: j is 0");


    /*
    Make j positive.
    */

    if (j < 0)
    {
	i = -i;
	j = -j;
    }

    mp_int_gcd(&i, &j);

    if (j == 1)
    {
	/*
	The following debugging statement may look inefficient, but when
	debugging is turned off the whole statement disappears.
	*/

	DEBUG_PRINTF_2("-} result = %d\n", mp_cmp_int(x, i));
	DEBUG_END();

	return mp_cmp_int(x, i);
    }
    

    /*
    Allocate a temporary float with extra digits so j * x can be
    formed exactly.
    */

    b = mp_b(xp);
    mp_acc_float_alloc(b, mp_t(xp) + mp_guard_digits(j, b), temp);

    save_round = round;
    round = MP_TRUNC;

    mp_move(x, temp);
    mp_mul_int_eq(temp, j);


    /*
    Now compare j * x with i.
    */

    result = mp_cmp_int(temp, i);

    round = save_round;
    mp_acc_float_delete(temp);

    DEBUG_PRINTF_2("-} result = %d\n", result);
    DEBUG_END();

    return result;
}


mp_int
mp_cmp_double	WITH_2_ARGS(
	mp_float,	x,
	double,		d
)
/*
Compares the mp number x with the (machine) double d, returning:

	-1	if x <  d,
	 0	if x == d,
	+1	if x >  d.

Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_acc_float	temp;
    mp_int		result;


    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+cmp_double {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_2("d = %g\n", d);

    
    mp_acc_float_alloc(mp_b(xp), mp_t(xp), temp);

    conv_double_to_mp(d, temp);
    result = mp_cmp(x, temp);

    mp_acc_float_delete(temp);

    DEBUG_PRINTF_2("-} result = %d\n", result);
    DEBUG_END();

    return result;
}
