
#include "defs.h"
#include "ring.h"
#include "mat.h"
#include "integer.e"

/*
 * This file includes
 *
 *	mat_ring_trans_sub()       - 
 *	mat_ring_trans()           - Returns transpose of a matrix or vector
 */


t_void
mat_ring_trans_sub WITH_3_ARGS(
t_handle,	ring,
matrix,	a,
matrix*,	pc
)
/*
 * Construct the transpose pc of a.
 * a can be an m x n matrix (pc will then be n x m).
 * If a is equal to *pc, the transpose overwrites the original matrix.
 * "ring" is the coefficient ring.
 */
{
	block_declarations;
	register t_int	i;
	register t_int	j;
	register t_int 	*ptr_a;
	register t_int	*ptr_c;
	register t_int	m;
	register t_int	n;
	register t_int	dummy;
	register t_int	aposn;
	register t_int	cposn;
	matrix	mata;
	matrix	matc;
	matrix	tempmat;
	Logical	respack;

	mata = matc = 0;
	m = mat_row(a);		n = mat_col(a);

	/*
	 * Unpack matrix
	 */
	respack = mat_result_pkd(ring, *pc);
	mat_create_unpkd(ring, a, mata, m, n);
	if (a != *pc)
	{
		mat_alloc_result_unpkd(respack, *pc, matc, n, m);
	}
	else
		matc = mata;

	/*
	 * Form transpose
	 */
	if (a == *pc)
	/*
	 * transpose overwrites original
	 */
	{
		if (m == 1)
		/*
		 * row vector
		 */
		{
			mat_row(mata) = n;
			mat_col(mata) = 1;
		}
		else if (n == 1)
		/*
		 * column vector
		 */
		{
			mat_row(mata) = 1;
			mat_col(mata) = m;
		}
		else if (m == n)
		/*
		 * square matrix
		 */
		{
			ptr_a = mat_elt0_ptr(mata);
			for (i = 1; i < m; i++)
				for (j = i + 1; j <= n; j++)
				{
					aposn = (i - 1) * n + j;
					cposn = (j - 1) * m + i;
					dummy = ptr_a[ cposn ];
					ptr_a[ cposn ] = ptr_a[ aposn ];
					ptr_a[ aposn ] = dummy;
				}
		}
		else
		/*
		 * m x n matrix
		 * The method is to copy the original matrix to a temporary matrix,
		 * then copy it back (with the entries in the appropriate position).
		 *
		 * There is probably a better way to do this but I don't know of any.
		 */
		{
			tempmat = mat_buff_alloc(m, n);
			dummy = m * n;
			ptr_a = mat_elt0_ptr(mata);
			ptr_c = mat_elt0_ptr(tempmat);
			for (i = 1; i <= dummy; i++)
				ptr_c[i] = ptr_a[i];

			for (i = 1; i <= m; i++)
				for (j = 1; j <= n; j++)
					ptr_a[(j - 1) * m + i] = ptr_c[(i - 1) * n + j];

			mat_row(mata) = n;
			mat_col(mata) = m;
			mat_buff_free(&tempmat);
		}
	}
	else
	/*
	 * transpose is put in another matrix
	 */
	{
		ptr_a = mat_elt0_ptr(mata);
		ptr_c = mat_elt0_ptr(matc);
		for (i = 1; i <= m; i++)
			for (j = 1; j <= n; j++)
				ptr_c[(j - 1) * m + i] = ptr_a[(i - 1) * n + j];

		mat_row(matc) = n;
		mat_col(matc) = m;
		/*
		 * Increment reference counts
		 */
		mat_incref_entries(ring, matc);

	}
	/*
	 * Pack result, if required
	 */
	mat_create_result(ring, respack, *pc, matc);
	if (a != *pc)
		mat_free_unpkd(a, mata);

	return;
}



matrix 
mat_ring_trans WITH_2_ARGS(
	t_handle,	cring,
	matrix,	a
)
/* 
 *	Construct the transpose of a.
 *	Handle to result is returned.
 *	a can be an m x n matrix (result will then be n x m).
 *
 *	"cring" is the coefficient ring.
 *
 *	It calls mat_ring_trans_sub
 */
{
	matrix c = 0;
	mat_ring_trans_sub(cring, a, &c);
	return c;
}

