/*  add.c
*/

#include "defs.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"


void
inthdl_add_beta		WITH_3_ARGS(
    inthdl_handle,	x,
    t_int,	y,
    inthdl_handle,	sum
)
/*
Add the integers in block x and betadigit y giving sum, which must be a
pre-allocated integer block of sufficient size, that is, with room for at
least as many beta-digits as inthdl_curr_size(x) + 1.

The result argument sum may be the same as argument x.
*/
{
    inthdl_length	xlen;
    inthdl_sign		xsign, ysign;

    DEBUG_INTHDL_BETA("+inthdl_add_beta", x, y);

    xlen = intbig_curr_size(x);	
    xsign = intbig_sign(x);

    if (y == 0)
    {
	if (x != sum)
	{
	    intbig_copy_digits(x, 0, xlen, sum, 0);
	    intbig_curr_size(sum) = xlen;
	    intbig_sign(sum) = xsign;
	}

	DEBUG_INTHDL_1("-inthdl_add_beta", sum);
	return;
    }

    if (y < 0)
    {
	ysign = -1;
	y = -y;
    }
    else
	ysign = 1;

    if (xlen == 0 || xsign == 0)
    {
	intbig_curr_size(sum) = 1;

	intbig_digit(sum, 0) = y;
	intbig_sign(sum) = ysign;

	DEBUG_INTHDL_1("-inthdl_add_beta", sum);
	return;
    }

    if (xsign == ysign)
    {
	/*
	Perform addition.
	*/

	register t_int	*xp, *sump, *limit;
	register intbig_medium	carry;

	xp = intbig_dig0_ptr(x);
	sump = intbig_dig0_ptr(sum);

	carry = y;
	limit = xp + xlen;

	while (xp < limit)
	{
	    register intbig_medium	s;

	    s = *xp++ + carry;
	    if (s >= BETA)
	    {
		*sump++ = s - BETA;
		carry = 1;
	    }
	    else
	    {
		*sump++ = s;
		carry = 0;
	    }
	}

	if (carry)
	    *sump++ = 1;

	intbig_curr_size(sum) = sump - intbig_dig0_ptr(sum);
	intbig_sign(sum) = xsign;
    }
    else if (xlen == 1)
    {
	/*
	Subtraction where x has only 1 digit.
	*/

	intbig_medium	s;

	s = intbig_digit(x, 0) - y;

	if (s > 0)
	{
	    intbig_digit(sum, 0) = s;
	    intbig_curr_size(sum) = 1;
	    intbig_sign(sum) = 1;
	}
	else if (s < 0)
	{
	    intbig_digit(sum, 0) = -s;
	    intbig_curr_size(sum) = 1;
	    intbig_sign(sum) = -1;
	}
	else
	    intbig_sign(sum) = 0;
    }
    else
    {
	/*
	Perform subtraction where x has 2 or more digits so x > y (ignoring
	signs).
	*/

	register t_int	*xp, *sump, *limit, *zerop;
	register intbig_medium	carry;

	xp = intbig_dig0_ptr(x);
	zerop = sump = intbig_dig0_ptr(sum);

	carry = -y;
	limit = xp + xlen;

	while (xp < limit)
	{
	    register intbig_medium s;
	    
	    s = *xp++ + carry;

	    if (s < 0)
	    {
		s += BETA;
		carry = -1;
	    }
	    else
		carry = 0;

	    if (*sump++ = s)
		zerop = sump;
	}

	DENY(carry);

	if (zerop == intbig_dig0_ptr(sum))
	    intbig_sign(sum) = 0;
	else
	{
	    intbig_sign(sum) = xsign;
	    intbig_curr_size(sum) = zerop - intbig_dig0_ptr(sum);
	}
    }

    DEBUG_INTHDL_1("-inthdl_add_beta", sum);
}


void
inthdl_add  WITH_3_ARGS(
    inthdl_handle,  x,
    inthdl_handle,  y,
    inthdl_handle,  sum
)
/*
Add the integers in blocks x and y giving sum, which must be a pre-allocated
integer block of sufficient size, that is, with room for at least as many
beta-digits as max(inthdl_curr_size(x), inthdl_curr_size(y)) + 1.

The result argument sum may be the same as either or both of the arguments
x and y.
*/
{
    inthdl_sign		xsign, ysign;
    inthdl_length	xlen, ylen;


    DEBUG_INTHDL_2("+inthdl_add", x, y);

    xsign = intbig_sign(x);
    ysign = intbig_sign(y);

    xlen = intbig_curr_size(x);
    ylen = intbig_curr_size(y);

    if (xsign == 0)
    {
	if (y != sum)
	{
	    intbig_copy_digits(y, 0, ylen, sum, 0);
	    intbig_curr_size(sum) = ylen;
	    intbig_sign(sum) = intbig_sign(y);
	}

	DEBUG_INTHDL_1("-inthdl_add", sum);
	return;
    }
    else if (ysign == 0)
    {
	if (x != sum)
	{
	    intbig_copy_digits(x, 0, xlen, sum, 0);
	    intbig_curr_size(sum) = xlen;
	    intbig_sign(sum) = intbig_sign(x);
	}

	DEBUG_INTHDL_1("-inthdl_add", sum);
	return;
    }

    /*
    Ensure that the length of x is not smaller than y.
    */

    if (xlen < ylen)
    {
	inthdl_handle	temph;
	inthdl_length	templ;
	inthdl_sign	temps;

	temph = x; x = y; y = temph;
	templ = xlen; xlen = ylen; ylen = templ;
	temps = xsign; xsign = ysign; ysign = temps;
    }

    if (xsign == ysign)
    {
	/*
	Perform addition.
	*/

	register t_int	*xp, *yp, *sump, *xlimit, *ylimit;
	register intbig_medium	carry;

	xp = intbig_dig0_ptr(x);
	yp = intbig_dig0_ptr(y);
	sump = intbig_dig0_ptr(sum);

	carry = 0;
	xlimit = xp + xlen;
	ylimit = yp + ylen;

	while (yp < ylimit)
	{
	    register intbig_medium	s;

	    s = *xp++ + *yp++ + carry;
	    if (s >= BETA)
	    {
		*sump++ = s - BETA;
		carry = 1;
	    }
	    else
	    {
		*sump++ = s;
		carry = 0;
	    }
	}


	if (carry)
	{
	    while (xp < xlimit)
	    {
		register intbig_medium	s;

		s = *xp++ + 1;
		if (s >= BETA)
		    *sump++ = s - BETA;
		else
		{
		    *sump++ = s;
		    while (xp < xlimit)
			*sump++ = *xp++;
		    carry = 0;
		    break;
		}
	    }
	    if (carry)
		*sump++ = 1;
	}
	else
	    while (xp < xlimit)
		*sump++ = *xp++;

	intbig_curr_size(sum) = sump - intbig_dig0_ptr(sum);
	intbig_sign(sum) = xsign;
    }
    else
    {
	/*
	Perform subtraction.
	*/

	register t_int	*xp, *yp, *sump, *zerop;
	register t_int	*xlimit, *ylimit;
	register intbig_medium	carry;

	intbig_sign(sum) = xsign;

	xp = intbig_dig0_ptr(x);
	yp = intbig_dig0_ptr(y);

	xlimit = xp + xlen;
	ylimit = yp + ylen;

	if (xlen == ylen)
	{
	    do
	    {
		xlimit--;
		ylimit--;
	    } while (xlimit >= xp && *xlimit == *ylimit);

	    if (xlimit < xp)
	    {
		intbig_sign(sum) = 0;
		DEBUG_INTHDL_1("-inthdl_add", sum);
		return;
	    }

	    if (*xlimit < *ylimit)
	    {
		t_int	*temp;

		temp = xp; xp = yp; yp = temp;
		temp = xlimit; xlimit = ylimit; ylimit = temp;
		intbig_sign(sum) = ysign;
	    }

	    xlimit++;
	    ylimit++;
	}

	/*
	Now x > y (ignoring signs).
	*/

	zerop = sump = intbig_dig0_ptr(sum);
	carry = 0;

	while (yp < ylimit)
	{
	    register intbig_medium s = *xp++ - *yp++ + carry;

	    if (s < 0)
	    {
		s += BETA;
		carry = -1;
	    }
	    else
		carry = 0;

	    if (*sump++ = s)
		zerop = sump;
	}

	if (carry)
	{
	    while (xp < xlimit)
	    {
		register intbig_medium	s;

		s = *xp++ - 1;
		if (s < 0)
		    *sump++ = s + BETA;
		else
		{
		    *sump++ = s;
		    if (s)
			zerop = sump;

		    while (xp < xlimit)
			if (*sump++ = *xp++)
			    zerop = sump;
		    carry = 0;
		    break;
		}
	    }
	    DENY(carry);
	}
	else
	    while (xp < xlimit)
		if (*sump++ = *xp++)
		    zerop = sump;

	if (zerop == intbig_dig0_ptr(sum))
	    intbig_sign(sum) = 0;
	else
	    intbig_curr_size(sum) = zerop - intbig_dig0_ptr(sum);
    }

    DEBUG_INTHDL_1("-inthdl_add", sum);
}
