#include "defs.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"

void
integer_gcd_mult    WITH_5_ARGS(
    integer_big,     aint,
    integer_big,     bint,
    integer_big *,   gcd,
    integer_big *,   m1,
    integer_big *,   m2
)
/*
 * Input:  aint, bint - single or infinite-precision integers
 * Output: gcd (aint, bint) - a single or infinite-precision integer
 * which is the greatest common divisor of aint and bint, and integers
 * m1 and m2 such that gcd = m1 * aint + m2 * bint.
 */
{
    block_declarations;

    inthdl_length   alen;
    inthdl_length   blen;
    inthdl_length   rlen;
    inthdl_handle   r1;
    inthdl_handle   r2;
    inthdl_handle   r3;
    inthdl_handle   ahdl;
    inthdl_handle   bhdl;

    /*
     * case aint or bint = 0?
     */

    if (integer_sign(aint) == 0)
    {
	*gcd = integer_abs(bint);
	*m1 = 0;

	if (integer_compare(bint, 0) > 0)
	    *m2 = 1;
	else
	    *m2 = -1;
	return;
    }
    else if (integer_sign(bint) == 0)
    {
	*gcd = integer_abs(aint);
	*m2 = 0;

	if (integer_compare(aint, 0) > 0)
	    *m1 = 1;
	else
	    *m1 = -1;
	return;
    }

    /*
     * case aint and bint single-precision
     */

    else if (integer_is_single(aint) && integer_is_single(bint))
    {
	ib_gcd_mult(aint, bint, gcd, m1, m2);
	return;
    }

    /*
     * non-trivial case...
     *      ONE or BOTH are infinite_precision
     */

    if (integer_is_single(aint))
        ahdl = inthdl_buf_alloc_1(aint);
    else
	ahdl = inthdl_big_to_handle(aint);

    alen = intbig_curr_size(ahdl);

    if (integer_is_single(bint))
        bhdl = inthdl_buf_alloc_1(bint);
    else
	bhdl = inthdl_big_to_handle(bint);

    blen = intbig_curr_size(bhdl);

    /*
     * allocate storage for result 
     */

    rlen = (alen > blen) ? alen : blen;
    r1 = inthdl_buf_alloc(rlen);
    r2 = inthdl_buf_alloc(rlen);
    r3 = inthdl_buf_alloc(rlen);

    /*
     * compute the gcd of aint and bint storing into result
     */

    inthdl_gcd_mult(ahdl, bhdl, r1, r2, r3);

    if (integer_is_single(aint))
	inthdl_buf_delete(ahdl);
    if (integer_is_single(bint))
	inthdl_buf_delete(bhdl);

    /*
     * convert result to a valid single or infinite-precision integer
     */

    *gcd = inthdl_standardize(r1);
    *m1 = inthdl_standardize(r2);
    *m2 = inthdl_standardize(r3);
}
