#include "mem.h"

t_void memi_extend_handle_table()
{
    if (!HANDLE_BASE)
    {
	/*
	First time.  Create a new handle table, etc.
	*/

	mem_handle_base = (t_mem_word **)memi_get_block(
	    INIT_HANDLES + HEADER_SIZE
	);
	memi_handle_limit = HANDLE_BASE + BLOCK_SIZE_NO_HEADER(HANDLE_BASE);
	memi_next_handle = (t_ptr)&HANDLE_BASE[1];
	memi_free_handle_list = NULL;
	BLOCK_HANDLE(HANDLE_BASE) = P_HANDLE_TABLE;
    }
    else
    {
	/*
	Extend the current table.  (Assumes handle table full, no free handles).
	*/

	t_ptr	old_base, old_limit, new_base, copy_old_base, bp;
	t_len	size;

	ASSERT(memi_handle_limit == memi_next_handle);
	ASSERT(memi_free_handle_list == 0);

	copy_old_base = old_base = HANDLE_BASE;
	old_limit = memi_handle_limit;
	size = old_limit - old_base;
	new_base = memi_get_block(size + EXTEND_HANDLES + HEADER_SIZE);
	HANDLE_BASE = new_base;
	BLOCK_HANDLE(new_base) = P_HANDLE_TABLE;
	memi_handle_limit = new_base + BLOCK_SIZE_NO_HEADER(new_base);
	memi_next_handle = new_base + size;

	VERBOSE(
	    mem_printf(
		"mem: extending handle table: old size: %d, new size %d\n",
		size, BLOCK_SIZE_NO_HEADER(new_base)
	    );
	)

	/* Skip handle 0 */
	old_base++;
	new_base++;

	while (old_base < old_limit)
	{
	    bp = DEREF(old_base++);
	    if (bp == P_DELETED_HANDLE)
		DEREF(new_base++) = bp;
	    else
	    {
		ASSERT(BLOCK_HANDLE(bp) == old_base - 1);
		BLOCK_HANDLE(bp) = new_base;
		DEREF(new_base++) = bp;
	    }
	}

	memi_free_block_coalesce(copy_old_base);
    }
}

t_handle f_mem_alloc FL1(n)
t_int	n;
{
    t_ptr	bp, hp;

    STATS(memi_stats_alloc++);
    CHECK("mem_alloc", n >= MIN_USER_SIZE, n);

    if (hp = memi_free_handle_list)
	memi_free_handle_list = DEREF(hp);
    else
    {
	if (memi_next_handle == memi_handle_limit)
	    memi_extend_handle_table();

	hp = memi_next_handle++;
    }

    bp = memi_get_block(BYTES_TO_PTRS(n) + HEADER_SIZE);
    DEREF(hp) = bp;
    BLOCK_HANDLE(bp) = hp;

    ASSIGN_CALLER(bp);
    ASSIGN_MARK(bp);
    TRACE("mem_alloc", HNORM(hp - HANDLE_BASE), n);

    return HNORM(hp - HANDLE_BASE);
}

t_handle f_mem_alloc_words FL1(n)
t_int	n;
{
    t_ptr	bp, hp;

    STATS(memi_stats_alloc_words++);
    CHECK("mem_alloc_words", n >= MIN_USER_SIZE, n);

    n = UWORDS_TO_PTRS(n);
    if (hp = memi_free_handle_list)
	memi_free_handle_list = DEREF(hp);
    else
    {
	if (memi_next_handle == memi_handle_limit)
	    memi_extend_handle_table();

	hp = memi_next_handle++;
    }

    bp = memi_get_block(n + HEADER_SIZE);
    DEREF(hp) = bp;
    BLOCK_HANDLE(bp) = hp;

    ASSIGN_CALLER(bp);
    ASSIGN_MARK(bp);
    TRACE("mem_alloc_words", HNORM(hp - HANDLE_BASE), PTRS_TO_BYTES(n));

    return HNORM(hp - HANDLE_BASE);
}

t_handle f_mem_alloc_zero FL1(n)
t_int	n;
{
    t_ptr	bp, hp, ep;

    STATS(memi_stats_alloc++);
    CHECK("mem_alloc_zero", n >= MIN_USER_SIZE, n);

    if (hp = memi_free_handle_list)
	memi_free_handle_list = DEREF(hp);
    else
    {
	if (memi_next_handle == memi_handle_limit)
	    memi_extend_handle_table();

	hp = memi_next_handle++;
    }

    bp = memi_get_block(BYTES_TO_PTRS(n) + HEADER_SIZE);

    DEREF(hp) = bp;
    BLOCK_HANDLE(bp) = hp;
    ASSIGN_CALLER(bp);
    ASSIGN_MARK(bp);

    ep = NEXT_BLOCK(bp) - HEADER_SIZE;
    while (bp < ep)
	DEREF(bp++) = 0;

    TRACE("mem_alloc_zero", HNORM(hp - HANDLE_BASE), n);

    return HNORM(hp - HANDLE_BASE);
}

t_handle f_mem_alloc_words_zero FL1(n)
t_int	n;
{
    t_ptr	bp, hp, ep;

    STATS(memi_stats_alloc_words++);
    CHECK("mem_alloc_words_zero", n >= MIN_USER_SIZE, n);

    n = UWORDS_TO_PTRS(n);

    if (hp = memi_free_handle_list)
	memi_free_handle_list = DEREF(hp);
    else
    {
	if (memi_next_handle == memi_handle_limit)
	    memi_extend_handle_table();

	hp = memi_next_handle++;
    }

    bp = memi_get_block(n + HEADER_SIZE);

    DEREF(hp) = bp;
    BLOCK_HANDLE(bp) = hp;
    ASSIGN_CALLER(bp);
    ASSIGN_MARK(bp);

    ep = NEXT_BLOCK(bp) - HEADER_SIZE;
    while (bp < ep)
	DEREF(bp++) = 0;

    TRACE(
	"mem_alloc_words_zero", HNORM(hp - HANDLE_BASE), PTRS_TO_BYTES(n)
    );

    return HNORM(hp - HANDLE_BASE);
}

t_handle f_mem_alloc_aligned FL2(alignment, n)
t_int	alignment, n;
/*
Return a handle-based block which is guaranteed to have its address a multiple
of the given alignment.
*/
{
    t_ptr	bp, hp, newp;
    t_int	newi, mod;

    STATS(memi_stats_alloc++);
    CHECK("mem_alloc", n >= MIN_USER_SIZE, n);

    if (hp = memi_free_handle_list)
	memi_free_handle_list = DEREF(hp);
    else
    {
	if (memi_next_handle == memi_handle_limit)
	    memi_extend_handle_table();

	hp = memi_next_handle++;
    }

    bp = memi_get_block(BYTES_TO_PTRS(n + alignment) + 2 * HEADER_SIZE + 1);
    if ((t_int)bp % alignment)
    {
	newp = bp + HEADER_SIZE + 1;
	newi = (t_int)newp;
	mod = newi % alignment;
	if (mod)
	    newp = (t_ptr)(newi + (alignment - mod));

	NEXT_BLOCK(newp) = NEXT_BLOCK(bp);
	PREV_BLOCK(NEXT_BLOCK(bp)) = newp;

	NEXT_BLOCK(bp) = newp;
	PREV_BLOCK(newp) = bp;

	memi_free_block(bp);
	bp = newp;
    }

    DEREF(hp) = bp;
    BLOCK_HANDLE(bp) = hp;
    ASSIGN_CALLER(bp);
    ASSIGN_MARK(bp);
    TRACE("mem_alloc", HNORM(hp - HANDLE_BASE), n);

    return HNORM(hp - HANDLE_BASE);
}

t_void f_mem_delete_h FL1(h)
t_handle	h;
{
    t_ptr	hp, bp;

    ENTER_CHECK();
    STATS(memi_stats_delete++);
    TRACE("mem_delete_h", h, 0);
    CHECK_HANDLE("mem_delete_h", h);

    HNORMALIZE(h);

    hp = (t_ptr)&HANDLE_BASE[h];
    bp = DEREF(hp);
    IF_DELCHECK(
	(
	    DEREF(hp) = P_DELETED_HANDLE
	),
	(
	    DEREF(hp) = memi_free_handle_list,
	    memi_free_handle_list = hp
	)
    );

    memi_free_block_coalesce(bp);
}

t_void f_mem_delete_hptr FL1(hptr)
t_handle	*hptr;
{
    t_ptr	hp, bp;
    t_handle	h;

    ENTER_CHECK();
    h = *hptr;
    *hptr = 0;
    STATS(memi_stats_delete++);
    TRACE("mem_delete_hptr", h, 0);
    CHECK_HANDLE("mem_delete_hptr", h);

    HNORMALIZE(h);
    hp = (t_ptr)&HANDLE_BASE[h];
    bp = DEREF(hp);
    IF_DELCHECK(
	(
	    DEREF(hp) = P_DELETED_HANDLE
	),
	(
	    DEREF(hp) = memi_free_handle_list,
	    memi_free_handle_list = hp
	)
    );

    memi_free_block_coalesce(bp);
}

t_void f_mem_reduce FL2(h, n)
t_handle	h;
int		n;
{
    t_ptr	bp;

    CHECK_HANDLE("mem_reduce", h);
    STATS(memi_stats_reduce++);

    HNORMALIZE(h);
    bp = HANDLE_ACCESS(h);
    n = BYTES_TO_PTRS(n) + HEADER_SIZE;

    CHECK("mem_reduce", n <= BLOCK_SIZE(bp), h);
    memi_reduce_block(bp, n);
}

t_void f_mem_reduce_words FL2(h, n)
t_handle	h;
int		n;
{
    t_ptr	bp;

    CHECK_HANDLE("mem_reduce", h);
    STATS(memi_stats_reduce++);

    HNORMALIZE(h);
    bp = HANDLE_ACCESS(h);
    n = UWORDS_TO_PTRS(n) + HEADER_SIZE;

    CHECK("mem_reduce", n <= BLOCK_SIZE(bp), h);
    memi_reduce_block(bp, n);
}

t_void f_mem_extend FL2(h, n)
t_handle	h;
t_len		n;
{
    t_ptr	hp, bp, newp;
    t_len	old_size, np;
    TRACE_ALLOCS_LOCAL(char *name; t_int line_num);
    MARKS_LOCAL(t_int mark);

    CHECK_HANDLE("mem_extend", h);
    STATS(memi_stats_extend++);

    HNORMALIZE(h);

    np = BYTES_TO_PTRS(n) + HEADER_SIZE;

    hp = (t_ptr)&HANDLE_BASE[h];
    bp = DEREF(hp);

    old_size = NEXT_BLOCK(bp) - bp;
    CHECK("mem_extend", np >= old_size, h);

    if (np > old_size)
    {
	TRACE("mem_extend", HNORM(h), n);

	/*
	Store name, etc. first because this may be over written if the
	block is shifted left.
	*/

	IF_TRACE_ALLOCS(
	    name = BLOCK_FILE_NAME(bp);
	    line_num = BLOCK_LINE_NUM(bp);
	)
	IF_USE_MARKS(
	    mark = BLOCK_MARK(bp);
	)

	newp = memi_extend_block(bp, np, 0);

	if (newp != bp)
	{
	    BLOCK_HANDLE(newp) = hp;
	    DEREF(hp) = newp;

	    IF_TRACE_ALLOCS(
		BLOCK_FILE_NAME(newp) = name;
		BLOCK_LINE_NUM(newp) = line_num
	    )
	    IF_USE_MARKS(
		BLOCK_MARK(newp) = mark;
	    )
	}
    }
}

t_void f_mem_extend_words FL2(h, n)
t_handle	h;
t_len		n;
{
    t_ptr	hp, bp, newp;
    t_len	old_size, np;
    TRACE_ALLOCS_LOCAL(char *name; t_int line_num);
    MARKS_LOCAL(t_int mark);

    n = UWORDS_TO_PTRS(n);
    CHECK_HANDLE("mem_extend_words", h);
    STATS(memi_stats_extend++);

    HNORMALIZE(h);

    np = n + HEADER_SIZE;

    hp = (t_ptr)&HANDLE_BASE[h];
    bp = DEREF(hp);

    old_size = NEXT_BLOCK(bp) - bp;
    CHECK("mem_extend_words", np >= old_size, h);

    if (np > old_size)
    {
	TRACE("mem_extend_words", HNORM(h), PTRS_TO_BYTES(n));

	/*
	Store name, etc. first because this may be over written if the
	block is shifted left.
	*/

	IF_TRACE_ALLOCS(
	    name = BLOCK_FILE_NAME(bp);
	    line_num = BLOCK_LINE_NUM(bp);
	)
	IF_USE_MARKS(
	    mark = BLOCK_MARK(bp);
	)

	newp = memi_extend_block(bp, np, 0);

	if (newp != bp)
	{
	    BLOCK_HANDLE(newp) = hp;
	    DEREF(hp) = newp;

	    IF_TRACE_ALLOCS(
		BLOCK_FILE_NAME(newp) = name;
		BLOCK_LINE_NUM(newp) = line_num
	    )
	    IF_USE_MARKS(
		BLOCK_MARK(newp) = mark;
	    )
	}
    }
}

t_void f_mem_extend_zero FL2(h, n)
t_handle	h;
t_len		n;
{
    t_ptr	hp, bp, newp;
    t_len	old_size, np;
    TRACE_ALLOCS_LOCAL(char *name; t_int line_num);
    MARKS_LOCAL(t_int mark);

    CHECK_HANDLE("mem_extend_zero", h);
    STATS(memi_stats_extend++);

    HNORMALIZE(h);

    np = BYTES_TO_PTRS(n) + HEADER_SIZE;
    hp = (t_ptr)&HANDLE_BASE[h];
    bp = DEREF(hp);

    old_size = NEXT_BLOCK(bp) - bp;
    CHECK("mem_extend_zero", np >= old_size, h);

    if (np > old_size)
    {
	TRACE("mem_extend_zero", HNORM(h), n);

	/*
	Store name, etc. first because this may be over written if the
	block is shifted left.
	*/

	IF_TRACE_ALLOCS(
	    name = BLOCK_FILE_NAME(bp);
	    line_num = BLOCK_LINE_NUM(bp);
	)
	IF_USE_MARKS(
	    mark = BLOCK_MARK(bp);
	)

	newp = memi_extend_block(bp, np, 1);

	if (newp != bp)
	{
	    BLOCK_HANDLE(newp) = hp;
	    DEREF(hp) = newp;

	    IF_TRACE_ALLOCS(
		BLOCK_FILE_NAME(newp) = name;
		BLOCK_LINE_NUM(newp) = line_num
	    )
	    IF_USE_MARKS(
		BLOCK_MARK(newp) = mark;
	    )
	}
    }
}

t_void f_mem_extend_words_zero FL2(h, n)
t_handle	h;
t_len		n;
{
    t_ptr	hp, bp, newp;
    t_len	old_size, np;
    TRACE_ALLOCS_LOCAL(char *name; t_int line_num);
    MARKS_LOCAL(t_int mark);

    CHECK_HANDLE("mem_extend_words_zero", h);
    STATS(memi_stats_extend++);

    n = UWORDS_TO_PTRS(n);
    HNORMALIZE(h);

    np = n + HEADER_SIZE;
    hp = (t_ptr)&HANDLE_BASE[h];
    bp = DEREF(hp);

    old_size = NEXT_BLOCK(bp) - bp;
    CHECK("mem_extend_words_zero", np >= old_size, h);

    if (np > old_size)
    {
	TRACE("mem_extend_words_zero", HNORM(h), PTRS_TO_BYTES(n));

	/*
	Store name, etc. first because this may be over written if the
	block is shifted left.
	*/

	IF_TRACE_ALLOCS(
	    name = BLOCK_FILE_NAME(bp);
	    line_num = BLOCK_LINE_NUM(bp);
	)
	IF_USE_MARKS(
	    mark = BLOCK_MARK(bp);
	)

	newp = memi_extend_block(bp, np, 1);

	if (newp != bp)
	{
	    BLOCK_HANDLE(newp) = hp;
	    DEREF(hp) = newp;

	    IF_TRACE_ALLOCS(
		BLOCK_FILE_NAME(newp) = name;
		BLOCK_LINE_NUM(newp) = line_num
	    )
	    IF_USE_MARKS(
		BLOCK_MARK(newp) = mark;
	    )
	}
    }
}

t_handle f_mem_copy FL1(s)
t_handle s;
/*
Return the handle of new block which has a copy of the contents of the
block associated with s.
*/
{
    t_len	words;
    t_handle	d;

    words = UWORDS_TO_PTRS(f_mem_words CALL_FL1(s));
    d = f_mem_alloc_words CALL_FL1(words);

    memi_copy_block_zero_tail(HANDLE_ACCESS(HNORM(s)), HANDLE_ACCESS(HNORM(d)));

    return d;
}

void f_mem_zero_block FL1(s)
t_handle s;
/*
Zero the block associated with s.
*/
{
    t_len	words;
    t_ptr	sp, ep;

    words = UWORDS_TO_PTRS(f_mem_words CALL_FL1(s));

    sp = HANDLE_ACCESS(HNORM(s));
    ep = sp + words;

    while (sp < ep)
	DEREF(sp++) = 0;
}

t_void f_mem_assure_space FL3(h, up, extra)
t_handle	h;
t_void		*up;
t_len		extra;
/*
Assure that *up points into the block associated with h - if not, the block
is extended by extra, and *up is appropriately modified.
*/
{
    t_ptr	p, bp;

    CHECK_HANDLE("mem_assure_space", h);
    HNORMALIZE(h);

    p = *(t_ptr *)up;
    bp = HANDLE_ACCESS(h);

    CHECK("mem_assure_space", p >= bp, HNORM(h));

    if (p >= LEFTMOST_HEADER(NEXT_BLOCK(bp)))
    {
	f_mem_extend_words CALL_FL2(
	    HNORM(h),
	    PTRS_TO_UWORDS(p - bp + HEADER_SIZE + 1 + BYTES_TO_PTRS(extra))
	);
	*(t_ptr *)up = HANDLE_ACCESS(h) + (p - bp);
    }
}
