/*******************************************************************************
  regula_falsi_zero.c 
    find_intervall
    my_value
    regula_falsi_zero
********************************************************************************/


#include "kant.h"                                            



#define MAX_INTERVALL 1000


t_logical
find_intervall  ( R      ,
	          start  ,
		  step   ,
		  func   , 
                  param1 ,
                  param2 ,
		  left   ,
                  right   )




         t_handle    R;
	 t_real    start;  
	 t_real    step; 
	 t_real    (*func) ();
         t_real    param1;
         t_real    param2;
         t_real    *left;                       
         t_real    *right;

/*******************************************************************************

Description:
                                                             
     Finds the an intervall [a,a+step] with func(a)*func(a+step) <0.
     The function starts searching in start a will test the intervalls
         
           [a+j*step,a+(j+1)*step]     -MAX_INTERVALL  < j < MAX_INTERVALL
      
     The function will return TRUE if there ex. such a j and FALSE otherwise.                                                     



Calling sequence:
                                                           
     t_handle     R           : A real field.
     t_real     start       : starting point.
     t_real     step        : as desc. above
     t_real     func ()     : A function with 4 parameters.
                              (Two are fixed : the real field R and the
                               evaluation point x. The other two parameters
                               are free and given in param1 and param2      )
                              The function func has to return a t_real value.
     t_real     param1      : The first free parameter.
     t_real     param2      : The second free parameter.
     t_real     *left       : The lower bound of the computed intervall.                            
     t_real     *right      : The upper bound of the computed intervall.                            
                                       
     t_logical    Found       : see above.                           

                                        

     Found = find_intervall (R,start,step,&func,param1,param2,left,right);
                                                             
                            
History:                                 
    
        92-06-16 MD    first version



********************************************************************************/
{
           t_real          temp,t_step;
           t_real          x_iter,f_iter;

           integer_small   sign,direction,j;
           t_logical         found,stop;




 
  direction = 1;     /* First test intervalls to the right hand  */
  stop      = FALSE; /* Continue to search                       */
  found     = FALSE; /* We found a zero of f                     */

  t_step    = real_incref (step); 

  do
  { 
    j = 0;             /* No. of tested intervalls                 */
    if (direction == -1)  
    {                       
      real_delete (&t_step);
      t_step = real_negate (R,step);
    }

    *left  = real_incref (start);
    x_iter = real_incref (start);
    f_iter = (*func) (R,x_iter,param1,param2);

    sign = real_sign (R,f_iter);

  
    while ((sign == real_sign (R,f_iter)) && (j <MAX_INTERVALL)) 
    {                                  
      j++;  /* One more intervall  */
                       
      temp = x_iter;
      x_iter = real_add (R,x_iter,t_step);
      real_delete (&temp);


      real_delete (&f_iter);
      f_iter = (*func) (R,x_iter,param1,param2);

      if (sign == real_sign (R,f_iter))
	{
          real_delete (left);
          *left = real_incref (x_iter);
	}
    }

    real_delete (&f_iter);
 
    if (j >= MAX_INTERVALL ) /* No zero of func found   */
    {
      if (direction == 1)    /* Continue to search  to the left side */
      {
        real_delete (left);
        real_delete (&x_iter);
        direction = -1;
        j         = 0;
      }
      else        
        {
          real_delete (left);
          real_delete (&x_iter);
          real_delete (&t_step);

          found = FALSE;
          stop  = TRUE;
        }
    }
    else
    {         
      real_delete (&t_step);
 
      *right = x_iter;
       
      if (direction == -1)   /* We have to interchange the upper and the lower */
                              /* bound.                                         */
      { 
        temp   = *right;
        *right = *left;
        *left  = temp;
      }
      found = TRUE;
      stop  = TRUE;
    }
  }
  while (!stop);               

             
  return found;

} 


t_real my_value WITH_5_ARGS (t_handle  , R,
                             t_real  , left,
                             t_real  , right,
                             t_real  , f_left,
                             t_real  , f_right)
/*******************************************************************************

Description:
                                                             
     A sub-function of regula - falsi.
     Within regula - falsi we create a linear function and we need the zero
     of this. This zero is named my and will be computed within this 
     subfunction.             
     The linear function is desc. by the two points 
                      (left,f_left)   (right,f_right) 


Calling sequence:
                                                           
     t_handle     R             : A real field.

     t_real     left,f_left   : The first desc. point of the lin. function g
     t_real     right,f_right : The second point 
                                                 
     t_real     my            : the zero of g;
                                 

         g =  my_value (R,left,right,f_left,f_rigth);

History:                                 
    
        92-06-16 MD    first version



********************************************************************************/
{
     t_real     delta,f_delta;
     t_real     temp1,temp2;

     t_real     my;


  delta = real_subtract (R,left,right);
  f_delta = real_subtract (R,f_left,f_right);

  temp1 = real_mult (R,delta,f_left);
  temp2 = real_divide (R,temp1,f_delta);

  my = real_subtract (R,left,temp2);

  real_delete (&temp1);
  real_delete (&temp2);
  real_delete (&delta);
  real_delete (&f_delta);


  return my;

}



t_real 
regula_falsi_zero   (  R      ,
                       start  ,
                       _step   ,
                       func   , 
                       param1 ,
                       param2 ,
                       lambda    )


         t_handle    R;
         t_real    start;
         t_real    _step;
         t_real    (*func) ();
         t_real    param1;
         t_real    param2;    
         t_real    *lambda;

/*******************************************************************************

Description:
                                                             
    Searches for a zero of func. The method used is regula - falsi. 
    To find the zero of func we need some add. information :
      
      1) The function itself (of course) and the parameters of it 
         (named param1 and param2) (these parameters are just 
         optional). 

      2) Since regula - falsi needs an intervall with a zero within
         the first thing we have to do is to find such an intervall.
         Therefor we need a start and the size of the intervall  (step).
         If step is the MEM_NH the function will define step as one.
         Obviously the values of start and step are very important.
         (i.e. if step > min {|x-y| : func (x) = func (y) = 0 ; x<>y}  the 
         function will not work properly).

    The function will return false if no zero was found. 

 


Calling sequence:
                                                           
     t_handle     R           : A real field.
     t_real     start       : starting point.
     t_real     step        : as desc. above
     t_real     func ()     : A function with 4 parameters.
                              (Two are fixed : the real field R and the
                               evaluation point x. The other two parameters
                               are free and given in param1 and param2      )
                              The function func has to return a t_real value.
     t_real     param1      : The first free parameter.
     t_real     param2      : The second free parameter.

     t_real     lambda      : The center of an intervall. Within this intervall
                              there is a point x with func(x) = 0.
                              The size of the intervall depends on the 
                              prec. of R.
                                       
     t_logical    Found       : see above.                           

                                        

     Found = regula_falsi (R,start,step,&func,param1,param2,&lambda);
                                                             
                            
History:                                 
    
        92-06-16 MD    first version



********************************************************************************/
{

     
     t_real     left,right,f_left,f_right;
     t_real     my,f_my,delta,f_delta;
     t_real     l_int,r_int,f_l_int,f_r_int;
     t_real     prec,zero,step;
     t_real     temp1,temp2;


     t_logical    ok;
                       

  if (_step == MEM_NH) 
    step = conv_int_to_real (R,1);
  else
    step = real_incref (_step);

  ok = find_intervall (R,start,step,func,param1,param2,&left,&right);
                               
  real_delete (&step);

  if (ok) 
  {
    if (anf_print_level > 4) 
    {
      printf ("(left,right) = ");real_write (R,left,5);printf ("   ");
                                 real_write (R,right,5);puts ("");
    }
 

    prec = real_make (R,1,-(real_dec_prec (R)-2));
    zero = real_make (R,0,0);

  
    f_left  = (*func) (R,left,param1,param2);
    f_right = (*func) (R,right,param1,param2);

    ok = TRUE;

    do
      {
        my   = my_value (R,left,right,f_left,f_right);
        f_my = (*func) (R,my,param1,param2);
  

        delta = real_mult (R,prec,my);
      
        l_int = real_subtract (R,my,delta);
        r_int = real_add (R,my,delta);
   
        f_r_int = (*func) (R,r_int,param1,param2);
        f_l_int = (*func) (R,l_int,param1,param2);

        temp1 = real_mult (R,f_r_int,f_l_int);
        temp2 = real_mult (R,f_my,f_left);


        real_delete (&delta);
        real_delete (&l_int);
        real_delete (&r_int);
        real_delete (&f_r_int);
        real_delete (&f_l_int);

     
        if (real_compare (R,temp1,zero) <= 0)
  	{
          real_delete (&f_my);
          real_delete (&f_left);
          real_delete (&f_right);
     
          real_delete (&left);
          real_delete (&right);


          ok = FALSE;
          *lambda = my;
	}
        else
          if (real_compare (R,temp2,zero) >= 0) 
	    { 
              real_delete (&left);
              real_delete (&f_left);
            
  
              left = my;
              f_left = f_my;
             }
             else
               {
                 real_delete (&right);
                 real_delete (&f_right);

                 right = my;
                 f_right = f_my;
               }    
          
        real_delete (&temp1);
        real_delete (&temp2);


        if ((ok) && (anf_print_level > 4))
        {
          printf ("(left,right) = ");
          real_write (R,left,5);printf ("   ");
          real_write (R,right,5);puts ("");
        }
  
      }
      while (ok);          

      real_delete (&prec);
      real_delete (&zero);


    if (anf_print_level > 4) 
    {
      printf ("lambda = ");real_write (R,*lambda,5);puts ("");
    } 
                
      return TRUE;   
  }
  
  return FALSE;
}
