/******************************************************************************
  order_prime_over.c
******************************************************************************/
#include "kant.h"

dyn_arr_handle
order_prime_over WITH_3_ARGS (
	order,		ord,
	order,		ordmax,
	integer_big,	p
)
/******************************************************************************
 
Description:	Computes all prime ideals which lie over a prime p.
		ord does not need to be maximal.
		    
Calling sequence:
 
	plst = order_prime_over(order,ordmax,p)

	ord	: order			= t_handle of order
	ordmax	: order			= maximal overorder of ord.
					  Following features are implemented:
					  (a) ord = ordmax (the handles must be equal)
					  (b) ord is the suborder of ordmax.
					  (c) ordmax = MEM_NH. In that case
					      the programm computes the maximal
					      overorder.
	p	: integer_big		= a prime in Z
	plst	: dyn_arr_handle	= dynamic array which contains the prime ideals
					  lying over p (dyn_arr_curr_length is set).

History:

	92-10-07 KW	written
 
******************************************************************************/
{
	block_declarations;
                            
	anf_ideal	pia,pib;
	dyn_arr_handle	plst;
	integer_small	i,j,r;
	t_handle	lst;
	t_logical	flag;

	order_must_be_over_z(ord);
	ordmax = (ordmax) ? order_incref(ordmax) : order_maximal(ord);

/*
**	Factorize (p) in the maximal overorder of ord.
*/
	lst = order_prime_factorize(ordmax,p);
	r = m_poly_z_faclst_len(lst);

/*
**	Let (p) = P_1^{e_1} * ... * P_r^{e_r} in ordmax for prime ideals
**	P_1,...,P_r of ordmax. If p is contained in prime ideals p_1,..,p_s
**	of ord then p_1,...,p_s \in { ord \cap P_i | 1 <= i <= r }.
*/

	if (ord == ordmax)
	{
/*
**		Trivial case.
*/
		plst = dyn_arr_alloc(r);
		dyn_arr_curr_length(plst) = r;
		for (i=1;i<=r;i++) dyn_arr_element(plst,i-1) = anf_ideal_incref(m_poly_z_faclst_factor(lst,i-1));
	}
	else
	{
/*
**		Second case: ord is a true suborder of ordmax.
*/
		for (i=1;i<=r;i++)
		{
			pia = m_poly_z_faclst_factor(lst,i-1);
			pib = anf_ideal_cap_suborder(ordmax,pia);

			if (i == 1)
			{
				plst = dyn_arr_alloc(1);
				dyn_arr_element(plst,0) = pib;
				dyn_arr_curr_length(plst) = 1;
			}
			else
			{
/*
**				Check whether pib is yet in plst.
*/
				flag = TRUE;
				for (j=dyn_arr_curr_length(plst)-1;(flag) && (j>=0);j--)
				{
					flag = !anf_ideal_equal(ord,pib,dyn_arr_element(plst,j));
				}
				if (flag)
				{
					dyn_arr_assure_space_fun(plst,dyn_arr_curr_length(plst)+1,1);
					dyn_arr_element(plst,dyn_arr_curr_length(plst)) = pib;
					dyn_arr_curr_length(plst)++;
				}
				else anf_ideal_delete(ord,&pib);
			}
		}
	}
	anf_ideal_faclst_delete(ordmax,&lst);
	order_delete(&ordmax);
	return(plst);
}
