/******************************************************************************
  order_basis_real_create.c
 
  This file contains:
  order_basis_real_create           (generic)
  order_basis_real_create_pure      (internal)
  order_basis_real_create_power     (internal)
  order_basis_real_create_trans     (internal)
 
******************************************************************************/
                    
#include "kant.h"
#include "real.e"
#include "mat.h"
#include "poly.e"
#include "anf.h"
#include "conv.e"


void
order_basis_real_create WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
Description:

       Creates the real matrix representing the basis of ord and its inverse.
       Depending on what we know about the order different functions are used.
 
       Until now is implemented:
         - equation orders
         - orders given via a transformation matrix over another order

Calling sequence:
 
	order_basis_real_create(ord);
 
      	order         ord      = t_handle of order 

 
History:
 
	91-10-10 JS    minor changes
	91-10-01 JS    first version
 
*******************************************************************************/
{
	block_declarations;
 
	void		order_basis_real_create_pure();
	void		order_basis_real_create_power();
	void		order_basis_real_create_trans();
 

 
	if (!order_reals_known(ord))
		error_internal("No real field given for real basis.");
  
	if(order_basis_is_pure(ord))
	{
		order_basis_real_create_pure(ord);
	}
	else if (order_basis_is_power(ord))
	{
		order_basis_real_create_power(ord);
	}
	else if (order_basis_is_rel(ord))
	{
		order_basis_real_create_trans(ord);
	}
  	else
	{	
		error_internal("order_basis_real_create: Unknown situation.");
	}
 	
	return;
}     
 
 
 
void
order_basis_real_create_pure WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
 
Description:

       Creates the real matrix representing the basis of ord and its inverse.
       This routine assumes that the basis is pure power basis.
       The computation is done straightforward.
 
       Not yet implemented!
 

Calling sequence:
 
	order_basis_real_create_pure(ord);
 
      	order         ord      = t_handle of order 

 
History:
 
	91-10-10 JS    minor changes
	91-10-01 JS    first version
  
*******************************************************************************/
{
	block_declarations;
 
	anf		field;
	integer_small	deg;
	integer_small	i, j, ind, r1, r2, r12;
	t_handle		reals;
	t_real		one, sq2, zero;
	anf_elt		rho, temp1, temp2;
	t_poly	pol;
	matrix		mat;
 
                                 
	deg   = order_abs_degree(ord);
	r1    = order_r1(ord);
	r2    = order_r2(ord);
	r12   = r1 + r2;
	reals = order_reals(ord);
	sq2   = order_sqrt_2(ord);
	field = order_anf(ord);
	pol   = order_poly(ord);
                        

	error_internal("Real basis not implemented for pure orders.");
 
	return;
}
 
 
 
void
order_basis_real_create_power WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
Description:

       Creates the real matrix representing the basis of ord and its inverse.
       This routine assumes that the basis is power basis of an equation
       order. First the zeroes of the generating polynomial are computed.
       They are then sorted according to KANT convention.
 

Calling sequence:
 
	order_basis_real_create_power(ord);
 
      	order         ord      = t_handle of order 

History:
 
	91-10-10 JS    minor changes
	91-10-01 JS    first version
  
*******************************************************************************/
{
	block_declarations;
 
	anf		field;
	integer_small	deg;
	integer_small	i, j, ind, r1, r2, r12;
	t_handle		reals;
	t_real		one, sq2, zero;
	anf_elt		rho, temp1, temp2;
	t_poly	pol;
	matrix		mat;
 
                                 
	reals = order_reals(ord);
	sq2   = order_sqrt_2(ord);
	field = order_anf(ord);
	pol   = order_poly(ord);
 
/*
    which number has the polynomial in the field table?
*/ 
	ind = 1;
	while (pol != anf_poly_z_poly(field, ind)) ind++;
 
/*
    Assuring precision of zeroes
*/
	anf_poly_zeroes(field, reals, ind);
 
/*
    Now we definitely know the signature
*/        
	deg   = order_abs_degree(ord);
	r1    = order_r1(ord);
	r2    = order_r2(ord);
	r12   = r1 + r2;
 
	mat = mat_new(deg, deg);
 
/*
   initializing the first column (=1)
*/                                   
 
	temp1 = anf_elt_con(ord, 1);
	anf_elt_to_mat_order_col(ord, temp1, mat, 1);
	anf_elt_delete(ord, &temp1);
	
/*
   initializing rho, this will contain the roots of the polynomial
*/

	anf_con_alloc(rho, deg);

	for (i=1; i<=r1; ++i)
		anf_con(rho, i) = real_incref(anf_poly_z_zero(field, ind, i));
 
	for (i=r1+1; i<=deg; ++i)
	{
		anf_con(rho, i) = real_mult(reals, sq2, 
						anf_poly_z_zero(field, ind, i));
	}
 
	anf_elt_to_mat_order_col(ord, rho, mat, 2);
 
	temp1 = anf_elt_incref(rho);
 
	for (j=3; j<=deg; ++j)
	{
		temp2 = anf_mult(ord, rho, temp1);
		anf_elt_to_mat_order_col(ord, temp2, mat, j);
		anf_elt_delete(ord, &temp1);
		temp1 = temp2;
	}
 
	anf_elt_delete(ord, &temp1);
	anf_elt_delete(ord, &rho);
 
	order_basis_real(ord) = mat;
 
	if (anf_print_level > 4)
	{
		printf("Real basis and its inverse:");
		mat_real_write(reals, mat);
	}
 
	mat_fld_inverse_sub(reals, mat, &order_basis_real_inv(ord));
 
	if (anf_print_level > 4)
		mat_real_write(reals, order_basis_real_inv(ord));
 
 
	return;
}
 
 
 
void
order_basis_real_create_trans WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
Description:

       Creates the real matrix representing the basis of ord and its inverse.
       This routine assumes that the basis is given via a transformation matrix
       referring to another order.
       The computation is done straightforward, using the transformation matrix.
       If necessary, order_real_create is called for the suborder.

Calling sequence:
 
	order_basis_real_create_trans(ord);
 
      	order         ord      = t_handle of order 

 
History:
        92-05-08 MJ    Output of order_basis_reals(ord)
	92-03-12 JS    real_delete(&den);
	91-10-10 JS    minor changes
	91-10-01 JS    first version  
        
 
*******************************************************************************/
{
	block_declarations;
 
	integer_small	deg, degsq, i;
	t_real		den;
	t_handle		reals;
	order		subord;
	matrix		mat1, mat2, mat3, sreal, sreali;
 
                                 
	deg    = order_abs_degree(ord);
	degsq  = deg*deg;
	reals  = order_reals(ord);
	subord = order_suborder(ord);
 
 
/*
    The suborder real data must exist and have the same precision
*/
	if (order_real_prec(subord) < order_real_prec(ord))
	{
		order_reals_copy(ord, subord);
		order_basis_real_create(subord);
	}
 
	sreal  = mat_real_to_mat_real(order_reals(subord), 
	 			      order_basis_real(subord), reals);
	sreali = mat_real_to_mat_real(order_reals(subord), 
				      order_basis_real_inv(subord), reals);
 
/*
    First the matrix itself
*/
 
	mat2 = mat_real_mat_z_mult(reals, sreal, order_tran(ord));
 
	if(order_tran_den(ord) == 1)
	{
		order_basis_real(ord) = mat2;
	}
	else
	{
		den = conv_int_to_real(reals, order_tran_den(ord));
		mat3 = mat_new(deg, deg);
 
		for (i=1; i<=degsq; ++i)
		{
			mat_entry(mat3, i) = 
				real_divide(reals, mat_entry(mat2, i), den);
		} 
 
		order_basis_real(ord) = mat3;
		mat_delref(reals, &mat2);       
                real_delete(&den);
	}
 
	if (anf_print_level > 4)
	{
		printf("Real basis and its inverse:");
		mat_real_write(reals, order_basis_real(ord));
	}
 
 
/*
    Now its inverse
*/
 
	mat1 = mat_z_to_mat_real(reals, order_invtran(ord)); 
 
	order_basis_real_inv(ord) = mat_ring_mult(reals, mat1, sreali);

	if (anf_print_level > 4)
		mat_real_write(reals, order_basis_real_inv(ord));
 
	mat_delref(reals, &mat1);
	mat_delref(reals, &sreal);
	mat_delref(reals, &sreali);
 
	return;
	
} 
