/******************************************************************************
  mat_real_orthonormal_expand.c
******************************************************************************/
#include "kant.h"

#define EPS 1E-1

t_logical
mat_real_orthonormal_expand WITH_5_ARGS(
	t_handle,	rrng,
	matrix,		mat,
	vector,		vec,      
	t_real,		eps,
	matrix*,	matn
)
/******************************************************************************
 
Description:	Tests whether a vector vec lies inside <b1,...,bk>.
		b1,...,bk are the columns of the matrix mat and must be
		orthonormal. If v does not lie inside <b1,...,bk> the program
		returns an orthonormal basis of <b1,...,bk,w>.
 
Calling sequence:
 
	Bool = mat_real_orthonormal(rrng,mat,vec,eps,&matn)

	t_logical 	Bool	: TRUE	v in <b1,...,bk>
				  FALSE	else				
	t_handle	rrng	: the (real) ring of the coefficients
	matrix		mat	: b1,...,bk
	vector		vec	: v
	t_real		eps	: the program computes the orthonormal projection
				  of vec in <b1,...,bk>. If the norm of this 
				  projection gets less than eps it assumes
				  "v inside <b1,...,bk>". The user does not need
				  to specify eps (eps = MEM_NH). In this 
				  case the program sets up eps to a default.
	matrix		matn	: orthonormal basis of <b1,...,bk,v> if Bool gets TRUE
 
History:
 
	92-06-29 KW	eps + real_zero_eps
	92-06-03 KW	written
 
******************************************************************************/
{
	block_declarations;

	t_logical	Bool;
	t_real		tempu,tempv,tempw;
	vector		u,v,w;

	if (!mat)
	{
		tempu = vec_ring_dot_product(rrng,vec,vec);
		if (real_zero_eps(rrng,tempu,eps))
		{
			Bool = FALSE;
		}
		else
		{
			Bool = TRUE;
			tempv = real_inverse(rrng,tempu);
			tempw = real_sqrt(rrng,tempv);
			u = mat_ring_scalar_left_mult(rrng,tempw,vec);
			*matn = mat_ring_col_append(rrng,mat,u);
			vec_delete(rrng,&u);
			real_delete(&tempv);					
			real_delete(&tempw);					
		}
	}
	else
	{
		u = vec_real_orthogonal_proj(rrng,mat,vec);
		v = mat_ring_subtract(rrng,vec,u);
		tempu = vec_ring_dot_product(rrng,v,v);
		if (real_zero_eps(rrng,tempu,eps))
		{
			Bool = FALSE;
		}
		else
		{
			Bool = TRUE;
			tempv = real_inverse(rrng,tempu);
			tempw = real_sqrt(rrng,tempv);
			w = mat_ring_scalar_left_mult(rrng,tempw,v);
			*matn = mat_ring_col_append(rrng,mat,w);
			real_delete(&tempv);
			real_delete(&tempw);
			vec_delete(rrng,&w);
		}
		vec_delete(rrng,&u);
		vec_delete(rrng,&v); 
	}
	real_delete(&tempu);

	return(Bool);
}
