#include "kant.h"
#include "integer.e"
#include "real.e"
#include "anf.h"
#include "conv.e"

/*
 
This file contains:
 
anf_add
anf_z_add
anf_con_add
 
*/
 

anf_elt
anf_add WITH_3_ARGS(
	order,		ord,
	anf_elt,	alpha,
	anf_elt,	beta
)
/*******************************************************************************
 
anf_add.c
 
JS August 1991           
Last modification: 11.10.91
 
Addition of two algebraic numbers with coefficients over an arbitrary order
 
The summation of the coefficients (referring to the basis of the order ord) 
is executed straightforward. The special cases that one of the submitted numbers
is a rational integer is handled separately. In the general case care is also
is taken of the denominators.
  
If the anf_elts submitted are conjugate vectors anf_con_add is called.
 
If the coefficient order is Z the routine anf_z_add is called.
 
For the reason of speed this routine doesn't care for simplifying 
coefficients/denominator. Call anf_simplify if denominators could occur.
 
So if a non-integer is submitted the result will definitely be a non-integer.

*******************************************************************************/
{
	block_declarations;
 
	anf_elt		gamma;
	integer_small	deg;
        order		ordcoef;
        integer_big	denom;	
        integer_big	multa, multb;
	anf_elt		temp, tempaa, tempbb;
	integer_small	onepos;
	integer_small	i;
        
/*
    Special case: beta (and alpha?) is integer
*/
	if (anf_elt_is_integer(beta)) 
	{	
        	if (anf_elt_is_integer(alpha)) return integer_add(alpha, beta);
		temp  = beta;
		beta  = alpha;
		alpha = temp;
	}
  

/*
    Is beta a conjugate vector?
*/
 
	if(anf_elt_is_con(beta))
		return anf_con_add(ord, alpha, beta);

/*  
    Is the coefficient order equal Z? 
*/
 
        ordcoef = order_coef_order(ord);
	if (ring_type(ordcoef) == RING_Z) return anf_z_add(ord, alpha, beta);
 
       	deg = order_rel_degree(ord);
       	anf_elt_alloc(gamma, deg);
 
/*
    Special case: only alpha is integer
*/
        
	if (anf_elt_is_integer(alpha))
        {
		onepos = order_one_position(ord);
		if (onepos <= 0)
                        error_internal("anf_add: One not in basis.");
		else
		{
		        denom = anf_elt_den(beta);
			anf_elt_den(gamma) = integer_incref(denom);
                        for (i=1; i<=deg; ++i)
                        {
                           	if (i==onepos) continue;
				temp = anf_elt_coef(beta, i);
				anf_elt_coef(gamma,i) = anf_elt_incref(temp);
 			}
			temp = (denom==1) 
			     ? alpha 
			     : integer_mult(denom,alpha);
                        anf_elt_coef(gamma, onepos) = 
                        anf_add(ordcoef, temp, anf_elt_coef(beta,onepos));
			if (denom != 1) anf_elt_delete(ordcoef, &temp);
			return gamma;
		}
	}
  
/*
    General case: Both are not integers
*/ 
    
	denom = integer_lcm(anf_elt_den(alpha), anf_elt_den(beta));
	anf_elt_den(gamma) = denom;
 
	multa = integer_div(denom, anf_elt_den(alpha));
	multb = integer_div(denom, anf_elt_den(beta));
 
	for (i=1; i<=deg; ++i)
	{
		tempaa = anf_mult_z(ordcoef, anf_elt_coef(alpha, i), multa);
		tempbb = anf_mult_z(ordcoef, anf_elt_coef(beta,  i), multb);
		anf_elt_coef(gamma, i) = anf_add(ordcoef, tempaa, tempbb);
	        anf_elt_delete(ordcoef, &tempaa);
        	anf_elt_delete(ordcoef, &tempbb);
	}
 
	integer_delref(multa);
	integer_delref(multb);
 
	return gamma;
}

  
 


anf_elt
anf_z_add WITH_3_ARGS(
	order,		ord,
	anf_elt,	alpha,
	anf_elt,	beta
)
/*******************************************************************************
 
anf_z_add.c
 
JS August 1991
Last modification: 25.09.91
 
Addition of two algebraic numbers with coefficients over Z
The user should call the generic routine anf_add.
 
The summation of the coefficients (referring to the basis of the order ord) 
is executed straightforward. The special cases that one of the submitted numbers
is a rational integer is handled separately. In the general case also care 
is taken of the denominators.
 
For the reason of speed this routine doesn't care for simplifying 
coefficients/denominator. Call anf_simplify if denominators could occur.

*******************************************************************************/
{
	block_declarations;
 
	anf_elt		gamma;
	integer_small	deg;
        integer_big	denom;	
        integer_big	temp, tempa, tempb, multa, multb;	
	anf_elt		tempaa;
	integer_small	onepos;
	integer_small	i;
  
/*
    Special case: beta (and alpha?) is integer
*/
	if (anf_elt_is_integer(beta)) 
	{	
        	if (anf_elt_is_integer(alpha)) return integer_add(alpha, beta);
		tempaa = beta;
		beta   = alpha;
		alpha  = tempaa;
	}

       	deg = order_rel_degree(ord);
       	anf_elt_alloc(gamma, deg);
 
/*
    Special case: only alpha is integer
*/
        
	if (anf_elt_is_integer(alpha))
        {
	   	onepos = order_one_position(ord);
	   	if (onepos <= 0)
		 	error_internal("anf_add: One not in basis.");
               	else
	 	{
	   		denom = anf_elt_den(beta);
			anf_elt_den(gamma) = integer_incref(denom);
                        for (i=1; i<=deg; ++i)
                        {
                        	if (i==onepos) continue;
				temp = anf_elt_coef(beta, i);
				anf_elt_coef(gamma,i) = integer_incref(temp);
 		        }
			temp = (denom==1) 
			       ? alpha 
			       : integer_mult(denom,alpha);
                        anf_elt_coef(gamma, onepos) = 
                              integer_add(temp, anf_elt_coef(beta, onepos));
			if (denom !=1 ) integer_delref(temp);
		        return gamma;
		}
	}
  
/*
    General case: Both are not integers
*/ 
   
	denom = integer_lcm(anf_elt_den(alpha), anf_elt_den(beta));
	anf_elt_den(gamma) = denom;
 
	multa = integer_div(denom, anf_elt_den(alpha));
	multb = integer_div(denom, anf_elt_den(beta));
 
	for (i=1; i<=deg; ++i)
	{
		tempa = integer_mult(anf_elt_coef(alpha, i), multa);
		tempb = integer_mult(anf_elt_coef(beta, i), multb);
		anf_elt_coef(gamma, i) = integer_add(tempa, tempb);
		integer_delref(tempa);
		integer_delref(tempb);
	}
 
	integer_delref(multa);
	integer_delref(multb);
 
	return gamma;
}
 
 
 
 


anf_elt
anf_con_add WITH_3_ARGS(
	order,		ord,
	anf_elt,	alpha,
	anf_elt,	beta
)
/*******************************************************************************
 
anf_con_add.c
 
JS October 1991
Last modification: 11.10.91
 
Addition of two conjugate vectors.
The user should call the generic routine anf_add.
 
The summation of the conjugates is done componentwise.

*******************************************************************************/
{
	block_declarations;
 
	anf_elt		gamma;
	integer_small	deg, r1, r2, r12;
	anf_elt		tempaa;
	t_handle		reals;
	t_real		ralpha, temp1;
	integer_small	i;
  
/*
    Special case: beta (and alpha?) is integer
*/
	if (anf_elt_is_integer(beta)) 
	{	
        	if (anf_elt_is_integer(alpha)) return integer_add(alpha, beta);
		tempaa = beta;
		beta   = alpha;
		alpha  = tempaa;
	}

       	deg = order_abs_degree(ord);
	r1  = order_r1(ord);
	r2  = order_r2(ord); 
	r12 = r1 + r2;
 
	reals = order_reals(ord);
 
       	anf_con_alloc(gamma, deg);
 
/*
    Special case: only alpha is integer
*/
        
	if (anf_elt_is_integer(alpha))
        {
		ralpha = conv_int_to_real(reals, alpha);
 
		for (i=1; i<=r1; ++i)
		{
			anf_con(gamma, i) = 
				real_add(reals, anf_con(beta, i), ralpha);
		}
 
 		if(r2) 
		{
			temp1 = real_mult(reals, ralpha, order_sqrt_2(ord)); 
 
			for (i=r1+1; i<=r12; ++i)
			{
				anf_con(gamma, i) = 
				real_add(reals, anf_con(beta, i), temp1);
			}
 
			for (i=r12+1; i<=deg; ++i)
			{
				anf_con(gamma, i) = 
				real_incref(anf_con(beta, i));
			}
 
			real_delete(&temp1);
		}
 
		real_delete(&ralpha);
 
	}
 
/*
    General case: addition componentwise
*/
	else
	{  
		for (i=1; i<=deg; ++i)
		{
			anf_con(gamma, i) = real_add(reals, anf_con(alpha, i),
							    anf_con(beta,  i));
		}
	}
 
	return gamma;
}
