/**************************************************************************
This file contains a set of functions to implement hashing.
void *hash_create( name, size, rev_size);
	Creates (initialises) a new hashtable to store one set of objects.
	It returns a pointer to such a hash table. The internal structuring
	of these hash tables is hided for the rest of the program: The
	data types are only locally defined, and the pointer returned
	is of type (void *).

int hash_add( table, name, data);
	Adds a new (name, data) pair to the hash storage.
	All names in a single hash table need to be unique.
	The data item associated with the name is an integer or a pointer
	to some other structure in the program.

int hash_get( table, name, data);
	retreives the data value for a given name.
	If the data values are integers, it is capable of creating a new entry
	for a not yet existing name with a new unique integer.

int hash_delete( table, name);
	Remove a (name data) pair from the hash structure.

char *hash_rev( table, data);
	A reverse query valid for integer data >=0 only:
	for a given data value, return the associated name.

void *hash_scan( table, restart);
	Implements a serial scan over all stored entries in a given table,
	and returns at each call the next data value,
	finally followed by 0.

void hash_explode( table_pointer, object_handler, arg)
	removes an entire hash table.
        All hash_entry structs are saved for later use.
        The  base hashtable struct is freed, and the provides pointer made zero.
	An object_handler can be provided, which will be called for each entry.
        It is called with 2 arguments: (arg, entry->data)

	Jos van Eijndhoven (jos@es.ele.tue.nl),  29-1-1991
	ANSI version 10-2-92
	Eindhoven Univ. of Technology,  The Netherlands
**************************************************************************/
#include <string.h>
#include <stdlib.h>
#include "defines.h"
#include "main.h"
#include "mem_alloc.h"

#ifdef Hash_table
#undef Hash_table
#endif

/***** imported global variables ******/

#define MAX_REVDATA 10000 /* limit max size (data value) for reverse table*/
#define N_ALLOC     256   /* bunch of hash elemnts to allocate together */

/***** hashing types available for this file only ****/
typedef struct hash_entry
{	struct hash_entry	*next;
	char			*name;
	void			*data;
} Hash_entry;

typedef struct hash_table
{	char		*name; /* basically for debugging only */
	int		size;  /* prime numbers are probably best */
	int		rev_size;
	int		highest;
	int		last_scan_index;
	Hash_entry	*last_scan_entry;
	Hash_entry	**hash;
	char		**rev;
} Hash_table;



/*** forward declarations ***/
static int hash_val( char *name, int size);
static Hash_entry *new_entry();
static void free_entry( Hash_entry *h);

/***** Creates hash-table **********************************************
	'size' is the length of the (open) hash array.
        This is preferrable a prime number, somewhat larger than the
        maximum number of entries expected.
	'revsize' is the initial size of the reverse table,
	used only if you hash names against (non-negative) ints,
	as opposed to object (void) pointers.
	(Usefull for instance to map language keywords to a small int)
	It should have a decent value suitable for most runs,
 	will be automaticaly enlarged if necessary.
	The 'name' of the hash table is probably usefull for
	error message printing only.
*/
export Hash_table_p hash_create( char *name, int size, int rev_size)
{
	Hash_table *table;

      	table = newstruct( Hash_table);
	table->name = name;
	table->size = size;
	table->rev_size = rev_size;
	table->hash = newarray( size, Hash_entry *);
	if (rev_size)
		table->rev = newarray( rev_size, char *);
	else	table->rev = NULL;
	return (table);
}

/***** hash store function **********************************************/
/***** locates named item, if found returns 1 ***************************/
/*****         else returns 0 and creates entry *************************/
export int hash_add( Hash_table_p table, char *name, void *data)
{
	int value, idata;
	Hash_entry *h, *new_entry();

	/**** locate entry ****/
	value = hash_val(name, table->size);
	for (h=table->hash[value]; h; h = h->next)
		if (!strcmp( h->name, name)) break;

	if (h)	/* entry found */
		return( 1);
	else
	{	
		h = new_entry();
		h->name = name;
		h->data = data;
		h->next = table->hash[value];
		table->hash[value] = h;
		if (table->rev)
		{	idata = (int)(data);
			if (idata < 0 || idata >= MAX_REVDATA)
			{	error( "BUG: "
				    "hash() reverse table: illegal data %d!\n",
				    idata);
			}
			if (idata >= table->rev_size) /* increase storage */
			{
				table->rev_size = (int)(idata * 1.5);
				table->rev = (char **) myrealloc(
				    table->rev, table->rev_size*sizeof(char *));
			}
			table->rev[idata] = h->name;
			if (idata > table->highest) table->highest = idata;
		}
		return(0);
	}
}

/***** hash retrieve function ********************************************/
/***** locates named item, if found exports data and returns 1 ***********/
/***** else returns 0 and creates entry if rev enabled *******************/
/***** if data == 0, it just looks for the existence of the entry ********/
export int hash_get( Hash_table_p table, char *name, void **data)
{
	int value, *idata;
	Hash_entry *h, *new_entry();

	/**** locate entry ****/
	value = hash_val(name, table->size);
	for (h=table->hash[value]; h; h = h->next)
		if (!strcmp( h->name, name)) break;

	if (h)	/* keyword found */
	{	if (data) *data = h->data;
		return( 1);
	} else
	{	if (table->rev && data) /* allowed to create new data (number)*/
		{	idata = (int *)(data);
			h = new_entry();
			h->name = name;
			table->highest++;
			*data = h->data = (void *)(table->highest);
			h->next = table->hash[value];
			table->hash[value] = h;
			if (*idata >= MAX_REVDATA)
			{	error(
				"BUG: hash() reverse table: illegal data %d!\n",
				*idata);
			}
			if ((*idata) >= table->rev_size) /* increase storage */
			{
				table->rev_size = (int)( *idata * 1.5);
				table->rev = (char **) myrealloc(
				    table->rev, table->rev_size*sizeof(char *));
			}
			table->rev[*idata] = h->name;
		}
		return(0);
	}
}

/***** reverse query: fetch name forgiven (integer) data ****************/
export char *hash_rev( Hash_table_p table, int data)
{
	if (!table->rev)
		error( "BUG: hash_rev(%s) called on non-rev table!\n",
			table->name );

	if (data < 0 || data >= table->rev_size)
		error( "BUG: hash_rev(%s) illegal data argument %d!\n",
			table->name, data);

	return( table->rev[data]);
}

/***** scan entries in table one by one *********************************/
/***** If restart != 0, the the first item is returned. *****************/
/***** then returns one by one all stored entries, NULL after the last **/
export void *hash_scan( Hash_table_p table, int restart)
{
	int i;
	Hash_entry *e;

	if (restart)  /* start at front */
	{	table->last_scan_index = 0;
		e = NULL;
	}
	else
	{	e = table->last_scan_entry;
		if (!e) /* was at end of table before, unexpected call....*/
			return( NULL);
		/* look for next element in linked list */
		e = table->last_scan_entry = e->next;
		if (e) /* next element in list found */
			return( e->data);
		/* else move to next hash index */
		table->last_scan_index++;
	}

	/* now search for new (non empty) index in hash array */
	/* now e == NULL ! */
	for (i = table->last_scan_index; i<table->size; i++)
		if (e = table->hash[i]) break;

	table->last_scan_index = i;
	table->last_scan_entry = e;

	if (e)	return( e->data);
	else	return( NULL);
}

/***** delete table entry ***********************************************/
export int hash_delete( Hash_table_p table, char *name)
{
	int value;
	Hash_entry *h, **hp;

	value = hash_val(name, table->size);
	for (hp = table->hash+value; h = *hp; hp = &(h->next))
		if (!strcmp( h->name, name)) break;

	if (h)	/* entry found */
	{	*hp = h->next;
		free_entry( h);
		return(1);
	} else
	{	return(0);
	}
}

/***** remove entire table ***********************************************/
export void hash_explode( Hash_table_p table, void (*handler)( char *, void *), char *arg)
{
	int i;
	Hash_entry *e;

	if (!table) return;

	if (table->hash)
	{	for (i=0; i<table->size; i++)
		{	while (e=table->hash[i])
			{	table->hash[i] = e->next;
				if (handler) (*handler)( arg, e->data);
				free_entry(e);
			}
		}
		free( table->hash);
	}

	if (table->rev) free( table->rev);
	free( table);
}

/***** compute hash value of name ***************************************/
static int hash_val( char *name, int size)
{
	long value;
	char *c;

	value = (long)(name[0]); 

	for (c = name+1; *c; c++)
	{	value <<= 1;              /* shift 1 bit */
		if (value < 0) value++;   /* turn shift into rotate...*/
		value += (long)(*c);      /* and add character value */
	}

	value %= size;
	if (value<0) value += size;

	return( (int)value);
}

/***** creates a new hash entry, speeded up by own alloc-free scheme ****/
static Hash_entry *free_entry_list;

static Hash_entry *new_entry()
{
	Hash_entry *h;
	int i;

	if (!free_entry_list)
	{	free_entry_list = newarray( N_ALLOC, Hash_entry);
		for(i=0; i<N_ALLOC-1; i++)
			free_entry_list[i].next = free_entry_list + i + 1;
		free_entry_list[N_ALLOC-1].next = NULL;
	}

	h = free_entry_list;
	free_entry_list = h->next;
	memset( h, '\0', sizeof( Hash_entry));
	return(h);
}

static void free_entry( Hash_entry *h)
{
	h->next = free_entry_list;
	free_entry_list = h;
}
