;;; 
;;; This file contains the predicates, selectors, and constructors
;;; necessary to provide a layer of abstraction
;;;
;;; Derek Lindner  buddha@theory.lcs.mit.edu
;;; Justin Liu     dondon@theory.lcs.mit.edu
;;; Brian So       brianso@theory.lcs.mit.edu
;;;


(define (tagged-list? exp tag)
  (and (pair? exp)
       (eq? (car exp) tag)))

; LAMBDA

(define lam-formals cadr)

(define lam-body caddr)

(define (lambda? exp)
  (tagged-list? exp 'lambda))

(define (make-lambda formallist body)
  (cons 'lambda (list formallist body)))

; COMBINATIONS

(define (combination? exp)
  (and (pair? exp) 
       (not (or (keyword? (car exp)) (define? exp)))))

(define (make-combination operator operandlist)
  (cons operator operandlist))

(define first car)

(define rest cdr)

(define operator car)

(define operands cdr)

; BETA-REDEX

(define (beta-redex? exp)
  (and (pair? exp) (lambda? (first exp))))

(define paramb cadar)

(define randb cdr)

(define bodyb caddar)

(define lambdab car)

; LET AND LETREC

(define (let? exp)
  (tagged-list? exp 'let))

(define (named-let? exp)
  (and (let? exp)
       (>= (length exp) 4)
       (variable? (cadr exp))
       (list? (caddr exp))))

(define (letrec? exp)
  (tagged-list? exp 'letrec))

(define (let*? exp)
  (tagged-list? exp 'let*))

(define (make-let bindinglist body)
  (cons 'let (list bindinglist body)))

(define (make-let* bindinglist body)
  (cons 'let* (list bindinglist body)))

(define (make-named-let name bindinglist body)
  (cons 'let (list name bindinglist body)))

(define (make-letrec bindinglist body)
  (cons 'letrec (list bindinglist body)))

(define bindings-letc cadr)

(define body-letc caddr)

(define make-binding list)

(define binding-variable-letc car)

(define binding-init-letc cadr)

(define (make-bindings-letc variablelist initlist)
  (map make-binding variablelist initlist))

(define (binding-variable-list bindings)
  (map car bindings))

(define (binding-init-list bindings)
  (map cadr bindings))

(define first-binding car)

(define rest-binding cdr)

(define (first-binding-init bindings)
  (cadr (first-binding bindings)))

(define (first-binding-var bindings)
  (car (first-binding bindings)))

(define (delete-bindings remvlist bindings) ;remvlist is a list of VARIABLES
  (cond ((null? bindings) '())
	((memq (first-binding-var bindings) remvlist)
	 (delete-bindings remvlist (rest-binding bindings)))
	(else (cons (first-binding bindings)
		    (delete-bindings remvlist
				     (rest-binding bindings))))))

; IF

(define (if? exp)
  (tagged-list? exp 'if))

(define (make-if if-clauses)  
  (cons 'if if-clauses))

(define test-if cadr)

(define consequent-if caddr)

(define (alternative-if ifexp)
      (cadddr ifexp))

; COND

(define (cond? exp)
  (tagged-list? exp 'cond))

(define (make-cond clauses)
  (cons 'cond clauses))

(define clauses-cond cdr)

(define make-clause-cond list)

(define (test-clause clause)            ;REVISED was "test-part"
      (car clause))

(define (consequent-clause clause)      ;REVISED was "expression-part"
  (cadr clause))

(define (clause? seq)                    ;REVISED
  (and (pair? seq)
       (list? seq)
       (if (eq? (car seq) 'else)
           (if (cdr seq) #t #f)
           #t)))

;(define (expression-part clause)
;  (cond ((= (length clause) 1) ; in case of tests w/o consq
;	 '())
;	((= (length clause) 2)
;	 (cadr clause))
;	((> (length clause) 2)
;	 (cons 'begin (cdr clause)))))

(define first-clauses car)

(define rest-clauses cdr)

(define (last-clause? clauses)
  (null? (rest-clauses clauses)))

(define (last-clause clauses)
  (car (reverse clauses)))

; AND and OR

(define (or? exp)
  (tagged-list? exp 'or))

(define (and? exp)
  (tagged-list? exp 'and))

(define (make-or clauses)
  (cons 'or clauses))

(define (make-and clauses)
  (cons 'and clauses))

(define (quote? exp)
  (tagged-list? exp 'quote))

; MAP

(define (map-application? exp)
  (tagged-list? exp '<<map>>))

; BEGIN

(define (sequence? exp)
  (tagged-list? exp 'begin))

(define expression-sequence cdr)

(define (make-sequence expressionseq)
  (cons 'begin expressionseq))          ;REVISED

;(define (last-seqexp? expressionseq)           ;REVISED: deleted
;  (null? (rest expressionseq)))

;(define (last-seqexp expressionseq)           ;REVISED: deleted
;  (car (reverse expressionseq)))

; APPLY

(define (apply? exp)
  (tagged-list? exp 'apply))

(define (make-apply proc lst)
  (list 'apply proc lst))

(define procedure-apply cadr)

(define arglist-apply caddr)


; DEFINE
(define (define? exp)
  (tagged-list? exp 'define)) 

(define (make-define var-and-formals body)
  (list 'define var-and-formals body))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                                                      ;
; Some utility functions on lists/sets.                                ;
;                                                                      ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define emptyset '())                   ;REVISED: new

(define emptyset? null?)                   ;REVISED:new

(define member-of member)                   ;REVISED:new

(define                   ;REVISED
 (union L1 L2)
  (cond ((emptyset? L1) L2)
	((emptyset? L2) L1)
	((member-of (car L1) L2) (union (cdr L1) L2))
	(else (cons (car L1) (union (cdr L1) L2)))))

(define (difference-set L1 L2) ; returns L1-L2               ;REVISED
  (cond ((emptyset? L1) emptyset)
	((emptyset? L2) L1)
	((member-of (car L2) L1) (difference-set (delete (car L2) L1) (cdr L2)))
	(else (difference-set L1 (cdr L2)))))

(define (intersect L1 L2)                   ;REVISED
  (cond ((or (emptyset? L1) (emptyset? L2)) emptyset)
	(else (let intersect-help ((foo L1) (bar L2))
		(cond ((emptyset? foo) emptyset)
		      ((member-of (car foo) bar)
		       (cons (car foo) (intersect-help
					(cdr foo)
					bar)))
		      (else (intersect-help (cdr foo) bar)))))))

(define (accunion sets)
  (accumulate union emptyset sets))

(define (accumulate combiner init-value alist)
  (if (null? alist)
      init-value
      (combiner (car alist)
		(accumulate combiner init-value (cdr alist)))))

(define (mapunion pr ls)
  (accunion (map pr ls)))

(define (same? list1 list2)             ;equal sets, assuming no repeats in lists
  (= (length list1) (length list2) (length (union list1 list2))))
