;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                    ;;;
;;; Garbage Collection ;;;
;;;                    ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (garbage-collect exp)           ;does top-level letrec only
  (if (letrec? exp)
      (clean (bindings-letc exp) (body-letc exp))
      exp))

(define (clean bindings body)         ;cleans one letrec frame
  (cond ((null? bindings)
         (if (not (same-previous-step? "GC"))
             (record-step "GC" 'empty-binding))
         body)
        (else
         (let ((dead-vars (dead-variables bindings (free-variables body))))
           (if dead-vars
               (let* ((newbinds (delete-bindings dead-vars bindings))
                      (string-of-deads
                       (string-append
                        "("
                        (apply string-append
                               (map
                                (lambda (var) (string-append (symbol->string var) " "))
                                dead-vars))
                        ")")))
                 (record-step "GC" (string->symbol string-of-deads))
               (if (null? newbinds)
                   body
                   (make-letrec newbinds body)))
           (make-letrec bindings body))))))

; LIVE-VARS returns a list of all live variables
; as defined in our CORRECTED rules

(define (live-variables bindings initially-live)
  (let loop ((alive emptyset)
             (new-lives (union
                         initially-live
                         (vars-bound-nonvalues bindings))))
    (let* ((newer-lives
            (mapunion
             (lambda (x)
               (let ((binding (assq x bindings)))
                 (if binding
                     (free-variables (binding-init-letc binding))
                     emptyset)))
             new-lives))
           (alive (union alive new-lives))
           (new-lives (difference-set newer-lives alive)))
      (if new-lives
          (loop alive new-lives)
          alive))))

(define (dead-variables bindings initially-live)
  (difference-set (binding-variable-list bindings)
                  (live-variables bindings initially-live)))

(define (vars-bound-nonvalues bindings)
  (append-map
   (lambda (bind)
     (if (value? (binding-init-letc bind))
         '()
         (list (car bind))))
   bindings))

