/* comb3d -- perform a pixelwise combination of a set of 3D images      */
/*           Each of N 3D images is specifed by a basename which is a   */
/*           common root name for a set of 2D images in a directory     */
/*           such that the set of 2D images in alphanumerical order     */
/*           form the 3D image. There must be 1 basename specified for  */
/*           each 3D image to be included in the combination. If only   */
/*           one file name is supplied after -i, that file must         */
/*           contain a list of N basenames for the N 3D images. An      */
/*           output basename must be supplied after -o. A directory     */
/*           path can be part of a basename.                            */
/*                                                                      */
/* operations: average (avg)                                            */
/*             minimum (min)                                            */
/*             maximum (max)                                            */
/*             median  (med)                                            */
/*             rank or order statistic (os) nnn (rank 1 == max)         */
/*             ordinary linear combination (olc) coefficients a1 ... aN */
/*             rank-ordered linear comb. (rlc) coefficients a1 ... aN   */


/* version 1.0     11 Aug 1992 */

/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */ 
/* Nashville, TN 37235                             */ 
/* rap2@vuse.vanderbilt.edu                        */ 
/*                                                 */ 
/* This software is freely redistributable if      */ 
/* the author's name and affiliation are included. */


#include <math.h>		
#include <stdio.h>
#include <strings.h>
#include <values.h>
#include <sys/types.h>
#include "rasterio.h"
#include "comb3d.h"


main( argc, argv ) 
   unsigned int argc;    /* count of arguments parsed by operating system    */
   char *argv[];         /* pointers to arguments parsed by operating system */
   {
   FILE *fptr;
   char filename[80];
   char compname[80];
   char **baselist;
   char *outbase;
   char *slash;
   byte **image;
   float *coefflist;
   float *c;
   float f;
   struct rasterfile H; /* rasterfile headers */
   byte *cmap;          /* colormaps */
   byte *I,*O;          /* image pointers */
   byte *u,*v,*w,*z;   	/* image pointers */
   int X,Y,N;           /* image dimensions */
   byte *pix;
   int i,j,k;           /* indices */
   int r,s;
   int IndexNum;
   int frames;
   int nimgs;
   int ncoeffs;
   int n = argc;
   int mode;
   int rank;
   int done;

   char Index[IXLGTH+2];
   char Suffix[SXLGTH+1];
   char Format[FTLGTH];



   if (argc < 7)
      {
      fprintf(stderr,
         "usage: comb3d -i basename_file | basename1 ... basenameN\n\
              -o output_basename  [-s nnn]  [-n nnn]\n\
              -c avg | min | max | med | os nnn | olc[|rlc] a1 ... aN\n");
      exit(0);
      }

   baselist = (char **)NULL;
   frames = -1;
   IndexNum = 1;
   mode = 0;
   rank = 0;

   /* get arguments */
   while ( --n )
      {
      if ( argv[n][0] == '-' )          /* then this is a switch */
         {
         if ( argv[n][1] == 'i' )       /* basenames follow */
            {
            if ( (n+2 >= argc) || (argv[n+2][0] == '-') ) /* then basenames */
               {                                          /* are in a file. */
               if ( !(fptr = fopen( argv[n+1], "r" )) )
                  {
                  fprintf(stderr,"Unable to open %s\n",argv[n+1]);
                  exit( 0 );
                  }
               nimgs = -1;
               while ( !feof(fptr) )
                  {
                  fscanf( fptr,"%s",filename );
                  ++nimgs;
                  }
               if ( nimgs < 2 )
                  {
                  fprintf(stderr,"There must be at least two basenames\n");
                  exit( 0 );
                  }
               else
                  {
                  fprintf(stderr,"%d basenames are specified.\n",nimgs);
                  }
               if ( !(baselist = 
                     (char **)calloc( nimgs, sizeof(char *) + NAMESIZE )) )
                  {
                  fprintf(stderr,"error allocating baselist\n");
                  exit( 0 );
                  }
               rewind(fptr);
               for ( i=0; i<nimgs; ++i)
                  {
                  *(baselist+i) = 
                   (char *)(baselist) + nimgs*sizeof(char *) + i*NAMESIZE;
                  fscanf( fptr, "%s",*(baselist+i) );
                  /*fprintf(stderr,"name[%d]: %s\n",i,*(baselist+i));*/
                  }
               fclose(fptr);
               }
            else /* basenames are on command line */
               {
               nimgs = 0;
               while ( (n+1+nimgs<argc) && (argv[n+1+nimgs][0] != '-') ) 
                 ++nimgs;
               if ( !(baselist = 
                     (char **)calloc( nimgs, sizeof(char *) )) )
                  {
                  fprintf(stderr,"error allocating baselist\n");
                  exit( 0 );
                  }
               for ( i=0; i<nimgs; ++i)
                  {
                  *(baselist+i) = argv[n+1+i];
                  /*fprintf(stderr,"name[%d]: %s\n",i,*(baselist+i));*/
                  }
               }
            }
         else if ( argv[n][1] == 'o' )       /* output basename follows */
            {
            outbase = argv[n+1];
            }
         else if ( argv[n][1] == 's' )       /* starting index follows */
            {
            IndexNum = atoi(argv[n+1]);
            }
         else if ( argv[n][1] == 'n' )       /* number of frames follows */
            {
            frames = atoi(argv[n+1]);
            }
         else if ( argv[n][1] == 'c' )       /* combination mode follows */
            {
            if ( !strcasecmp( argv[n+1], "avg" ) )
               {
               mode = MAVG;
               }
            else if ( !strcasecmp( argv[n+1], "min" ) )
               {
               mode = MMIN;
               }
            else if ( !strcasecmp( argv[n+1], "max" ) )
               {
               mode = MMAX;
               }
            else if ( !strcasecmp( argv[n+1], "med" ) )
               {
               mode = MMED;
               }
            else if ( !strcasecmp( argv[n+1], "os" ) )
               {
               mode = MOS;
               rank = atoi( argv[n+2] );
               }
            else if ( !strcasecmp( argv[n+1], "olc" ) )
               {
               mode = MOLC;
               ncoeffs = 0;
               while ( (n+2+ncoeffs<argc) && (argv[n+2+ncoeffs][0] != '-') ) 
                 ++ncoeffs;
               if ( !(coefflist = 
                     (float *)calloc( 1, ncoeffs*(sizeof(float)) )) )
                  {
                  fprintf(stderr,"error allocating coefficient list\n");
                  exit( 0 );
                  }
               for ( i=0; i<ncoeffs; ++i)
                  {
                  *(coefflist+i) = atof(argv[n+2+i]);
                  /*fprintf(stderr,"coeff[%d]: %f\n",i,*(coefflist+i));*/
                  }
               }
            else if ( !strcasecmp( argv[n+1], "rlc" ) )
               {
               mode = MRLC;
               ncoeffs = 0;
               while ( (n+2+ncoeffs<argc) && (argv[n+2+ncoeffs][0] != '-') ) 
                 ++ncoeffs;
               if ( !(coefflist = 
                     (float *)calloc( 1, ncoeffs*(sizeof(float)) )) )
                  {
                  fprintf(stderr,"error allocating coefficient list\n");
                  exit( 0 );
                  }
               for ( i=0; i<ncoeffs; ++i)
                  {
                  *(coefflist+i) = atof(argv[n+2+i]);
                  /*fprintf(stderr,"coeff[%d]: %f\n",i,*(coefflist+i));*/
                  }
               }
            }
         }
      }

   if ( !baselist || !outbase || !mode )
      {
      fprintf(stderr,"-i, -o, and -c must be specified\n");
      exit( 0 );
      }

   if ( ((mode == MOLC) || (mode == MRLC)) && (nimgs != ncoeffs) )
      {
      fprintf(stderr,"Number of basenames must equal number of coefficients\n");
      exit( 0 );
      }

   if ( mode == MMED )
      {
      rank = nimgs/2+1;
      }
   else if ( mode == MOS )
      {
      if ( (rank < 0) || (rank > nimgs) )
         {
         fprintf(stderr,"Rank must be between 1 and %d\n",nimgs);
         exit( 0 );
         }
      else if ( rank == 1 )
         {
         mode == MMAX;
         }
      else if ( rank == nimgs )
         {
         mode == MMIN;
         }
      }

      

   if ( !(image = (byte **)calloc( nimgs, sizeof(byte *) )) )
      {
      fprintf(stderr,"error allocating image pointer list\n");
      exit( 0 );
      }

   /* get size of images */

   /* write index field width in format string */
   sprintf( Format, "%%0%1dd", IXLGTH );

   /* create index field for output file name */
   Index[0] = '.';
   sprintf( Index+1, Format, IndexNum );

   /* make output file name from basename + index + suffix */
   strcpy( filename, *baselist);
   strcat( filename, Index );
   strcat( filename, ".ras" );

   /* try to open the file */
   fptr = OpenFile( filename, "", "r" );
   if (  ReadRasterFile( fptr, &H, &cmap, &I, 0, 0, 0, 0, ALLOC )  ) exit( 0 );
   fclose( fptr );

   /* get size info */
   X = H.ras_width;
   Y = H.ras_height;
   N = X*Y;

   /* allocate image buffers */

   *image = I;
   for ( i=1; i<nimgs; ++i)
      {
      if ( !(*(image+i) = (byte *)calloc( N, sizeof(byte)) ) )
         {
         fprintf(stderr,"error allocating image %d\n", i);
         exit( 0 );
         }
      }
   if ( !(O = (byte *)calloc( N, sizeof(byte)) ) )
      {
      fprintf(stderr,"error allocating output image \n");
      exit( 0 );
      }

   if ( !( (mode == MAVG)  || (mode == MOLC) ) )
      if ( !(pix = (byte *)calloc( nimgs, sizeof(byte)) ) )
         {
         fprintf(stderr,"error allocating pixel sort array \n");
         exit( 0 );
         }



   /* main loop */
   done = FALSE;
   while ( frames )
      {
      /* make the file name index */
      sprintf( Index+1, Format, IndexNum );

      /* get the images */
      for ( i=0; i<nimgs; ++i)
         {
         strcpy( filename, *(baselist+i) );
         strcat( filename, Index );
         strcat( filename, ".ras" );

         fptr = OpenFile( filename, "", "r" );
         if (  ReadRasterFile( fptr, &H, &cmap, image+i, 0, 0, 0, 0, 0 )  ) 
            exit( 0 );
         fclose(fptr);
         /* remap the image */
         ExtractLuminance( &H, cmap, *(image+i), 0, 0, 0, 0 );
         }

      /* do the various combinations */
      if ( mode == MAVG)
         {
         z=O;
         for ( j=0; j<N; ++j )
            {
            r = 0;
            for ( i=0; i<nimgs; i++ )
               r += *(*(image+i)+j);
            *(z++) = r / nimgs;
            }
         }
       else if ( mode == MOLC )  /* ordinary linear combination */
         {
         z=O;
         for ( j=0; j<N; ++j )
            {
            f = 0;
            c = coefflist;
            for ( i=0; i<nimgs; i++ )
               f += *(c++) * (float)(*(*(image+i)+j));
            *(z++) = (byte)(f > WHITE) 
                   ? WHITE : ( (f < BLACK) ? BLACK : f );
            }
         }
     else if ( mode == MMAX )
         {
         z=O;
         for ( j=0; j<N; ++j )
            {
            r = BLACK;
            for ( i=0; i<nimgs; i++ )
               {
               s = *(*(image+i)+j);
               if ( s > r )
                  r = s;
               }
            *(z++) = r;
            }
         }
      else if ( mode == MMIN )
         {
         z=O;
         for ( j=0; j<N; ++j )
            {
            r = WHITE;
            for ( i=0; i<nimgs; i++ )
               {
               s = *(*(image+i)+j);
               if ( s < r )
                  r = s;
               }
            *(z++) = r;
            }
         }
      else if ( (mode == MMED) || (mode == MOS) || (mode == MRLC) )
         {
         z=O;
         for ( k=0; k<N; ++k ) /* process loop */
            {
            w = pix;
            for ( i=0; i<nimgs; i++ )  /* get pixels */
               {
               *(w++) = *(*(image+i)+k);
               }

            for ( i=1; i<nimgs; i++ )  /* sort loop */
               {
               v = pix+i;
               u = v-1;
               if ( *u > *v )  /* order loop */
                  {
                   r = *u;
                  *u = *v;
                  *v =  r;
                  j=i-1;
                  while ( j )  /* bubble loop */
                     {
                     v = pix+j;
                     u = v-1;
                     if ( *u <= *v ) /* are ok */
                        {
                        break;       /* go to next order */
                        }
                     else            /* swap them and */
                        {
                         r = *u;
                        *u = *v;
                        *v =  r;
                        }
                     --j;            /* check next bubble */
                     }  /* bubble loop */
                  }  /* order loop */
               }  /* sort loop */

            if ( mode != MRLC ) /* (mode == MMED) || (mode == MOS) */
               {
               *(z++) = *(pix + nimgs - rank);
               }
            else /* mode == MRLC */
               {
               f = 0;
               u = pix;
               c = coefflist + ncoeffs;
               for ( i=0; i<nimgs; i++ )
                  {
                  f += *(--c) * (float)*(u++);
                  }
               *(z++) = (byte)(f > WHITE) 
                      ? WHITE : ( (f < BLACK) ? BLACK : f );
               }

            }  /* process loop */
         }  /* mode == MMED, MOS, or MRLC */

      /* make the output file */
      strcpy( filename, outbase );
      strcat( filename, Index );
      strcat( filename, ".ras" );
      fptr = OpenFile( filename, "", "w" );
      WriteRasterFile( fptr, &H, cmap, O, 0, 0, 0, 0 ); 
      fclose( fptr );
      fprintf( stderr, "%s\n", filename );

      ++IndexNum;
      --frames;
      }
               
   }

