/* mclean.c  --  mclean vers. 2.0 main program */

/* contains: main(), GetArgs(), OptErr(), AvgImg(), CalcResidual(), */
/*           GetThresh(), CleanUp(), WriteIt(), AndImg(), OrImg(),  */
/*           SetSub(), Skeleton();                                  */

/* dependencies: ReadRasterFile(), MorphSub(), WriteRasterFile()   */

/* mclean version 1.0   15 July 1991               */
/* morphological image noise reduction program     */
/*                                                 */
/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */
/* Nashville, TN 37235                             */
/* rap2@vuse.vanderbilt.edu                        */
/*                                                 */
/* This software is freely redistributable if      */
/* the author's name and affiliation are included. */


#include "mclean.h"
#include "rasterio.h"

#define AVOC  9        /* pointwise average of opening and closing */
#define OPCL 10        /* opening followed by closing (OC) */
#define CLOP 11        /* closing followed by opening (CO) */
#define OCCO 12        /* pointwise average of OC and CO */
#define NBANDS    16   /* maximum number of morphological bands */
#define FNLGTH    64   /* maximum file name string length */
#define DFLTBNDS   1   /* default number of morpho bands */
#define DFLTSESZ   5   /* default structuring element size */
#define DFLTTMF  1.0   /* default threshold multiplication factor */
#define DFLTSPRT   2   /* default topbot noise support threshold */

static void GetArgs();
static void OptErr();
static void AndImg();
static void OrImg();
static void SetSub();
static void AddImg();
static void AvgImg();
static void SubImg();
static void GetThresh();
static void TopBot();
static void Skeleton();
static void SmoothImg();
static void ReadIt();
static void WriteIt();


static   struct rasterfile RasHd;    /* raster file header */
static   unsigned char *cmap;        /* colormap */
 
main( argc, argv ) 
   unsigned int argc;    /* count of arguments parsed by operating system    */
   char *argv[];         /* pointers to arguments parsed by operating system */
   {

   byte *Img;          /* input image       */
   byte *Sm;           /* smoothed image    */
   byte *Th,*Bh;       /* tophat & bothat image          */
   byte *TA,*BA;       /* accumulated top and bot images */
   byte *I,*S,*T;      /* image pointers */
   int X,Y;            /* image horiz, vert. dimensions  */
   int M;              /* number of pixels in image      */
   int OpType;         /* type of smoothing operation to perform */
   int top;            /* do tophat flag    */
   int bot;            /* do bothat flag    */
   int bands;          /* number of morphological bands to create */
   int SEdim[NBANDS];  /* list of SE dimensions */
   int sX,sY;          /* structuring element x,y dimensions */
   double glev;        /* SE gray level differential */
   double ttmf;        /* top threshold multiplication factor */
   double btmf;        /* bottom threshold multiplication factor */
   int tsupp;          /* remove top areas with this many or fewer pixels */
   int bsupp;          /* remove bot areas with this many or fewer pixels */
   int iname;          /* position in arg list of oc file name */
   int i;              /* index variable    */
   int multilev;       /* TRUE => more than one morpho band to process */
   FILE *fp;           /* file pointer  */
   int readem;         /* T => read smooth images */
   int writem;         /* T => write smooth images */
   char fname[FNLGTH]; /* name of output file for smooth input/output */
   char *dot;          /* pointer to '.' in filename string */



   /* begin */



   if (argc < 3 ) /* display usage */
      {
      fprintf(stderr,
         "usage: mclean InFile OutFile [-m a|b|c|d|e|f|g] [-t] [-b]\n              [-l n [n n ... n]] [-g nn.nn] [-f nn.nn [nn.nn]] [-s n [n]]\n              [-r|-w SmImgName]\n");
      exit(0);
      }



   /* initialize everything that needs it */

   sX = sY = 0;
   X = Y = 0;


   /* Get the arguments */

   GetArgs(argc,argv,&OpType,&top,&bot,&bands,
           SEdim,&glev,&ttmf,&btmf,&tsupp,&bsupp,&readem,&writem,&iname );

   multilev = (bands > 1);



   /* read the input image into memory */
   ReadIt( argv[1], &RasHd, &cmap, &Img, ALLOC );


   /* get the image dimensions */

   X = RasHd.ras_width;
   Y = RasHd.ras_height;
   M = X*Y;



   /* allocate image space */

   if ( !( Sm = (byte *)calloc( M, sizeof(byte) )) ) /* smoothed image */
      {
      fprintf(stderr,"main: Unable to allocate smoothed image.\n");
      exit(0);
      }

   if ( top )  /* doing a tophat -- need space for it */
      {
      if ( !( Th = (byte *)calloc( M, sizeof(byte) )) )
         {
         fprintf(stderr,"main: Unable to allocate tophat image\n");
         exit(0);
         } 
      if ( multilev ) /* need to accumulate tophat */
         { 
         if ( !( TA = (byte *)calloc( M, sizeof(byte) )) )
            {
            fprintf(stderr,"main: Unable to allocate top accum image\n");
            exit(0);
            }
         }
      else
         TA = Th;
      }

   if ( bot )  /* doing a bothat -- need space for it */
      {
      if ( !( Bh = (byte *)calloc( M, sizeof(byte) )) )
         {
         fprintf(stderr,"main: Unable to allocate bothat image\n");
         exit(0);
         } 
      if ( multilev ) /* need to accumulate bothat */
         { 
         if ( !( BA = (byte *)calloc( M, sizeof(byte) )) )
            {
            fprintf(stderr,"main :Unable to allocate bot accum image\n");
            exit(0);
            } 
         }
      else
         BA = Bh;
      }



   /* setup file name for smoothed image (if requested) */

   if ( iname )
      {
      strcpy( fname, argv[iname] );
      if ( !(dot = rindex( fname, '.' )) ) 
         dot = fname + strlen( fname );
      }



   /* main algorithm loop */

   I = Img;
   S = Sm;

   for ( i=0; i<bands; ++i )
      {
      sX=sY=SEdim[i];
      fprintf( stderr, "Band: %d   SE diameter: %d\n",i+1,sX);

      /* read in, or smooth the image */
      if ( readem )
         {
         sprintf( dot, ".%03d", sX );
         strcat( fname, ".ras" );
         ReadIt( fname, &RasHd, &cmap, &S, 0 );
         }
      else
         {
         fprintf( stderr, "  Smoothing\n");
         SmoothImg( I, S, X, Y, sX, sY, glev, OpType );
         }

      /* if requested, write out the smoothed image */
      if ( writem )
         {
         sprintf( dot, ".%03d", sX );
         strcat( fname, ".ras" );
         WriteIt( fname, &RasHd, cmap, S );
         }

      /* if requested, get the tophat */
      if (top) 
         {
         fprintf( stderr, "  tophat\n");
         SubImg( I, S, Th, X, Y );

         TopBot( Th, X, Y, sX, sY, 0, ttmf, tsupp );
         if ( multilev ) AddImg( Th, TA, TA, X, Y );
         }

      /* if requested, get the bothat */
      if (bot) 
         {
         fprintf( stderr, "  bothat\n");
         SubImg( S, I, Bh, X, Y );

         TopBot( Bh, X, Y, sX, sY, 0, btmf, bsupp );
         if ( multilev ) AddImg( Bh, BA, BA, X, Y );
         }

      T = I;
      I = S;
      S = T;
      }



   /* construct cleaned-up image */

   if ( top ) AddImg( TA, I, I, X, Y );
   if ( bot ) SubImg( I, BA, I, X, Y );



   /* output cleaned image */

   WriteIt( argv[2], &RasHd, cmap, I );



   /* free memory */

   free( Img );
   free( Sm );
   if (cmap) free(cmap);
   if (top) { free( Th ); if (multilev) free( TA ); }
   if (bot) { free( Bh ); if (multilev) free( BA ); }



   } /* end */





/* parse (argc, argv) argument list for morph */

static void GetArgs(
   argc,argv,OpType,top,bot,bands,SEdim,glev,ttmf,btmf,tsupp,bsupp,
   readem,writem,iname )
   unsigned int argc;/* count of arguments parsed by operating system    */
   char *argv[];     /* pointers to arguments parsed by operating system */
   int *OpType;      /* type of smoothing operation to perform */
   int *top;         /* do tophat flag    */
   int *bot;         /* do bothat flag    */
   int *bands;       /* number of morphological bands to create */
   int *SEdim;       /* list of SE dimensions */
   double *glev;     /* SE gray level differential */
   double *ttmf;     /* threshold multiplication factor */
   double *btmf;     /* bottom threshold multiplication factor */
   int *tsupp;       /* remove top areas with this many or fewer pixels */
   int *bsupp;       /* remove bot areas with this many or fewer pixels */
   int *readem;      /* T => read smooth images */
   int *writem;      /* T => write smooth images */
   int *iname;       /* position in arg list of smooth image file name */
   {
   int n = argc;
   int i,j;
   char *s;
   int tflag,bflag;  /* -t,-b specified in the arg list */

   /* initially zero or set to default value */
   *OpType = OCCO;
   *top = *bot = TRUE;
   *bands = DFLTBNDS;
   *SEdim = DFLTSESZ;
   *glev = (double)BLACK;
   *ttmf = DFLTTMF;
   *btmf = DFLTTMF;
   *tsupp = DFLTSPRT;
   *bsupp = DFLTSPRT;
   *iname = 0;
   s = "average of open-close and close-open";
   tflag = bflag = FALSE;
   *readem = *writem = FALSE;


   /* get arguments */
   while ( --n )
      {
      if ( argv[n][0] == '-' )          /* then this is a switch */
         {
         if ( argv[n][1] == 'm' )       /* smoothing op spec is next */
            {
            if ( argv[n+1][0] == 'a' )
               {
               *OpType = OCCO;
               s = "average of open-close and close-open.";
               }
            else if ( argv[n+1][0] == 'b' )
               {
               *OpType = OPCL;
               s = "open-close.";
               }
            else if ( argv[n+1][0] == 'c' )
               {
               *OpType = CLOP;
               s = "close-open.";
               }
            else if ( argv[n+1][0] == 'd' )
               {
               *OpType = AVOC;
               s = "average of open and close.";
               }
            else if ( argv[n+1][0] == 'e' )
               {
               *OpType = OPEN;
               *top = TRUE;
               *bot = FALSE;
               s = "tophat.";
               }
            else if ( argv[n+1][0] == 'f' )
               {
               *OpType = CLOSE;
               *top  = FALSE;
               *bot  = TRUE;
               s = "bothat.";
               }
            else if ( argv[n+1][0] == 'g' )
               {
               *OpType = RANK;
               s = "median filter.";
               }
            else 
               {
               fprintf( stderr,"-m must be followed by one of a,b,c,d,e,f.\n");
               exit(0);
               }
            }
         else if ( argv[n][1] == 't' ) /* operate on top residuals only */
            tflag = TRUE;
         else if ( argv[n][1] == 'b' ) /* operate on bot residuals only */
            bflag = TRUE;
         else if ( argv[n][1] == 'l' ) /* band specs follow */
            {
            if ( n+1<argc && isdigit( argv[n+1][0] ) ) /* a number follows -l */
               {
               *bands = atoi( argv[n+1] ); /* it is the number of bands */
               if (*bands == 1) 
                  {     /* if there is only one and it is given, get it */
                  if ( n+2<argc && isdigit( argv[n+2][0] ) )
                     *SEdim = atoi( argv[n+2] );
                  else  /* default to diameter 3 */
                     *SEdim = 3;
                  }
               else /* there is more than one band */
                  {
                  i=0;
                  while ( i<*bands && n+2+i<argc && isdigit( argv[n+2+i][0] ) )
                     {  /* get the sizes that are specified on the cmd line */
                     SEdim[i] = atoi( argv[n+2+i] );
                     if ( (i > 0) && (SEdim[i] <= SEdim[i-1]) )
                        {
                        fprintf(stderr,"SE sizes must increase\n");
                        exit( 0 );
                        }
                     ++i;
                     }
                  if ( i<*bands ) /* there are bands that were not specified */
                     {
                     if ( i > 0 ) /* get smallest pwr-o-2 > last SE size */
                        {
                        j = (int)((log((double)SEdim[i-1])/log(2))+0.5) + 1;
                        }  
                     else
                        {
                        j = 1; /* no bands were specified, start with 3 */
                        }  
                     while ( i<*bands )
                        {
                        SEdim[i++] = (int)pow( 2, ( (double)(j++) ) ) + 1;
                        }
                     }
                  }
               }
            else 
               {
               fprintf( stderr,"-l must be followed by 1 or more integers.\n");
               exit(0);
               }
            }
         else if ( argv[n][1] == 'f' )    /* thresh mult. factor follows */
            {
            *btmf = atof( argv[n+1] );    /* if there is only 1 number  */
            if ( (n+2<argc) && isdigit(argv[n+2][0]) )  /* else if two */
               *ttmf = atof( argv[n+2] ); /* top is second */
            else
               *ttmf = *btmf;             /* they are the same */
            }
         else if ( argv[n][1] == 's' )     /* support value follows */
            {
            *bsupp = atoi( argv[n+1] );    /* if there is only 1 number  */
            if ( (n+2<argc) && isdigit(argv[n+2][0]) )  /* else if two */
               *tsupp = atoi( argv[n+2] ); /* top is second */
            else
               *tsupp = *bsupp;            /* they are the same */
            }
         else if ( argv[n][1] == 'r' )  /* read smoothed images */
            {
            *readem = TRUE;
            *iname = n+1;
            }
         else if ( argv[n][1] == 'w' )  /* write smoothed images */
            {
            *writem = TRUE;
            *iname = n+1;
            }
         else if ( argv[n][1] == 'g' )  /* gray-level differential follows */
            *glev = atof( argv[n+1] );
         }    /* end of if ( argv[n][0] == '-' ) */
      }   /* end of  while ( --n ) "get arguments" loop */

   if ( tflag )
      {
      if (*OpType == CLOSE)
         {
         fprintf( stderr,"Cannot specify both -m f and -t\n");
         exit( 0 );
         }
      if ( !bflag )
         *bot  = FALSE;
      }

   if ( bflag )
      {
      if (*OpType == OPEN)
         {
         fprintf( stderr,"Cannot specify both -m e and -b\n");
         exit( 0 );
         }
      if ( !tflag )
         *top  = FALSE;
      }

   if ( *readem && *writem )
      {
      fprintf( stderr,"Cannot specify both -r and -w.\n");
      exit( 0 );
      }

   fprintf( stderr,"Will perform %s.\n",s);
   fprintf(stderr,"using %d bands:",*bands);
   for ( i=0; i<*bands; ++i)
      fprintf(stderr," %d",SEdim[i]);
   fprintf(stderr,".\n");

   return;
   }



static void SmoothImg( In, Out, X, Y, sX, sY, gld, type )
   byte *In;
   byte *Out;
   int X,Y;
   int sX, sY;
   double gld;
   int type;
   {
   byte *Op, *Cl, *Sm;
   byte *O, *S;
   int i,n;
   int sZ;
   int M;

   M = X*Y;
   Op=Cl=Sm=NULL;

   if ( gld > 0 )
      sZ = (int)(gld * (double)(( sX<=sY ? sX : sY ) / 2) + 0.5);
   else
      sZ = 0;

/***fprintf(stderr,"GLD: %f  sX: %d  sY: %d  sZ: %d\n",gld,sX,sY,sZ);*/

   /* if called for, do an opening of the input image */

   if ( type==OCCO || type==OPCL || type==AVOC || type==OPEN ) 
      {   
      if ( !(Op = (byte *)calloc( M, sizeof(byte) )) )
         {
         fprintf(stderr,"SmoothImg: Unable to allocate opened image\n");
         exit(0);
         } 
      MorphSub( OPEN, In, Op, X, Y, GRAIMG, FALSE, 0, 0,
                NULL, AUTO, sX, sY, sZ, GRASE, SET, 0, FALSE, FALSE);

      if ( type==OCCO || type==OPCL )
         MorphSub( CLOSE, Op, Op, X, Y, GRAIMG, FALSE, 0, 0,
                   NULL, AUTO, sX, sY, sZ, GRASE, SET, 0, FALSE, FALSE);
      }

 
   /* if called for, do a closing of the input image */

   if ( type==OCCO || type==CLOP || type==AVOC || type==CLOSE ) 
      {
      if ( !(Cl = (byte *)calloc( M, sizeof(byte) )) )
         {
         fprintf(stderr,"SmoothImg: Unable to allocate closed image\n");
         exit(0);
         }
      MorphSub( CLOSE, In, Cl, X, Y, GRAIMG, FALSE, 0, 0,
                NULL, AUTO, sX, sY, sZ, GRASE, SET, 0, FALSE, FALSE);

      if ( type==OCCO || type==CLOP )
         MorphSub( OPEN, Cl, Cl, X, Y, GRAIMG, FALSE, 0, 0,
                   NULL, AUTO, sX, sY, sZ, GRASE, SET, 0, FALSE, FALSE);
      }


   /* do the median if requested */
   
   if ( type==RANK )
      {
      if ( !(Op = (byte *)calloc( M, sizeof(byte) )) )
         {
         fprintf(stderr,"SmoothImg: Unable to allocate median filtered image\n");
         exit(0);
         } 
      MorphSub( RANK, In, Op, X, Y, GRAIMG, FALSE, 0, 0,
                NULL, AUTO, sX, sY, sZ, GRASE, SET, 0, FALSE, FALSE);
      }



   /* get the average */

   if ( type==OPEN || type== OPCL || type==RANK ) /* don't average the two, */ 
      Sm = Op;                     /* the smooth img is med or open or clop */
   else if ( type==CLOSE || type== CLOP )  /* don't average the two, */ 
      Sm = Cl;                    /* the smooth img is close or clop */
   else  /* must average the two */
      {
      if ( !( Sm = (byte *)calloc( M, sizeof(byte) )) ) /* smoothed image */
         {
         fprintf(stderr,"SmoothImg: Unable to allocate smoothed image\n");
         exit(0);
         }
      AvgImg( Op, Cl, Sm, X, Y );
      }


   /* copy the smoothed image to the output */

   n=X*Y;
   O = Out;
   S = Sm;
   for ( i=0; i<n; ++i )
      *(O++) = *(S++);


   if ( Op ) free( Op );
   if ( Cl ) free( Cl );
   if ( (type == OCCO) || (type == AVOC) ) free( Sm );

   return;
   }




static void AvgImg ( I1, I2, O, X, Y )
   byte *I1;  /* image 1 */
   byte *I2;  /* image 2 */
   byte *O;   /* result  */
   int X,Y;   /* image horizontal, vertical dimensions */
   {
   byte *A = I1;
   byte *B = I2;
   byte *C = O;
   int   n = X*Y;
   int i;
   int r;

   for (i=0; i<n; ++i)
      {
      r = (int)*(A++) + (int)*(B++);
      *(C++) = (byte)(r >> 1);
      }

   return;
   }

      

static void AddImg( I1, I2, O, X, Y )
   byte *I1;  /* image 1 */
   byte *I2;  /* image 2 */
   byte *O;   /* result  */
   int X,Y;   /* image horizontal, vertical dimensions */
   {
   byte *A = I1;
   byte *B = I2;
   byte *C = O;
   int   n = X*Y;
   int i,p;

   for (i=0; i<n; ++i)
      {
      p = (int)(*(A++)) + (int)(*(B++));
      *(C++) = (p > WHITE) ? WHITE : ( (p < BLACK) ? BLACK : p );
      }

   return;
   }



static void SubImg( I1, I2, O, X, Y )
   byte *I1;  /* image 1 */
   byte *I2;  /* image 2 */
   byte *O;   /* result  */
   int X,Y;   /* image horizontal, vertical dimensions */
   {
   byte *A = I1;
   byte *B = I2;
   byte *C = O;
   int   n = X*Y;
   int i,p;

   for (i=0; i<n; ++i)
      {
      p = (int)(*(A++)) - (int)(*(B++));
      *(C++) = (p > 0) ? p : BLACK;
      }

   return;
   }



static void AndImg ( I1, I2, O, X, Y )
   byte *I1;  /* image 1 */
   byte *I2;  /* image 2 */
   byte *O;   /* result  */
   int X,Y;   /* image horizontal, vertical dimensions */
   {
   byte *A = I1;
   byte *B = I2;
   byte *C = O;
   int   n = X*Y;
   int i;

   for (i=0; i<n; ++i)
      {
      *C = (*A && *B) ? *A : BLACK;
      ++A;
      ++B;
      ++C;
      }

   return;
   }

      

static void OrImg ( I1, I2, O, X, Y )
   byte *I1;  /* image 1 */
   byte *I2;  /* image 2 */
   byte *O;   /* result  */
   int X,Y;   /* image horizontal, vertical dimensions */
   {
   byte *A = I1;
   byte *B = I2;
   byte *C = O;
   int   n = X*Y;
   int i;

   for (i=0; i<n; ++i)
      {
      *C = (*A || *B) ? (*A ? *A : *B) : BLACK;
      ++A;
      ++B;
      ++C;
      }

   return;
   }

      

static void SetSub ( I1, I2, O, X, Y )
   byte *I1;  /* binary image 1 */
   byte *I2;  /* binary image 2 */
   byte *O;   /* binary result  */
   int X,Y;   /* image horizontal, vertical dimensions */
   {
   byte *A = I1;
   byte *B = I2;
   byte *C = O;
   int   n = X*Y;
   int i;

   for (i=0; i<n; ++i)
      {
      *C = (*A && *B) ? BLACK : *A;
      ++A;
      ++B;
      ++C;
      }

   return;
   }



static void Skeleton( In, Out, X, Y, sX, sY )
   byte *In;
   byte *Out;
   int  X,Y;
   int  sX, sY;
   {
   int sx,sy;
   int stelem;
   char *SEName;
   byte *Mask1, *Mask2, *Mask3;
   byte *I, *O;
   int i,n;

   /* allocate space for image masks */

   if ( !( Mask1 = (byte *)calloc( X*Y, sizeof(byte) )) )
      {
      fprintf(stderr,"Skeleton: Unable to allocate image mask 1\n");
      exit(0);
      } 
   if ( !( Mask2 = (byte *)calloc( X*Y, sizeof(byte) )) )
      {
      fprintf(stderr,"Skeleton: Unable to allocate image mask 2\n");
      exit(0);
      } 
   if ( !( Mask3 = (byte *)calloc( X*Y, sizeof(byte) )) )
      {
      fprintf(stderr,"Skeleton: Unable to allocate image mask 3\n");
      exit(0);
      } 

/*** WriteIt( "xxx.ras", &RasHd, cmap, In ); */

 
   MorphSub( OPEN, In, Mask2, X, Y, BINIMG, FALSE, 0, 0,
             NULL, AUTO, 2, 2, 0, GRASE, SET, 0, FALSE, FALSE);

/*** WriteIt( "xxx.ras", &RasHd, cmap, Mask2 ); */

   SetSub( In, Mask2, Mask1, X ,Y );

/*** WriteIt( "xxx.ras", &RasHd, cmap, Mask1 ); */

   sx = sy = 2;

   while ( (sx < sX) && (sy < sY) )
      {
      MorphSub( ERODE, In, Mask2, X, Y, BINIMG, FALSE, 0, 0,
                NULL, AUTO, sx, sy, 0, GRASE, SET, 0, FALSE, FALSE);

/*** WriteIt( "xxx.ras", &RasHd, cmap, Mask2 ); */

      MorphSub( OPEN, Mask2, Mask3, X, Y, BINIMG, FALSE, 0, 0,
                NULL, AUTO, 2, 2, 0, GRASE, SET, 0, FALSE, FALSE);

/*** WriteIt( "xxx.ras", &RasHd, cmap, Mask3 ); */

      SetSub( Mask2, Mask3, Mask2, X, Y);

/*** WriteIt( "xxx.ras", &RasHd, cmap, Mask2 ); */

      OrImg( Mask1, Mask2, Mask1, X, Y);

/*** WriteIt( "xxx.ras", &RasHd, cmap, Mask1 ); */

      ++sx;
      ++sy;
      }

   n = X*Y;
   I = Mask1;
   O = Out;
   for ( i=0; i<n; ++i )
      *(O++) = *(I++);
   
   free(Mask1);
   free(Mask2);
   free(Mask3);

   return;
   }
   


static void TopBot( Img, X, Y, sX, sY, threshold, factor, support )
   byte *Img;
   int   X,Y;
   int   sX,sY;
   int   threshold;
   double factor;
   int support;
   {
   byte *I;
   byte *Mask1, *Mask2;
   int i,j,n;
   int *histo, *h;
   int area;
   int rank;
   int T;
   double a,m,p;

   /* allocate space for image masks */

   if ( !( Mask1 = (byte *)calloc( X*Y, sizeof(byte) )) )
      {
      fprintf(stderr,"TopBot: Unable to allocate image mask 1\n");
      exit(0);
      } 
   if ( !( Mask2 = (byte *)calloc( X*Y, sizeof(byte) )) )
      {
      fprintf(stderr,"TopBot: Unable to allocate image mask 2\n");
      exit(0);
      } 

   T = threshold;

   if ( !T ) /* calculate threshold as the 2nd moment */
      {      /* of the histogram othe residual image. */
      /* allocate space for histogram */
      if ( !( histo = (int *)calloc( WHITE+1, sizeof(int) )) )
         {
         fprintf(stderr,"TopBot: Unable to allocate histogram\n");
         exit(0);
         } 

      /* collect histogram */
      n = X*Y;
      I = Img;
      for (i=0; i<n; ++i)  
         (*(histo + *(I++)))++;    


      /* calculate area of histogram */
      area = 0;
      h = histo;
      for (i=0; i<=WHITE; ++i)  
         area += *(h++);

      /* calculate second moment of histogram */
      a = (double)area;
      h = histo;
      m = 0;
      for (i=0; i<=WHITE; ++i)
         {
         p  = (double)(*(h++)) / a;
         m += p * (double)(i*i);
         }
      m = sqrt(m);
      T = (int)(factor*m + 0.5);
      }

   /* threshold residual to create mask 1 */
   MorphSub( ERODE, Img, Mask1, X, Y, GRAIMG, FALSE, T, WHITE,
             NULL, AUTO, 1, 1, 0, GRASE, SET, 0, FALSE, FALSE);

/*** WriteIt( "xxx.ras", &RasHd, cmap, Mask1); */

   /* delete white regions in mask 1 with 'support' or fewer pixels */
   /* rank = min number of pixels in 8-nbhd necessary to keep nbhd in mask */
   rank = ++support;
   if ( (rank < 1) || (rank > 9) )
      {
      fprintf(stderr,"Support (opt -b) must be >=0 and <9.\n");
      exit(0);;
      } 
   MorphSub( RANK, Mask1, Mask2, X, Y, BINIMG, FALSE, 0, 0,
             NULL, S3X3, 0, 0, 0, GRASE, SET, rank, FALSE, FALSE);

   /* Each pixel in result is a nbhd pointer, not the nbhd itself. */
   /* Therefore need to mask the nbhds using a 8 connected dilation */
   MorphSub( DILATE, Mask2, Mask2, X, Y, BINIMG, FALSE, 0, 0,
             NULL, S3X3, 0, 0, 0, GRASE, SET, 0, FALSE, FALSE);

   /* grab all nbhds with more than 'support' pixels (Mask1 AND Mask2) */
   AndImg( Mask1, Mask2, Mask1, X, Y);

   /* delete isolated pixels in result (Mask1) */
   MorphSub( ERODE | NOTFLG, Mask1, Mask2, X, Y, BINIMG, FALSE, 0, 0,
             "isod8.tfm", NULL, 0, 0, 0, BINSE, SET, 0, FALSE, FALSE);

   /* do it again */
   MorphSub( RANK, Mask2, Mask1, X, Y, BINIMG, FALSE, 0, 0,
             NULL, S3X3, 0, 0, 0, GRASE, SET, rank, FALSE, FALSE);

   /* Each pixel in result is a nbhd pointer, not the nbhd itself. */
   /* Therefore need to mask the nbhds using a 8 connected diation */
   MorphSub( DILATE, Mask1, Mask1, X, Y, BINIMG, FALSE, 0, 0,
             NULL, S3X3, 0, 0, 0, GRASE, SET, 0, FALSE, FALSE);

   /* grab all nbhds with more than 'support' pixels (Mask1 AND Mask2) */
   AndImg( Mask2, Mask1, Mask2, X, Y);

   /* delete isolated pixels in result (Mask2) */
   MorphSub( ERODE | NOTFLG, Mask2, Mask1, X, Y, BINIMG, FALSE, 0, 0,
             "isod8.tfm", NULL, 0, 0, 0, BINSE, SET, 0, FALSE, FALSE);
   free(Mask2);

   /* get morphological skeleton of mask */
   Skeleton( Mask1, Mask1, X, Y, sX, sY );

   /* dilate skeleton with with original SE */
   MorphSub( DILATE, Mask1, Mask1, X, Y, BINIMG, FALSE, 0, 0,
             NULL, AUTO, sX, sY, 0, BINSE, SET, 0, FALSE, FALSE);

   /* pick up the indicated pixels from the residual */
   AndImg( Img, Mask1, Img, X, Y);

   free( Mask1 );
   if ( histo ) free( histo );
   return;
   }



static void ReadIt( name, H, C, I, Alloc )
   char *name;                /* filename */
   struct rasterfile *H;      /* raster file header */
   unsigned char **C;         /* colormap */
   byte **I;                  /* image */
   int Alloc;                 /* allocate flag */
   {
   FILE *fp;

   fp = OpenFile( name, "IMGPATHI", "r" );
   ReadRasterFile( fp, H, C, I, 0, 0, 0, 0, Alloc );
   fclose( fp );

   /* extract the luminance */
   ExtractLuminance( H, *C, *I, 0, 0, 0, 0 );

   return;
   }


static void WriteIt( name, H, C, I )
   char *name;                /* filename */
   struct rasterfile H;       /* raster file header */
   unsigned char *C;          /* colormap */
   byte *I;                   /* image */
   {
   FILE *fp;

   fp = OpenFile( name, "IMGPATHO", "w" );
   WriteRasterFile( fp, H, C, I, 0, 0, 0, 0 );
   fflush( fp );
   fclose( fp );

   return;
   }




static void OptErr( s, t )
   char *s,*t;
   {
   fprintf(stderr,"Invalid Op: %s %s\n",s,t);
   exit( 0 );
   }



