/*
 * Handle module dependencies
 *
 * Copyright 1994, 1995, 1996, 1997: Jacques Gelinas <jack@solucorp.qc.ca>
 * Additional modifications: Bjrn Ekwall <bj0rn@blox.se> February 1999
 *
 * This file is part of the Linux modutils.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <sys/stat.h>
#include <ctype.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <elf.h>
#include ELF_MACHINE_H
extern "C" {
#include "obj.h"
#include "util.h"
};
#include "depmod.h"

#define ALLOC_MODULE	5000

MODULES::MODULES()
{
	tbmod = (MODULE *) xmalloc(ALLOC_MODULE * sizeof(MODULE));
	nbmod = 0;
	base = NULL;
	base_len = 0;
}

/*
 *	Finalize the information about a module.
 */
void MODULES::setmod(
	       MODULE * mod,
	       SYMBOL * tbpub[],
	       int nbpub,
	       SYMBOL * tbext[],
	       int nbext)
{
	mod->pub.nb = nbpub;
	if (nbpub > 0) {
		int size = nbpub * sizeof(SYMBOL *);

		mod->pub.tb = (SYMBOL **) xmalloc(size);
		memcpy(mod->pub.tb, tbpub, size);
	} else
		mod->pub.tb = NULL;

	mod->ext.nb = nbext;
	if (nbext > 0) {
		int size = nbext * sizeof(SYMBOL *);

		mod->ext.tb = (SYMBOL **) xmalloc(size);
		memcpy(mod->ext.tb, tbext, size);
		mod->resolved = 0;
	} else {
		mod->ext.tb = NULL;
		mod->resolved = 1;
	}
}

/*
 *	Create a dummy module entry in the list of modules.
 *	This module is used to handle the exported kernel symbols.
 */
MODULE *MODULES::setdummy(const char *name)
{
	MODULE *mod = tbmod + nbmod++;

	mod->name = xstrdup(name);
	mod->pub.nb = 0;
	mod->pub.tb = NULL;
	mod->ext.nb = 0;
	mod->ext.tb = NULL;

	return mod;
}

void MODULES::resolve(SYMBOLS & syms)
{
	MODULE *mod;
	SYMBOL **sym;
	int i;
	int n;
	int resolved;

	/*
	 * Resolve, starting with the most recently added
	 * module symbol definitions.
	 */
	if (nbmod > 0) {
		for (i = nbmod - 1, mod = tbmod + i; i >= 0; --i, --mod) {
			if (mod->resolved)
				continue;

			for (resolved = 1, n = 0, sym = mod->ext.tb;
			     n < mod->ext.nb; ++sym, ++n) {
				if ((*sym)->status == SYM_UNDEF &&
				    syms.resolve(*sym) == 0)
					resolved = 0;
			}
			mod->resolved = resolved;
		}
	}
}

/*
 *	Read the symbols in an object and register them in syms
 *	Return -1 if there is an error.
 */
int MODULES::loadobj(SYMBOLS & syms, const char *objname)
{
	MODULE *mod;
	FILE *fp;
	struct obj_file *f;
	struct obj_section *sect;
	struct obj_symbol *objsym;
	SYMBOL *tbext[5000];
	int nbext = 0;
	SYMBOL *tbpub[5000];
	int nbpub = 0;
	int ksymtab;
	int i;
	int is_2_2;

	if ((fp = fopen(objname, "r")) == NULL)
		return 1;

	if (!(f = obj_load(fp))) {
		fclose(fp);
		return -1;
	}
	fclose(fp);

	/*
	 * If we have changed base directory
	 * then use the defined symbols from modules
	 * in the _same_ directory to resolve whatever
	 * undefined symbols there are.
	 *
	 * This strategy ensures that we will have
	 * as correct dependencies as possible,
	 * even if the same symbol is defined by
	 * other modules in other directories.
	 */
	char *p = strrchr(objname, '/');
	int len = 1 + (int)(p - objname);
	if (base_len != len || base == NULL ||
	    strncmp(base, objname, len) != 0) {
		if (base)
			resolve(syms);
		base = xstrdup(objname);
		base_len = len;
	}

	mod = tbmod + nbmod++;
	mod->name = xstrdup(objname);

	if ((sect = obj_find_section(f, "__ksymtab")) != NULL)
		ksymtab = sect->idx; /* Only in 2.2 (or at least not 2.0) */
	else
		ksymtab = -1;

	if (sect ||
	    obj_find_section(f, ".modinfo") ||
	    obj_find_symbol(f, "__this_module"))
		is_2_2 = 1;
	else
		is_2_2 = 0;

	for (i = 0; i < HASH_BUCKETS; ++i) {
		for (objsym = f->symtab[i]; objsym; objsym = objsym->next) {
			if (objsym->secidx == SHN_UNDEF) {
				if (ELFW(ST_BIND)(objsym->info) != STB_WEAK) {
					tbext[nbext++] = syms.add(objsym->name,
								  mod,
								  SYM_UNDEF);
				}
				continue;
			}
			/* else */

			if (is_2_2 && ksymtab != -1) {
				/* A 2.2 module using EXPORT_SYMBOL */
				if (objsym->secidx == ksymtab &&
				    ELFW(ST_BIND)(objsym->info) == STB_GLOBAL) {
					/* Ignore leading "__ksymtab_" */
					tbpub[nbpub++] = syms.add(objsym->name + 10,
								  mod,
								  SYM_DEFINED);
				}
				continue;
			}
			/* else */

			if (is_2_2) {
				/*
				 * A 2.2 module _not_ using EXPORT_SYMBOL
				 * It might still want to export symbols,
				 * although strictly speaking it shouldn't...
				 * (Seen in pcmcia)
				 */
				if (ELFW(ST_BIND)(objsym->info) == STB_GLOBAL) {
					tbpub[nbpub++] = syms.add(objsym->name,
								  mod,
								  SYM_DEFINED);
				}
				continue;
			}
			/* else */

			/* Not undefined or 2.2 ksymtab entry */
			if (objsym->secidx < SHN_LORESERVE
			/*
			 * The test below is removed for 2.0 compatibility
			 * since some 2.0-modules (correctly) hide the
			 * symbols it exports via register_symtab()
			 */
			/* && ELFW(ST_BIND)(objsym->info) == STB_GLOBAL */
			     ) {
				tbpub[nbpub++] = syms.add(objsym->name,
							  mod,
							  SYM_DEFINED);
			}
		}
	}
	setmod(mod, tbpub, nbpub, tbext, nbext);
	obj_free(f);

	return 0;
}

/*
 *	Format the dependancy list of a module into a simple makefile
 */
void MODULES::prtdepend(
		  FILE * fout,
		  int skipchars,	/* For depmod -a in image of a tree */
		  int verbose,		/* Print all modules visiteds */
		  int quiet,		/* Don't print errors */
		  int showerror)	/* Shows undefined symbols */
{
	MODULE **tbdep = new MODULE *[nbmod];
	MODULE *ptmod = tbmod;

	for (int i = 0; i < nbmod; i++, ptmod++) {
		/* Don't print a dependency list for the _kernel_ ... */
		if (strcmp(ptmod->name, "-") == 0)
			continue;

		SYMBOL **ptext = ptmod->ext.tb;
		int nbext = ptmod->ext.nb;
		int nbdepmod = 0;
		int nberr = 0;

		/* Look for unresolved symbols in this module */
		for (int e = 0; e < nbext; e++, ptext++) {
			MODULE *mod = (*ptext)->module;

			if ((*ptext)->status == SYM_RESOLVED) {
				/* Resolved by self or exported kernel symbol */
				if (strcmp(mod->name, ptmod->name) == 0 ||
				    strcmp(mod->name, "-") == 0)
					continue;

				/* No duplicate dependencies, please */
				int m;
				for (m = 0; m < nbdepmod; m++) {
					if (tbdep[m] == mod)
						break;
				}
				if (m == nbdepmod)
					tbdep[nbdepmod++] = mod;
			} else {
				if (!quiet && nberr == 0)
					error("*** Unresolved symbols in %s",
					      ptmod->name);
				if (!quiet && showerror)
						error("\t%s", (*ptext)->name);
				nberr++;
			}
		}

		if (verbose)
			printf("%s\n", ptmod->name + skipchars);

		fprintf(fout, "%s:", ptmod->name + skipchars);
		for (int m = 0; m < nbdepmod; m++) {
			if (m != 0 /*&& (m & 3) == 0*/)
				fprintf(fout, " \\\n");
			fprintf(fout, "\t%s", tbdep[m]->name + skipchars);
		}
		fprintf(fout, "\n\n");
	}
	delete[]tbdep;
}
