#include <stddef.h>	/* for offsetof() macro */
#include <OI/oi.H>


/*
 *	Resources
 *	
 *	This program provides a complete resource and command
 *	line argument fetching example.  The program presents
 *	the user with a daily calendar.  The start and end
 *	times can be specified on both the command line as
 *	arguments, as well as within a resource file.
 *
 *	Using the techniques shown in this program, the user
 *	can set the start and end times via command line arguements
 *	as follows:
 *		-start <time> 
 *		-end <time>
 *		where <time> is a military time such as 0930, or 1400
 *	or the user can specify start and end times within their ~/.Xdefaults
 *	file:
 *		*calendar*timeStart:	1200
 *		*calendar*timeEnd:	1900
 *	or, in a similar fashion within another resource file to be
 *	specified as command line arguments:
 *		-config calendar_resource_file
 *	
 *
 *	The reader should refer to the OI documentation for
 *	information on the following member functions.
 *		- OIIntro
 *			OI_init
 *			OI_begin_interaction
 *			OI_fini
 *		- OI_resources
 *		- OI_d_tech( 30 )
 *			layout_associated_object
 *			set_associated_object
 *			set_layout
 *		- OI_app_window( 3O )
 *			oi_create_app_window
 *		- OI_entry_field( 3O )
 *			oi_create_entry_field
 *
 */

/*
 *	Resources
 *
 *	define the additional resources available for this object.
 *	The OI_resource structure is further discussed in the documentation
 *	under OIresources( 3O ), but for review:
 *		struct OI_resource {
 *			- resource name (OI_n_...)
 *				refer to	<OI/resname.H>
 *						<X11/StringDefs.h>
 *			- resource class (OI_c_...)
 *				refer to	<OI/resname.H>
 *						<X11/StringDefs.h>
 *			- representation type desired (OI_r_...)
 *				refer to	<OI/resname.H>
 *						<X11/StringDefs.h>
 *			- size in bytes of representation
 *			- offset from base to put resource value
 *				uses the macro:
 *					offsetof( class, parameter );
 *			- representation type of specified default (OI_r_...)
 *			- address of default resources
 *				two examples (representation type, address of resource)
 *					OI_r_Immediate, (XtPointer) 5
 *					OI_r_String, "black"
 *			- callback member function for resources being set
 *				for example:
 *					OI_RESOURCE_MEMFN_CAST(&HexEntryField::res_maximum_value)
 *				please note that res_maximum_value would decode the (void * arg) as follows:
 *					set_maximum_value(*((int*)arg));
 *			- callback procedure for resource being set
 *			- callback member function for resource being gotten
 *				for example:
 *					OI_RESOURCE_GET_MEMFN_CAST(&HexEntryField::maximum_value)
 *			- callback procedure for resource being gotten
 *		}
 *
 */

#define OI_n_timeStart		"timeStart"		/* start time resource name */
#define OI_n_timeEnd		"timeEnd"		/* end time resource name */

#define OI_c_TimeStart		"TimeStart"		/* start time resource class */
#define OI_c_TimeEnd		"TimeEnd"		/* end time resource class */

#define DefaultTimeStart	"0800"			/* default start time */
#define DefaultTimeEnd		"1800"			/* default end time */

#define DefaultTimeInterval	30			/* default time interval (should also be a resource) */

typedef struct _MyRes {					/* array into which to store the resources */
	int		timeStart;
	int		timeEnd;
} MyRes;
MyRes MyResources;

static OI_resource resources[] = {
	{ OI_n_timeStart, OI_c_TimeStart, OI_r_Int, sizeof(int), offsetof(MyRes, timeStart),
		OI_r_String, DefaultTimeStart, NULL_PMF, NULL, NULL_PMF, NULL },
	{ OI_n_timeEnd, OI_c_TimeEnd, OI_r_Int, sizeof(int), offsetof(MyRes, timeEnd),
		OI_r_String, DefaultTimeEnd, NULL_PMF, NULL, NULL_PMF, NULL },
};


/*
 *	define the command line arguments
 */
static XrmOptionDescRec opTable[] = {
	{"-start",	"*timeStart",	XrmoptionSepArg, (caddr_t)NULL },
	{"-end",	"*timeEnd",	XrmoptionSepArg, (caddr_t) NULL},
} ;


/*
 *	main
 *
 *	This program demonstrates
 *		- initializing OI, opening a connection to the server
 *		- fetching additional resources
 *		- create the application window.
 *		- create and layout entry fields.
 */
main (int argc, char **argv)
{
	OI_connection	*conp;			/* the connection to the server */
	OI_app_window	*appWindow;		/* the enclosing app window */
	OI_entry_field	*entryField;		/* used to create each entry field */
	int		time, row;		/* the current time and the row in which to put the entry field */
	char		*label;			/* the label for each entry field */
	char		*convertTime(int *);	/* convert a number into a time string */

	/*
	 *	Open a connection to the server.
	 *	OI_init also logs the command line arguments specified in the opTable.
	 */
	if ((conp = OI_init(&argc, argv, "calendar", "Calendar", opTable, OI_count(opTable)))) {
		/*
		**	fetch resources, as noted in the resource array, into the MyResources array.
		*/
		conp->get_resources(resources, OI_count(resources), (char *)&MyResources);

		/*
		 *	Create the calendar main window.
		 *	Make it row layout.
		 */
		appWindow = oi_create_app_window("myapp", 10, 10, "Calendar");
		appWindow->set_layout(OI_layout_row);

		/*
		 *	Present Entry Fields for each unit of time,
		 *		from start till end, delta time interval.
		 *	Layout these entry fields within the application window.
		 */
		for (time = MyResources.timeStart, row = 0; time <= MyResources.timeEnd; time += DefaultTimeInterval) {
			label = convertTime(&time);
			entryField = oi_create_entry_field("timeSlot", 30, label, NULL, 30);
			entryField->layout_associated_object(appWindow, 0, row++, OI_ACTIVE);
		}

		/*
		 *	display main window, then begin interaction.
		 */
		appWindow->set_associated_object(conp->root(), OI_DEF_LOC, OI_DEF_LOC, OI_ACTIVE);
		OI_begin_interaction();
	}

	/*
	 * Cleanup.  Make sure that we cleanup the library.
	 */
	OI_fini();
	return( 0 );
}


/*
 *	convert an integer time into text military time (HH:MM)
 */
char *
convertTime(int *time)
{
	static char label[20];
	int hour;
	int minute;

	hour = *time/100;
	minute = *time - (hour*100);
	if (minute >= 60) {
		minute -= 60;
		hour += 1;
	}
	*time = hour*100 + minute;
	sprintf(label, "%02d:%02d: ", hour, minute);
	return (label);
}
