/***
 *** configfile.c: funtions to process config file, and some more general parsing functions
 *** Written by Koen Gadeyne (kmg@barco.be)
 ***
 *** Version: 0.2
 ***/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "configfile.h"
#include "messages.h"


void cleanupstring(char *tstring)
{
     char *p=tstring;
     while (*p) 
     {
        switch (*p)
          {
             case '#':              /* discard remarks */
             case '\n': *p = '\0';  /* convert \n terminated strings to \0 terminated ones */
                        break;
             case '\t':             /* TABS to spaces */
             case '"': *p = ' ';    /* convert " (quotes) to spaces */
                       break;
          }
          if (*p == '\0') break;    /* stop processing */
          p++;
     }
}


char* findlabel(FILE* inputfile, char* reqlabel)
{
   /* look for 'reqlabel' in config file, exit if not found */
   static char video_param_str[1024];
   char inlabel[256]="";
   int nfound=1;
   
   PDEBUG(("findlabel: Looking for label '%s' in Configfile '%s'", reqlabel, CONFIGFILE));
   rewind(inputfile);
   while ( fgets(video_param_str,1024,inputfile) != NULL )
   {
      cleanupstring(video_param_str);
      sscanf(video_param_str, "%s",inlabel);
      if (!(nfound = strcasecmp(inlabel,reqlabel))) break;
   }
   if (nfound) PERROR(("Could not find '%s' line in config file '%s'. Aborting...",reqlabel,CONFIGFILE))
   else PDEBUG(("findlabel: found the following '%s' line:\n--> %s", reqlabel, video_param_str));
   return(video_param_str);
}


FILE* open_param_file(char* conf_file)
{
  FILE* param_file;
  PDEBUG(("Opening parameter file '%s'",conf_file));
  if ((param_file = fopen(conf_file,"r")) == NULL)
  {
      perror("fopen");
      PERROR(("Could not open Text mode config file '%s'",conf_file));
  }
  return(param_file);
}


char* showopts(const char* optstrings[], int num_optstrings)
{
  int i;
  static char optionstring[256];
  for (i=0; i<num_optstrings; i++)
  {
     strcat(optionstring, optstrings[i]);
     strcat(optionstring, (i==num_optstrings-1) ? "." : ", ");
  }
  return optionstring;
}


int findoption(char* inputstr, const char* optstrings[], int num_optstrings, char* optiontype)
{
  /* look which option string from given list matches input string, return option number, or exit when not found */
  int i, foundindex=-1;
  PDEBUG(("findoption: Looking for one out of %d %s option string(s) in '%s'",num_optstrings, optiontype, inputstr));
  for (i=0; i<num_optstrings; i++) { if (!strcasecmp(inputstr,optstrings[i])) foundindex=i; };
  if ((foundindex < 0) || (foundindex > num_optstrings))
    PERROR(("Unknown %s definition '%s'.
     Valid %s definitions are: %s ",optiontype, inputstr, optiontype, showopts(optstrings, num_optstrings)));
  PDEBUG(("findoption: Selecting %s #%d = %s", optiontype, foundindex, optstrings[foundindex]));
  return(foundindex);
}


void check_int_range(int cvalue, int lmin, int lmax, char* descstr)
{
  if (cvalue<lmin || cvalue>lmax)
    PERROR(("%s = %d (0x%x) out of range [%d..%d]!", descstr, cvalue, cvalue, lmin, lmax));
}

int getbyte(char* instring, char* descrstring, int lmin, int lmax)
  /* convert the byte in 'instring' into an integer. Must be within specified limits 'lmin' and 'lmax' */
  /* 'descrstring' contains a description of the number to be parsed, used in error message */
{
  char** errstr=NULL;
  int readbyte;
  readbyte = strtol(instring,errstr,0);
  if (errstr != NULL) PERROR(("Illegal character '%c' in %s: '%s'", *errstr, descrstring, instring));
  check_int_range(readbyte, lmin, lmax, descrstring);
  return(readbyte);
}

float getfloat(char* instring, char* descrstring, int lmin, int lmax)
  /* convert the float in 'instring' into a float. Must be within specified INTEGER limits 'lmin' and 'lmax' */
  /* 'descrstring' contains a description of the number to be parsed, used in error message */
{
  char** errstr=NULL;
  double readfloat;
  readfloat = strtod(instring,errstr);
  if (errstr != NULL) PERROR(("Illegal character '%c' in %s: '%s'", *errstr, descrstring, instring));
  check_int_range(readfloat, lmin, lmax, descrstring);
  return((float)readfloat);
}

