
%{

/**
 *
 * @File:	lua_scanner.ll
 * @Title:	Scanner (lexer) for the Lua Language
 *				to be used with Source-Highlight
 * @Author:	Marc Ct
 * @Date:	2003-11-17
 *
 **/

static int lineno = 1;		/* number of scanned lines */
static int stack[ 1024 ];
static int sp = 0;

static void YYBEGIN( int state )
{
	if( sp > 1023 )
	{	fprintf( stderr, "Stack Overflow in lexer.\n" );
		exit(1);
	}
//	fprintf( stderr, "push: %d\n", ((yy_start - 1) / 2) );
//	fprintf( stderr, "start is now: %d\n", state );
	stack[ sp++ ] = ((yy_start - 1) / 2);
	yy_start = 1 + 2 * state;
}


static void YYPOP( void )
{
	if( sp < 1 )
	{	fprintf( stderr, "Stack Underflow in lexer.\n" );
		exit(1);
	}
	yy_start = 1+2*stack[--sp];
//	fprintf( stderr, "pop, start is now: %d\n", ((yy_start - 1) / 2) );	
}

#include "genfun.h"


void generateEmail( char * text )
{
	generateKeyWord( text );
}

void generateOperator( char * text )
{
	generateSymbol( text );
}
	
%}

%option prefix = "lua_scanner_"
%option noyywrap

W					[ \t\v\f]+
O					[ \t\v\f]*
id					[a-zA-Z_][$a-zA-Z0-9_]*
n1					0[xX][0-9a-fA-F]*
n2					[0-9_]+(\.[0-9_]*[eE]?[-+]?[0-9_]*)?
number			{n1}|{n2}
nl 				\n
cr 				\r
tab				\t
emailid			[a-zA-Z_][$a-zA-Z0-9_.]*
email				"<"{O}{emailid}@{emailid}{O}">"

%x					LONGSTRING
%x					LONGSTRINGR
%x					SHORTCOMMENT
%x					LONGCOMMENT
%x					LONGCOMMENTR
%x					LITERAL
%x					STRING

%%

and |
break |
do |
else |
elseif |
end |
false |
for |
function |
if |
in |
local |
nil |
not |
or |
repeat |
return |
then |
true |
until |
while				generateKeyWord( yytext );

{id}:{id}/"("	generateFunction( yytext );
{id}/"("			generateFunction( yytext );
{id}				generate_normal( yytext );
{number}                        generateNumber( yytext );

"--[["			startComment(yytext); YYBEGIN( LONGCOMMENT );
"--"				startComment(yytext); YYBEGIN( SHORTCOMMENT );
"[["				startString(yytext); YYBEGIN( LONGSTRING );
\"					startString(yytext); YYBEGIN( STRING );
\'					startString(yytext); YYBEGIN( LITERAL);
"..."				generateOperator( yytext );
".."				generateOperator( yytext );
"."				generateOperator( yytext );
"=="				generateOperator( yytext );
"="				generateOperator( yytext );
"<="				generateOperator( yytext );
"<"				generateOperator( yytext );
">="				generateOperator( yytext );
">"				generateOperator( yytext );
"~="				generateOperator( yytext );
"~"				generateOperator( yytext );
"("				generateOperator( yytext );
")"				generateOperator( yytext );
"["				generateOperator( yytext );
"]"				generateOperator( yytext );
","				generateOperator( yytext );
"+"				generateOperator( yytext );
"-"				generateOperator( yytext );
"*"				generateOperator( yytext );
"/"				generateOperator( yytext );
"{"				generateCBracket( yytext );
"}"				generateCBracket( yytext );
\f					;
{tab}				generateTab( );
{cr}{nl}			lineno++; generateNewLine( );
{nl}				lineno++; generateNewLine( );

.					generate_preproc( yytext );


<LONGSTRING>
{
	"]]"			endString( yytext ); YYPOP( );
	"[["			generate_preproc( yytext ); YYBEGIN( LONGSTRINGR );
	{email}		endComment("");generateEmail(yytext);startComment("");
	\n				endString("");generateNewLine();startString("");lineno++;
	.				generate_preproc( yytext );
}

<LONGSTRINGR>
{
	"]]"			generate_preproc( yytext ); YYPOP( );
	"[["			generate_preproc( yytext ); YYBEGIN( LONGSTRINGR );
	{email}		endComment("");generateEmail(yytext);startComment("");
	\n				endString("");generateNewLine();startString("");lineno++;
	.				generate_preproc( yytext );
}


<SHORTCOMMENT>
{
	\n				endComment("");generateNewLine();lineno++;YYPOP( );
	\t				endComment("");generateTab();startComment("");
	{email}		endComment("");generateEmail(yytext);startComment("");
	.				generate_preproc( yytext );
}

<LONGCOMMENT>
{
	"]]"			endComment(yytext); YYPOP( );
	"[["			generate_preproc(yytext);YYBEGIN( LONGCOMMENTR );
	{email}		endComment("");generateEmail(yytext);startComment("");
	\n				endComment("");generateNewLine();startComment("");lineno++;
	\t				endComment("");generateTab();startComment("");
	.				generate_preproc( yytext );
}

<LONGCOMMENTR>
{
	"]]"			generate_preproc( yytext ); YYPOP( );
	"[["			generate_preproc( yytext ); YYBEGIN( LONGCOMMENTR );
	{email}		endComment("");generateEmail(yytext);startComment("");
	\n				endComment("");generateNewLine();startComment("");lineno++;
	\t				endComment("");generateTab();startComment("");
	.				generate_preproc( yytext );
}

<STRING>
{
	\"\"			yymore();
	\\.			yymore();
	\"				endString( yytext ); YYPOP( );
	.				yymore();
}

<LITERAL>
{
	\'\'			yymore();
	\\.			yymore();
	\'				endString( yytext ); YYPOP( );
	.				yymore();
}

%%

/* eof */

    