/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var sinpi = require( './../../../../base/special/sinpi' );
var isnan = require( './../../../../base/assert/is-nan' );
var isInfinite = require( './../../../../base/assert/is-infinite' );
var PI = require( '@stdlib/constants/float64/pi' );


// MAIN //

/**
* Computes the normalized cardinal sine of a number.
*
* ## Method
*
* For \\( x \neq 0 \\), the normalized cardinal sine is calculated as
*
* ```tex
* \operatorname{sinc}(x) = \frac{\operatorname{sin}(\pi x)}{\pi x}.
* ```
*
*
* ## Special Cases
*
* ```tex
* \begin{align*}
* \operatorname{sinc}(0) &= 1 & \\
* \operatorname{sinc}(\infty) &= 0 & \\
* \operatorname{sinc}(-\infty) &= 0 & \\
* \operatorname{sinc}(\mathrm{NaN}) &= \mathrm{NaN}
* \end{align*}
* ```
*
*
* @param {number} x - input value
* @returns {number} cardinal sine
*
* @example
* var v = sinc( 0.5 );
* // returns ~0.637
*
* @example
* var v = sinc( -1.2 );
* // returns ~-0.156
*
* @example
* var v = sinc( 0.0 );
* // returns 1.0
*
* @example
* var v = sinc( NaN );
* // returns NaN
*/
function sinc( x ) {
	if ( isnan( x ) ) {
		return NaN;
	}
	if ( isInfinite( x ) ) {
		return 0.0;
	}
	if ( x === 0.0 ) {
		return 1.0;
	}
	return sinpi( x ) / ( PI*x );
}


// EXPORTS //

module.exports = sinc;
