//
//  XTPendingWhitespaceQueueTests.m
//  XTads
//
//  Created by Rune Berg on 11/11/2016.
//  Copyright © 2016 Rune Berg. All rights reserved.
//

#import <XCTest/XCTest.h>
#import "XTPendingWhitespaceQueue.h"


#define UNICHAR_QUOTED_SPACE 0x15
#define UNICHAR_EN_SPACE 8194
#define UNICHAR_EM_SPACE 8195
#define UNICHAR_FIG_SPACE 8199
#define EXPANDED_EN_SPACE @"  "
#define EXPANDED_EM_SPACE @"    "
#define EXPANDED_FIG_SPACE @"  "
#define PRE_EXPANDED_TAB @" \t"


@interface XTPendingWhitespaceQueueTests : XCTestCase

@property XTPendingWhitespaceQueue *queue;
@property NSAttributedString* regularSpace;
@property NSAttributedString* quotedSpace;
@property NSAttributedString* enSpace;
@property NSAttributedString* emSpace;
@property NSAttributedString* figSpace;
@property NSAttributedString* tab;

@end


@implementation XTPendingWhitespaceQueueTests

- (void)setUp {
    [super setUp];
	self.queue = [XTPendingWhitespaceQueue new];
	self.regularSpace = [[NSAttributedString alloc] initWithString:@" "];
	self.quotedSpace = [[NSAttributedString alloc] initWithString:[NSString stringWithFormat:@"%C", UNICHAR_QUOTED_SPACE]];
	self.enSpace = [[NSAttributedString alloc] initWithString:[NSString stringWithFormat:@"%C", UNICHAR_EN_SPACE]];
	self.emSpace = [[NSAttributedString alloc] initWithString:[NSString stringWithFormat:@"%C", UNICHAR_EM_SPACE]];
	self.figSpace = [[NSAttributedString alloc] initWithString:[NSString stringWithFormat:@"%C", UNICHAR_FIG_SPACE]];
	self.tab = [[NSAttributedString alloc] initWithString:PRE_EXPANDED_TAB];
}

- (void)test_append_initialState {
	
	XCTAssertEqual(self.queue.length, 0);
}

//-------

- (void)test_append_basic {
	
	[self.queue append:self.regularSpace];
	XCTAssertEqual(self.queue.length, 1);
}

//-------

- (void)test_reset_1 {
	
	[self.queue reset];
	XCTAssertEqual(self.queue.length, 0);
}

- (void)test_reset_2 {

	[self.queue append:self.regularSpace];
	XCTAssertEqual(self.queue.length, 1);
	
	[self.queue reset];
	XCTAssertEqual(self.queue.length, 0);
}

- (void)test_reset_3 {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	XCTAssertEqual(self.queue.length, 2);
	
	[self.queue reset];
	XCTAssertEqual(self.queue.length, 0);

	[self.queue append:self.regularSpace];
	XCTAssertEqual(self.queue.length, 1);
	
	[self.queue reset];
	XCTAssertEqual(self.queue.length, 0);
}

//-------

- (void)test_combinedWhitespace_emptyQueue {
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	XCTAssertEqual(combdWs.count, 0);
}

//-------

- (void)test_combinedWhitespace_s { // s: one regular space
	
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_ss { // ss: two regular spaces
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_ssss {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

//----------

- (void)test_combinedWhitespace_q { // q: quoted space
	
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_qq {
	
	[self.queue append:self.quotedSpace];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_qqqqq {
	
	[self.queue append:self.quotedSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 5);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[2];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[3];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[4];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

//-------

- (void)test_combinedWhitespace_sq {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_ssq {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_sssq {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

//-----

- (void)test_combinedWhitespace_qs {
	
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_qss {
	
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_qsss {
	
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

//-----

- (void)test_combinedWhitespace_sqs {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_ssqss {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_sqqs {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

//-----

- (void)test_combinedWhitespace_qsq {
	
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_qssq {
	
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

//-----

- (void)test_combinedWhitespace_qsqsq {
	
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 3);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[2];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_sqsqs {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_sqqsqqqsq {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 6);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[2];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[3];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[4];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[5];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_combinedWhitespace_ssqsssqqss {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.quotedSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 3);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
	attrStr = combdWs[2];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

//-------

- (void)test_combinedWhitespace_n { // n: en space
	
	[self.queue append:self.enSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
}

- (void)test_combinedWhitespace_nn {
	
	[self.queue append:self.enSpace];
	[self.queue append:self.enSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
}

- (void)test_combinedWhitespace_nnnnn {
	
	[self.queue append:self.enSpace];
	[self.queue append:self.enSpace];
	[self.queue append:self.enSpace];
	[self.queue append:self.enSpace];
	[self.queue append:self.enSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 5);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
	attrStr = combdWs[2];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
	attrStr = combdWs[3];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
	attrStr = combdWs[4];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
}

- (void)test_combinedWhitespace_nsnnssnn {
	
	[self.queue append:self.enSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.enSpace];
	[self.queue append:self.enSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.enSpace];
	[self.queue append:self.enSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 5);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
	attrStr = combdWs[2];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
	attrStr = combdWs[3];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
	attrStr = combdWs[4];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
}

//-------

- (void)test_combinedWhitespace_m { // m: em space
	
	[self.queue append:self.emSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(EXPANDED_EM_SPACE, attrStr.string);
}

- (void)test_combinedWhitespace_mm {
	
	[self.queue append:self.emSpace];
	[self.queue append:self.emSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(EXPANDED_EM_SPACE, attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(EXPANDED_EM_SPACE, attrStr.string);
}

- (void)test_combinedWhitespace_smsss {
	
	[self.queue append:self.regularSpace];
	[self.queue append:self.emSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(EXPANDED_EM_SPACE, attrStr.string);
}

//-------

- (void)test_combinedWhitespace_f { // f: fig space
	
	[self.queue append:self.figSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(EXPANDED_FIG_SPACE, attrStr.string);
}

//-------

- (void)test_combinedWhitespace_nsm {
	
	[self.queue append:self.enSpace];
	[self.queue append:self.regularSpace];
	[self.queue append:self.emSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(EXPANDED_EN_SPACE, attrStr.string);
	attrStr = combdWs[1];
	XCTAssertEqualObjects(EXPANDED_EM_SPACE, attrStr.string);
}

//-------

- (void)test_combinedWhitespace_g // g: inline tag
{
	[self.queue appendInlineTag];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	
	XCTAssertEqual(combdWs.count, 0);
}

- (void)test_combinedWhitespace_sgs
{
	[self.queue append:self.regularSpace];
	[self.queue appendInlineTag];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

//-------

- (void)test_inlineTag_qgs
{
	// "\ <b> " -> " "
	
	[self.queue append:self.quotedSpace];
	[self.queue appendInlineTag];
	[self.queue append:self.regularSpace];

	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_inlineTag_qggs
{
	// "\ <b></b> " -> " "
	
	[self.queue append:self.quotedSpace];
	[self.queue appendInlineTag];
	[self.queue appendInlineTag];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_inlineTag_qgsgs
{
	// "\ <b> </b> " -> " "
	
	[self.queue append:self.quotedSpace];
	[self.queue appendInlineTag];
	[self.queue append:self.regularSpace];
	[self.queue appendInlineTag];
	[self.queue append:self.regularSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 1);
	
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_inlineTag_sgq
{
	// " <b>\ " -> "  "
	
	[self.queue append:self.regularSpace];
	[self.queue appendInlineTag];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_inlineTag_sggq
{
	// " <b></b>\ " -> "  "
	
	[self.queue append:self.regularSpace];
	[self.queue appendInlineTag];
	[self.queue appendInlineTag];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_inlineTag_sgsgq
{
	// " <b></b>\ " -> "  "
	
	[self.queue append:self.regularSpace];
	[self.queue appendInlineTag];
	[self.queue append:self.regularSpace];
	[self.queue appendInlineTag];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_inlineTag_sgsq
{
	// " <b> \ " -> "  "
	
	[self.queue append:self.regularSpace];
	[self.queue appendInlineTag];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];

	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

- (void)test_inlineTag_qgsq
{
	// "\ <b> \ " -> "  "
	
	[self.queue append:self.quotedSpace];
	[self.queue appendInlineTag];
	[self.queue append:self.regularSpace];
	[self.queue append:self.quotedSpace];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	
	XCTAssertEqual(combdWs.count, 2);
	
	attrStr = combdWs[0];
	XCTAssertEqualObjects(@" ", attrStr.string);
	
	attrStr = combdWs[1];
	XCTAssertEqualObjects(@" ", attrStr.string);
}

//-------

- (void)test_combinedWhitespace_t  // t = pending tab
{
	// "<tab>"

	[self.queue appendTab:self.tab];

	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	NSUInteger idx = 0;
	
	XCTAssertEqual(combdWs.count, 1);
	
	attrStr = combdWs[idx++];
	XCTAssertEqualObjects(PRE_EXPANDED_TAB, attrStr.string);
}

- (void)test_combinedWhitespace_tt
{
	// "<tab><tab>"
	
	[self.queue appendTab:self.tab];
	[self.queue appendTab:self.tab];
	
	NSMutableArray *combdWs = [self.queue combinedWhitespace];
	NSAttributedString *attrStr;
	NSUInteger idx = 0;
	
	XCTAssertEqual(combdWs.count, 2);
	
	attrStr = combdWs[idx++];
	XCTAssertEqualObjects(PRE_EXPANDED_TAB, attrStr.string);
	
	attrStr = combdWs[idx++];
	XCTAssertEqualObjects(PRE_EXPANDED_TAB, attrStr.string);
}

//TODO more...

@end
