/*
** Arr_h.t defines all the new verbs and classes for Arrival.
**
** Copyright (c) 1998, Stephen Granade. All rights reserved.
*/

#pragma C+

// In arr_ship.t, there is a universal translator object (named, appropriately
// enough, universal_translator). It has a 'dialect filter' button
// (translator_button) which, when pressed, makes the aliens talk in Valley
// Speak. This #define allows me to see if that button is pressed w/o having
// to type (translator_button.isPushed)
#define VALLEYSPEAK         (translator_button.isPushed)


/*
** Whenever you create a new verb, it's a good idea to put a default verb
** handler in thing. Using the 'modify' keyword, I'm going to do just that.
*/
modify thing
    doBreak(actor) = {
        "Haven&rsquo;t you given up on temper tantrums yet? ";
    }
    verDoWhois(actor) = { "\^<<self.thedesc>> <<self.isdesc>>n&rsquo;t
        actually a who. "; }
    verDoWhatis(actor) = {}
    doWhatis(actor) = (self.whatdesc)
    whatdesc = "Looking at <<self.itobjdesc>> should tell you all you
        need to know. "
    verDoCount(actor) = { "No need. "; }
    verDoDry(actor) = { "You don&rsquo;t need to dry <<self.thedesc>>. "; }
    verDoRub(actor) = { "There&rsquo;s no need, Aladdin. "; }
    verDoGoto(actor) = { "You&rsquo;re already there. "; }
    verDoDisplay(actor) = {
        "There is no permanent picture associated with <<self.thedesc>>. ";
    }
    verDoFastenTo(actor, io) = {
        "There&rsquo;s no need to fasten <<self.thedesc>> to anything. ";
    }
    verIoFastenTo(actor) = {
        "There&rsquo;s no need to fasten <<self.thedesc>> to anything. ";
    }
    verDoScrewInto(actor, io) = {
        "There&rsquo;s no need to screw <<self.thedesc>> into anything. ";
    }
    verIoScrewInto(actor) = {
        "There&rsquo;s no need to screw <<self.thedesc>> into anything. ";
    }
    verDoUnscrewFrom(actor, io) = {
        "I don&rsquo;t know how to unscrew <<self.thedesc>>. ";
    }
    verIoUnscrewFrom(actor) = {
        "I don&rsquo;t know how to unscrew <<self.thedesc>>. ";
    }
    verDoSprayAt(actor, io) = {
        "You can&rsquo;t spray <<self.thedesc>> at anything. ";
    }
    verIoSprayAt(actor) = {}
    ioSprayAt(actor, dobj) = { dobj.doSprayAt(actor, self); }
    verDoSprayWith(actor, io) = {}
    verIoSprayWith(actor) = {
        "You can&rsquo;t spray anything with <<self.thedesc>>. ";
    }
    verDoPointAt(actor, io) = {
        "There is no need to do that. ";
    }
    verIoPointAt(actor) = {}
    ioPointAt(actor, dobj) = { dobj.doPointAt(actor, self); }
    verDoHidebehind(actor) = {
        "There is no need to hide behind <<self.thedesc>>. ";
    }
    verDoUnhidebehind(actor) = {
        "But %you're% not behind <<self.thedesc>>. ";
    }
    verDoSqueeze(actor) = {
        "There&rsquo;s no need. ";
    }
    verDoThrow(actor) = {
        "There&rsquo;s no need. ";
    }
    verDoAttack(actor) = {
        "Temper, temper. ";
    }
    verDoAskFor(actor, io) = {
        "\^<<self.thedesc>> is unresponsive. ";
    }
    verIoAskFor(actor) = {}
    ioAskFor(actor, dobj) = { dobj.doAskFor(actor, self); }
    doSynonym('Turn') = 'TurnCW' 'TurnCCW'
;

modify Actor
    verDoWhatis(actor) = { "\^<<self.thedesc>> <<self.isdesc>>n&rsquo;t
        really a what. "; }
    verDoWhois(actor) = {}
    doWhois(actor) = (self.whodesc)
    whodesc = "There&rsquo;s nothing to say, really. "
;


/*
** Some functions to handle graphics &c.
*/

// Display an image. Call this function with the file name of the image.
// Usage: gprintf('image.jpg');
gprint: function(image)
{
    if (!global.graphicsOn)
        return;
    if (find(global.fileList, image) == nil)
        global.fileList += image;
    else if (global.graphicsOnce)
        return;
    "<IMG SRC=\"<<image>>\">";
}

// Display an image with extra formatting.
// Usage: gprintf('image.jpg', 'ALT="An image"');
gprintf: function(image, options)
{
    if (!global.graphicsOn)
        return;
    if (find(global.fileList, image) == nil)
        global.fileList += image;
    else if (global.graphicsOnce)
        return;
    "<IMG SRC=\"<<image>>\" <<options>> >";
}

// Create a link. Give this function the text to be linked and the command
// to which the text should be linked.
// Usage: link('refrigerator', 'x refrigerator');
link: function(contents, command)
{
    if (global.isHTMLRuntime)
        "<A HREF=\"<<command>>\"><<contents>></a>";
    else say(contents);
}

// shuffle() randomly shuffles a list. The algorithm is from Knuth.
shuffle: function(list)
{
    local i, retlist, tempitem, ndx;

    retlist = list;
    for (i = 1; i < length(list); i++) {
        ndx = _rand(length(list));
        tempitem = retlist[i];
        retlist[i] = retlist[ndx];
        retlist[ndx] = tempitem;
    }
    return retlist;
}

// inside returns true if obj is contained in loc, no matter how deeply nested
inside: function(ob, loc)
{
    local cur_loc;

    cur_loc = ob.location;
    while (cur_loc) {
        if (cur_loc == loc)
            return true;
        cur_loc = cur_loc.location;
    }
    return nil;
}

// moveList moves all items in a list to a destination. Note that this
// function won't move an item which has the isfixed property set to true
moveList: function(list, dest)
{
    local i, len;

    len = length(list);
    for (i = 1; i <= len; i++)
        if (!list[i].isfixed)
            list[i].moveInto(dest);
}

// Print a line of '*' to denote a page break.
pageBreak: function
{
    if (global.isHTMLRuntime)
        "<CENTER>* * * * *</CENTER>";
    else "\(* * * * *\)\b";
}

// A function for printing a string of random characters. The random list
// of characters has a bunch of extra 'x' and 'v' characters because everyone
// knows alien words are replete with them
randString: function(num)
{
    local chars = ['a' 'x' 'b' 'c' 'd' 'e' 'v' 'f' 'g' 'x' 'h' 'i' 'j'
        'k' 'l' 'm' 'v' 'n' 'o' 'x' 'p' 'q' 'r' 's' 't' 'u' 'v' 'w'
        'x' 'y' 'z'], i, len = length(chars), spaceCount = 0;

    "\^";
    for (i = 0; i < num; i++) {
        say(chars[_rand(len)]);
        if (spaceCount > 2 && i != (num - 1) &&
            (_rand(100) < 35 || spaceCount > 8)) {
            " ";
            spaceCount = 0;
        }
        else spaceCount++;
    }
    ".";
}

// There are two aliens in this game named Zigurt and Floban. They speak in
// a different font & color, so I've made these two functions to print what
// they say. Note that these functions automatically put quotation marks
// around the string you pass them. If you don't want quotes, call zigNQ and
// floNQ. Also note that, if their translator is turned off, all that is
// printed is a random string of characters and spaces.
zig: function(str)
{
    "<FONT FACE=\"Goudy, AGaramond\" COLOR=\"green\"><q>";
    if (!universal_translator.isOn)
        randString(length(str));
    else say(str);
    "</q></FONT>";
}

flo: function(str)
{
    "<FONT FACE=\"Goudy, AGaramond\" COLOR=\"purple\"><q>";
    if (!universal_translator.isOn)
        randString(length(str));
    else say(str);
    "</q></FONT>";
}

zigNQ: function(str)
{
    "<FONT FACE=\"Goudy, AGaramond\" COLOR=\"green\"><<str>></FONT>";
}

floNQ: function(str)
{
    "<FONT FACE=\"Goudy, AGaramond\" COLOR=\"purple\"><<str>></FONT>";
}


/*
** New classes and some global objects
*/

// Whenever the player is inside, I want him/her to be able to look at the
// ceiling or the floor. This class doesn't do anything other than mark
// the room as one that is inside and thus should have a floor and a ceiling.
class insideRm: room
;

// I need a specific type of insideRm for the house. Hence, houseRm
class houseRm: insideRm
;

// I want a way of knowing when the player is inside the aliens' ship, so
// I'm creating a class to handle that. This class also handles the 'goto'
// verb--see gotoVerb below for details.
class shipRoom: insideRm
    verDoGoto(actor) = {
        if (!startroom.landed)        // Can't go here if the ship isn't around
            "Okay, okay, you&rsquo;ve made your point&mdash;you&rsquo;ve
                played <i>Arrival</i>
                before. We&rsquo;re all quite impressed with you now.
                But think about it:\ how are you going to get there
                until the ship&rsquo;s arrived?
            \bSmart-alec. ";
        else if (!real_ship.rampDown) // Ditto if the ship's ramp is up
            "Not until the ship&rsquo;s ramp lowers. Be patient. ";
        else if (actor.location == self)   // Is the player already there?
            "You&rsquo;re already there. Wow! It&rsquo;t magic! ";
    }
    doGoto(actor) = {
        // This function is more byzantine than is strictly necessary, but I'm
        // feeling baroque today. There are three types of player locations
        // which must be handled:
        //
        // a) Outside the ship. Each room outside the ship has a property
        //    called "gotoDir" which has a pointer to the direction in
        //    which the player should move to get closer to the ship. We'll
        //    move the player to the ship's hallway one step at a time in this
        //    case
        // b) In another room of the ship. No room of the ship is more than
        //    one move away from the hallway, so we move the player directly
        //    to the hallway
        //
        // In both cases, the rooms have a 'gotoString' which contains the
        // name of the direction the player is moving in, so we can print
        // "(moving <direction>)" with each step.
        //
        // After we handle cases a) and b), the player's location should
        // fall under the last case:
        //
        // c) In the ship's hallway. In that case, go directly to the room,
        //    do not pass Go, do not collect $200. Oh, and print the direction
        //    in which the player is moving, which is in the final location's
        //    'gotoFinalString' property.
        
        if (!isclass(actor.location, shipRoom)) {       // Case a)
            local loc = actor.location;
            
            while (!isclass(loc, shipRoom)) {
                "(moving <<loc.gotoString>>)\n";
                actor.travelTo(loc.(loc.gotoDir));
                "\b";                         // Separate the room descs
                loc = actor.location;
            }
        }
        if (actor.location != ship_hallway) {           // Case b)
            "(moving <<actor.location.gotoString>>)\n";
            actor.travelTo(ship_hallway);
            "\b";
        }
        "(moving <<self.gotoFinalString>>)\n";
        actor.travelTo(self);                           // Case c)
    }
    // Because I've given most of the shipRooms a noun, I have to make sure
    // the player can look at the room (i.e. look around). I'll modify
    // their isVisible() code so that they're visible from inside them.
    isVisible(vantage) = {
        if (vantage.location == self) return true;
        pass isVisible;
    }
;

// The ceiling is a floatingItem (which means that it's available in more
// than one room) and it's a decoration (which means that anything the player
// tries to do to it other than look at it will result in a "Don't worry about
// the ceiling" message).
ceiling: floatingItem, decoration
    noun = 'ceiling'
// Here's how the ceiling's location works: check to see if the room
// that the player (Me) is in is of class insideRm. If so, return the player's
// location so that the player can refer to the ceiling. Otherwise, pretend
// that the ceiling is nowhere.
    location = {
        if (isclass(Me.location, insideRm))
            return Me.location;
        else return nil;
    }
    sdesc = "ceiling"
    ldesc = "The ceiling is above you. "
;


/*
** Modified adv.t objects
*/

// I don't want the player to have to muck about with his or her inventory,
// so I side-step the entire problem by removing items' bulk and weight
modify item
    weight = 0
    bulk = 0
;

// I also want the responses an Actor makes to be enclosed in nice quotes, not
// plain old '\"' quotes, so I'm modifying movableActor. Yeah, I know it's
// being picky. So sue me.
// [N.B. This code is ripped from adv.t]
modify movableActor
    doAskAbout(actor, iobj) = {
        local lst, i, tot;
        
        lst = objwords(2);       // get actual words asked about
        tot = length(lst);
        if ((tot == 1 && (find(['it' 'them' 'him' 'her'], lst[1]) != nil))
            || tot == 0)
        {
            self.specifyMore;   // I'm changing this so individual actors
                                // can override this message.
            return;
        }
        
        // try to find a response for each word
        for (i = 1 ; i <= tot ; ++i)
        {
            if (self.askWord(lst[i], lst))
                return;
        }
        
        // didn't find anything to talk about
        self.disavow;
    }
    specifyMore = "<q>Could you be more specific?<q>"
    disavow = "<q>I don&rsquo;t know much about that.</q>"
;

// While I'm at it, I'll modify basicStrObj
// [N.B. This code is ripped from adv.t]
modify basicStrObj
    sdesc = "<q><<value>></q>"
    adesc = "<q><<value>></q>"
    thedesc = "<q><<value>></q>"
    doSay(actor) = {
        "Okay, <q>"; say( self.value ); "</q>.";
    }
;

// I'm going to muck about some with the status line of rooms. Why? Because
// I like the status line to print the string 'Score: #  Turns: #' instead
// of the '#/#' default. In fact, for this game, I'm going to use a graphic
// representation of tick-marks for the score. I'm also going to allow the
// player to click on the room title in the status bar to look around.
// [N.B. This code is (mostly) ripped from adv.t]
modify room
    statusLine = {
        if (systemInfo(__SYSINFO_SYSINFO) == true
            && systemInfo(__SYSINFO_HTML) == 1)
        {
            "<banner id=StatusLine border><body
            bgcolor=statusbg text=statustext>";
            "<table width=\"100%\"><tr><td align=\"LEFT\" valign=\"TOP\"
            width=\"20%\">";
            "<b><a plain href=\"Look\"><<self.statusRoot>></a></b></td>";
            if (compass.ison)
                "<td align=\"CENTER\"
                    width=\"60%\"><<compass.drawCompass(self)>></td>";
            "<td align=\"RIGHT\" valign=\"TOP\"><i>Score:\ ";
            if (global.score == 0 || !global.isHTMLRuntime)
                "<<global.score>> ";
            else {
                "<img src=\"Resources/Score<<global.score>>.jpg\" align=top> ";
            }
            "Turns:\ <<global.turnsofar>></i></td></tr></table></banner>";
        }
        else
        {
            self.statusRoot; "\n\t";
        }
    }
;

// One problem: non-HTML-enabled runtimes still use scoreStatus, so I have
// to change that as well.
replace scoreStatus: function(points, turns)
{
    setscore('Score: '+cvtstr(points)+' Turns: '+cvtstr(turns));
}

// I think that you should not only be able to switch a switch, but that
// you should be able to "throw switch" and "flip switch". doSynonym()
// tells TADS that any time doThrow or doFlip is called in a switch item,
// doSwitch should handle things. In effect, I'm making the verbs "throw"
// and "flip" mean the same thing as "switch" for any switchItem.
modify switchItem
    doSynonym('Switch') = 'Throw' 'Flip'
;

// For atmospheric reasons, I often like to modify the boring "You can't have
// that" message that fixeditems print, so now each fixeditem has a takedesc.
modify fixeditem
    verDoTake(actor) = { self.takedesc; }
    takedesc = "%You% can&rsquo;t have <<self.thedesc>>. "
;

// You should be able to ask "What is..." about decorations and distantItems,
// so I need to modify their dobjGen.
modify distantItem
    dobjGen(a, v, i, p) =
    {
        if (v != askVerb && v != tellVerb && v != inspectVerb &&
            v != whatisVerb) {
            "It's too far away.";
            exit;
        }
    }
;

modify decoration
    dobjGen(a, v, i, p) =
    {
        if (v != inspectVerb && v != whatisVerb) {
            "\^<<self.thedesc>> <<self.isntdesc>> important.";
            exit;
        }
    }
;


/*
** New compound word--outfrombehind
*/
compoundWord 'out' 'from' 'outfrom';
compoundWord 'outfrom' 'behind' 'outfrombehind';


/*
** New verb/prep section
*/

// I want to add a verb which allows the player to get information about
// the game. To do so, I have to define the verb's sdesc (printed under
// certain circumstances), and the vocabulary word of this verb. Since this
// verb will never be applied to an object (the player won't be able to
// say "about table", for example), I only need define the action() routine.
// I've defined this as a sysverb instead of a deepverb because this verb
// lies outside the normal game universe--like SAVE and RESTORE, it's a
// command which has no effect on the game itself.
aboutVerb: sysverb
    sdesc = "about"
    verb = 'about'
    action(actor) = {
"<b>Arrival</b> is copyright &copy; 1998 by Stephen Granade. It may be freely
copied, distributed and played as long as it is not modified in any way.
You may not distribute copies of the game for a fee exceeding the cost of
distribution.

\bYou may have seen a version of this game which claimed to be by Samantha
Clark. That&rsquo;s okay&mdash;<q>Samantha</q> was my pseudonym for the
4th Annual Interactive Fiction Competition.

\b<b>Arrival</b> has some special commands; you can find out
all about them by typing <strong><<link('COMMANDS', 'commands')>></strong>.
<strong><<link('HINT', 'hint')>></strong> will give you hints about
the game&rsquo;s puzzles; the <strong><<link('CREDITS', 'credits')>></strong>
command will list all applicable credits for <b>Arrival</b>.

\bThere is a chance you may not have the resource files which contain all of
<b>Arrival</b>&rsquo;s pictures and sounds. You can find out more about this on
my web page at http://www.phy.duke.edu/~sgranade/arrival.html.

\bI&rsquo;d love to hear what you thought of the game. I can be reached at
sgranade@phy.duke.edu. ";

        abort;    // This way the player doesn't lose a turn
    }
;

// I've left this in for historical purposes.
soopersekritaboutVerb: sysverb
    sdesc = "actual information"
    verb = 'printcopyinfo'
    action(actor) = {
"Say the secret woid and you win the actual copyright information.\b

<b>Arrival</b> is copyright &copy; 1998 by Stephen Granade. <q>Samantha
Clark</q> is a pseudonym for Stephen Granade, used when the game was
entered in the 1998 IF Author&rsquo;s Competition. <b>Arrival</b> may be
freely copied, distributed, and played as long as it is not modified in
any way. You may not distribute copies of the game for a fee exceeding
the cost of distribution.\b

Speaking of pseudonyms, several other ones are listed in the credits. My
beta testers were Dan Shiovitz (<q>Rodney Spilheim</q>), Michael Self
(<q>Anderson Howe</q>), and Michael Kinyon (<q>Emmy Noether</q>).

In addition, the music <i>Alien Groove</i> is copyright &copy; 1998 by
Stephen Granade, and the <b>Arrival</b> logo was created in part by Misty
Granade. ";
    }
;

// Prints the special commands. Functionally, this is identical to aboutVerb.
commandsVerb: sysverb
    sdesc = "commands"
    verb = 'commands'
    describeMe = {
"ABOUT: Prints information about <b>Arrival</b>.

\bAGAIN or G: Repeats your last command. If your last input line was
composed of several commands, only the last command on the line is repeated.

\bCOMMANDS: Shows a list of these special commands.

\bCREDITS: Lists the credits for <b>Arrival</b>.

\bDEDICATION: Prints the dedication for <b>Arrival</b>.

\bDISPLAY: Shows or hides banners such as the compass banner.";

    // Only print this next part if we can handle graphics
    if (global.canDoGraphics)
"\bGRAPHICS: Prints whether graphics are shown always, never, or once.

\bGRAPHICS ONCE: Toggles whether each picture is show only once or not.";

"\bHINT: Gives hints about the game&rsquo;s puzzles.

\bINVENTORY or I: Shows the list of items you are carrying.

\bLOOK or L: Gives the full description of your location.

\bNOTIFY: Turns score notification on or off. Notification is on by default.

\bOOPS: Allows you to correct the spelling of a word in the last command. You
can use OOPS when the game displays this complaint: <q>I don&rsquo;t know
the word
&lt;word&gt;.</q>  Immediately after this message, you can type OOPS followed
by the
corrected spelling of the misspelled word. You can only type one word after
OOPS, so this command doesn&rsquo;t allow you to correct certain types of
errors, such as when you run two words together without a space.

\bQUIT or Q: Ends the game.

\bRESTART: Starts the game over from the beginning.

\bRESTORE: Restores a position previously saved with the SAVE command.

\bSAVE: Stores the current state of the game in a disk file, so that you can
come back to the same place later with the RESTORE command.

\bSCORE: Shows you your current score and the maximum possible score.

\bSCRIPT: Starts writing everything you see on the screen (both your commands
and the game&rsquo;s responses) to a disk file. The game will ask you for
a filename
to be used for the transcript; you should select a filename that does not yet
exist on your disk, because if you use an existing filename, data in that
file will be destroyed. Use the UNSCRIPT command to stop making the
transcript.

\bTERSE: For impatient users, this tells the game that you wish to see only
short descriptions of locations you have already seen when you reenter them.
See also the VERBOSE command.

\bUNDO: Tells the game you want to take back your last command. The game state
will be restored to the way it was before the previous command, as though the
command were never issued at all. You can do this more than once in a row.

\bUNSCRIPT: Turns off the transcript that was begun with the SCRIPT command.

\bVERBOSE: For amnesiac players, this tells the game to show you the full
description of every location you enter, whether or not you have seen the
description before. This is the default mode. See also the TERSE command.

\bVERSION: Shows the current version number of the game.

\bWAIT or Z: Causes game time to pass. When the game is waiting for you to type
a command, no game time passes; you can use this command to wait for something
to happen.

\bWALKTHROUGH: Prints a step-by-step walkthrough for the game.

\bWHO IS/WHAT IS: You can occasionally get more information about people and
things using this command. ";
    }
    action(actor) = {
        self.describeMe;
        abort;
    }
;

// I want to make sure I give credit where credit is due
creditsVerb: sysverb
    sdesc = "credits"
    verb = 'credits'
    describeMe = {
        version.sdesc;
        if (global.isHTMLRuntime)
            "<center><h3>Graphics</h3></center>";
        else "\b\(Graphics\)\b";
        "Some graphics were created using the finest crayons and modelling clay
            $2.83 could buy. Others were created using Adobe Illustrator 7.0.
            All were wonkified by Adobe Photoshop 5.0. Sounds were assembled
            in Cool Edit Pro 1.1.
            The <b>Arrival</b> poster and logo were created with the help
            of Misty Granade. ";

        if (global.isHTMLRuntime)
            "<center><h3>Music</h3></center>";
        else "\b\(Music\)\b";
        "<i>Alien Groove</i>\n
        Copyright &copy; 1998 by Stephen Granade\n
        Winner, Juno Award for Best Instrumental Piece<sup>*</sup>";

        if (global.isHTMLRuntime)
            "<center><h3>Other Shout-Outs</h3></center>";
        else "\b\(Other Shout-Outs\)\b";
        "Props to my beta-testers:\ Michael Self, Dan Shiovitz, and Michael
            Kinyon.";
        "\bThanks to everyone who made suggestions after
            <b>Arrival</b>&rsquo;s release:\ Cody Sandifer, Suzanne
            Skinner, David Glasser, and Mark Musante.";
        "\bMystery Science Theatre 3000. <q>Attack of the the Eye Creatures</q>
            anyone?";
        "\bI would especially like to thank Mike Roberts for the chance to add
            some graphics to my game, and for putting up with my endless
            stream of bug reports. ";

        "\b<sup>*</sup>No, not really. ";
    }
    action(actor) = {
        self.describeMe;
        abort;
    }
;

// Print the walkthrough
walkthruVerb: deepverb
    sdesc = "walkthru"
    verb = 'walkthru' 'walkthrough'
    commands = [ 'x dirty dishes', 'x clean dishes', 'x cabinets',
        'look out window', 'clean dish.g.g.g', 'x refrigerator',
        'open refrigerator', 'open cabinets', 'x sink', 'e', 'e',
        'x knick-knacks', 'look out window', 'tell dad about ship',
        'tell dad about ship', 'tell dad about ship', 'w', 'n', 'x ship',
        'x bumper stickers', 'x patio', 'x curtain', 'x zigurt', 'x floban',
        'x faucet', 'n', 'x plate', 'rub plate', 'x map', 'se', 'x junk',
        'search junk', 'again.g.g.g.g.g', 'x hose', 'nw', 'ne', 'x ovoids',
        'open ovoids', 'enter ovoids', 'undo', 'sw', 'n', 'x controls',
        'x console', 'x button', 'x lever', 'x gauge', 'push button',
        'x panel', 'push red triangle', 'push green square', 'again', 's',
        'sw', 'x tools', 'x panel', 'push lever', 'flip switch', 'turn knob',
        'pull controls', 'ne', 'nw', 'x atlas', 'x bottle', 'get pill',
        'x translator', 'x book', 'get book', 'x counter', 'x space',
        'enter space', 'i', 'x lid', 'x bottle', 'put lid on bottle',
        'get book', 'read it', 'stand', 'se', 'nw', 'get book', 'se', 'sw',
        'z', 'z', 'ne', 'n', 'x console', 'x panel', 'x note', 'hit panel',
        'x button', 'x lever', 'x gauge', 'push button', 'push lever', 's',
        's', 'x puddle', 'put hose on faucet', 'x hatch',
        'put hose in opening', 'drop hose', 'turn on faucet', 's', 'e',
        'ask mom about rock salt', 'ask mom about ho-hos', 'w', 'w',
        'open cabinets', 'x salt', 'x cupcakes', 'get salt and cupcakes',
        'eat cupcake', 'e', 'n', 'x lights', 'x control',
        'turn knob clockwise', 'z.z.z.z', 's', 'w', 'push clean dishes', 'z',
        'z', 'z', 'e', 'e', 'z', 'tell dad about light', 'get knick-knack',
        'w', 'n', 'x aliens', 'z', 'z', 'n', 'nw', 'give salt to floban',
        'give doll to floban', 'give cupcakes to zigurt', 'z', 'quit'
    ]
    action(actor) = {
        local key, i, len;

        "Reading the walkthrough is likely to stomp flat your enjoyment of
            this game. Are you sure? (y/n) ";
        if (yorn() != 1)
            abort;
        len = length(self.commands);
        for (i = 1; i <= len; i++) {
            "&gt;<<self.commands[i]>>\n";
        }
        "That&rsquo;s it! ";
        abort;
    }
;

// Allow the player to get hints
hintVerb: deepverb
    sdesc = "hint"
    verb = 'hint' 'help' 'hints'
    action(actor) = {
        top_menu.display;
        abort;
    }
;

// The game's dedication
dedicationVerb: deepverb
    sdesc = "dedication"
    verb = 'dedication'
    action(actor) = {
        "For Paul Price.\nConsider it payment for getting me addicted to
            MST3K. ";
    }
;

// Almost every game nowadays recognizes the verb 'xyzzy'. It was a magic
// word in the very first adventure game, Colossal Cave. Most games just
// print a smart-alek response.
xyzzyVerb: deepverb
    sdesc = "xyzzy"
    verb = 'xyzzy'
    action(actor) = {
        "You&rsquo;d think a fancy graphic game like this would show you a
            pretty picture of some room from Adventure, or at least something
            flashy for your trouble.\bYeah, well. ";
    }
;

// Some smart-alec player is bound to try this, so I'll add it in.
winVerb: deepverb
    sdesc = "win"
    verb = 'win'
    action(actor) = {
        theEnd('You have won\bSatisfying, isn&rsquo;t it?');
    }
;

// Get information about things via 'who is'/'what is'
isPrep: Prep
    preposition = 'is'
    sdesc = "is"
;

arePrep: Prep
    preposition = 'are'
    sdesc = "are"
;

amPrep: Prep
    preposition = 'am'
    sdesc = "am"
;

whoisVerb: deepverb
    sdesc = "who is"
    verb = 'who is' 'who are' 'who am'
    doAction = 'Whois'
    validDo(actor, obj, seqno) = { return true; }
    validDoList(actor, prep, iobj) = (nil)
;

whatisVerb: deepverb
    sdesc = "what is"
    verb = 'what is' 'what are' 'what am'
    doAction = 'Whatis'
    validDo(actor, obj, seqno) = { return true; }
    validDoList(actor, prep, iobj) = (nil)
;

oncePrep: Prep
    preposition = 'once'
    sdesc = "once"
;

graphicsVerb: sysverb
    sdesc = "graphics"
    verb = 'graphics'
    action(actor) = {
        if (!global.canDoGraphics) {
            "This runtime does not support graphics functions. ";
            return;
        }
        if (!global.graphicsOn)
            "Graphics are currently not being shown. ";
        else if (global.graphicsOnce)
            "Graphics are currently shown once. ";
        else "Graphics are currently being shown. ";
        abort;
    }
;

graphicsOnceVerb: sysverb
    sdesc = "graphics once"
    verb = 'graphics once'
    action(actor) = {
        if (!global.canDoGraphics) {
            "This runtime does not support graphics functions. ";
            return;
        }
        if (!global.graphicsOn)
            "You must first turn on graphic support. ";
        else {
            global.graphicsOnce = !global.graphicsOnce;
            "Graphics are now being shown ";
            if (!global.graphicsOnce) "more than ";
            "once. ";
        }
        abort;
    }
;

// Just for fun
countVerb: deepverb
    sdesc = "count"
    verb = 'count'
    doAction = 'Count'
;

// You can clean dishes, so you should be able to dry them
dryVerb: deepverb
    sdesc = "dry"
    verb = 'dry'
    doAction = 'Dry'
;

// To let the player rub the shiny metal plate in the ship
rubVerb: deepverb
    touch = true        // More information on 'touch' below
    sdesc = "rub"
    verb = 'rub'
    doAction = 'Rub'
;

// For moving quickly from place to place
gotoVerb: deepverb
    sdesc = "goto"
    verb = 'goto'
    doAction = 'Goto'
    // I have to modify validDo so that the player can 'goto'
    //  places which aren't in the same room as the player (!)
    validDo(actor, obj, seqno) = { return true; }
;

// For displaying pictures of items (when possible)
displayVerb: deepverb
    verb = 'display'
    sdesc = "display"
    doAction = 'Display'
;

// For spraying, pointing, and squirting the hose at stuff
sprayVerb: deepverb
    verb = 'spray' 'squirt'
    sdesc = "spray"
    ioAction(atPrep) = 'SprayAt'
    ioAction(withPrep) = 'SprayWith'
;

pointVerb: deepverb
    verb = 'point'
    sdesc = "point"
    ioAction(atPrep) = 'PointAt'
;

// For ducking behind the formica counter in the ship
hideVerb: deepverb
    verb = 'hide behind' 'get behind' 'go behind'
    sdesc = "hide behind"
    doAction = 'Hidebehind'
;

// For un-ducking behind said counter
unhideVerb: deepverb
    verb = 'get outfrombehind'
    sdesc = "get out from behind"
    doAction = 'Unhidebehind'
;

outfrombehindPrep: Prep
    preposition = 'outfrombehind'
    sdesc = "out from behind"
;

// For the stress ball
squeezeVerb: deepverb
    touch = true
    verb = 'squeeze' 'knead'
    sdesc = "squeeze"
    doAction = 'Squeeze'
;

forPrep: Prep
    preposition = 'for'
    sdesc = "for"
;

// For the current control knob
turnclockwiseVerb: deepverb
    touch = true
    sdesc = "turn clockwise"
    verb = 'turn clockwise' 'turn cw' 'turn right'
    doAction = 'TurnCW'
;

turnccwVerb: deepverb
    touch = true
    sdesc = "turn counterclockwise"
    verb = 'turn counterclockwise' 'turn ccw' 'turn anticlockwise' 'turn left'
    doAction = 'TurnCCW'
;

cwPrep: Prep
    sdesc = "clockwise"
    preposition = 'clockwise' 'cw' 'right'
;

ccwPrep: Prep
    sdesc = "counterclockwise"
    preposition = 'counterclockwise' 'anticlockwise' 'ccw' 'left'
;


/*
** I'm going to fiddle with some of the error messages TADS prints, to jazz
** up the game a bit.
*/
parseErrorParam: function(errNum, errMsg, ...)
{
    switch (errNum) {
        case 2:                 // I don't know the word "%s".
            return 'I never learned the word <q>'+getarg(3)+'</q> in school.';

        case 9:                 // I don't see any %s here.
            return ('I don&rsquo;t see no&mdash;I mean any&mdash;'+getarg(3)
                +' here.');

        case 12:                // You can only speak to one person at a time
            return 'Don&rsquo;t speak to more than one person at a time.
                It&rsquo;s rude. Unless your a politician or something.';

        case 17:                // There's no verb in that sentence!
            return 'You forgot a verb, not to mention what you learned in
                English class.';

        case 100:               // Let's try it again: 
            return 'They say children learn by repetition, so: ';

        default:
            return nil;
    }
}


/*
** I like it when the player's command is bold-faced, which I do using
** commandPrompt() and commandAfterRead(). I also hook into commandPrompt
** to handle score notifications--this way, the player is notified that their
** score went up AFTER everything else has happened.
*/
commandPrompt: function(code)
{
    scoreWatcher.heartbeat;     // Handle score notification
    "\b&gt;<b>";
}

commandAfterRead: function(code)
{
    "</b>";
}


/*
** I want the player to receive notification when their score increases.
*/
replace incscore: function( amount )
{
    global.score += amount;
    if (scoreWatcher.notifyMe != nil) {
        scoreWatcher.amt += amount;
    }
    scoreStatus( global.score, global.turnsofar );
}

scoreWatcher: object
  warned = nil
  notifyMe = true
  amt = 0
  heartbeat = {
    if (self.amt == 0)
      return;
    "\b<b>[Your score has <<(amt > 0) ? "increased" : "decreased">> by <<
      amt>> point";
    if ((amt > 1) || (amt < -1)) "s";
    if (self.warned == nil) {
// This message is only received the first time
      ".  You can turn these notifications off at any time by typing ";
      link('NOTIFY', 'notify');
      self.warned = true;
    }
    ".]</b>\n";
	self.amt = 0;
  }
;

notifyVerb: sysverb
  verb = 'notify'
  sdesc = "notify"
  action( actor ) = {
     "Notification is now ";
     if (scoreWatcher.notifyMe) {
       "off";
       scoreWatcher.notifyMe = nil;
     }
     else {
       "on";
       scoreWatcher.notifyMe = true;
     }
     ".  ";
  }
;


/*
** Modified adv.t verbs
*/

// I only want you to be able to use certain verbs with the word 'all'.
// This requires some fiddling with the rejectMultiDobj function in deepverb
modify deepverb
    rejectMultiDobj(prep) = {
        if (self.allowall || find(objwords(1), 'all') == nil)
            return nil;
        "You can&rsquo;t use <q>all</q> with this verb. ";
        abort;
    }
;

modify takeVerb
    allowall=true
;

// Besides allowing "all" to work with drop, allow "drop xxx in yyy" ==
//  "put xxx in yyy" and "drop xxx on yyy" == "put xxx on yyy".
modify dropVerb
    allowall=true
    ioAction(inPrep) = 'PutIn'
    ioAction(onPrep)='PutOn'
;

modify putVerb
    allowall=true
;

// I want the 'sleep' command to print a snappy response.
modify sleepVerb
    action(actor) = {
        "Are you kidding? Not until mom puts you to bed. No way you&rsquore
            gonna let her know you actually <i>want</i> to go to bed. ";
    }
;

// Add the synonym 'scream' to 'yell' and modify its action
modify yellVerb
    verb = 'scream'
    action(actor) = {
        if (isclass(actor.location, houseRm))
            "Not in the house, you don&rsquo;t! ";
        else if (zigurt.location == actor.location)
            "Zigurt glances at you, frowning, then sniffs. ";
        else if (actor.location == counter &&
            zigurt.location == rec_room) {
            "There is silence from the other side of the counter. Then
                you hear Floban say, ";
            if (VALLEYSPEAK)
                flo('Whoa. That&rsquo;s, like, a weird sound.');
            else flo('What an odd sound.');
        }
        else pass action;
    }
;

modify cleanVerb
    verb = 'scrub' 'wash' 'wipe'
;

modify turnOnVerb
    verb = 'light'
;

modify lookThruVerb
    verb = 'look out'    // So the player can 'look out window'
;

modify attachVerb
    verb = 'join' 'splice'
;

modify throwVerb        // So the player can 'throw the switch'
    doAction = 'Throw'
;

// Don't ask why I'm adding 'waylay' as a synonym for 'attack'. It's a long
// story. I'm also allowing you to attack things w/o specifying an iobj.
modify attackVerb
    verb = 'waylay' 'whack' 'smack'
    doAction = 'Attack'
;

modify breakVerb
    verb = 'smash'
;

// I want the player to be able to fasten items *to* other items
modify fastenVerb
    ioAction(toPrep) = 'FastenTo'
;

// I also want the player to be able to screw items *into* other items
modify screwVerb
    ioAction(inPrep) = 'ScrewInto'
;

// If you can screw things into other things, then you should be able
// to unscrew things from other things
modify unscrewVerb
    ioAction(fromPrep) = 'UnscrewFrom'
;

// Add 'ask <dobj> for <iobj>'
modify askVerb
    ioAction(forPrep) = 'AskFor'
;

// Put a flag in verbs which require you to touch the item. That way, if I
// want something special to happen if/when a player touches an item, I don't
// have to check for *every* verb. Instead, I just see if the verb has the
// touch flag set.
modify digVerb
    touch = true;
modify pushVerb
    touch = true;
modify attachVerb
    touch = true;
modify wearVerb
    touch = true;
modify dropVerb
    touch = true;
modify removeVerb
    touch = true;
modify openVerb
    touch = true;
modify closeVerb
    touch = true;
modify putVerb
    touch = true;
modify takeVerb
    touch = true;
modify plugVerb
    touch = true;
modify screwVerb
    touch = true;
modify unscrewVerb
    touch = true;
modify turnVerb
    touch = true;
modify switchVerb
    touch = true;
modify flipVerb
    touch = true;
modify turnOnVerb
    touch = true;
modify turnOffVerb
    touch = true;
modify sitVerb
    touch = true;
modify lieVerb
    touch = true;
modify breakVerb
    touch = true;
modify attackVerb
    touch = true;
modify climbVerb
    touch = true;
modify eatVerb
    touch = true;
modify drinkVerb
    touch = true;
modify giveVerb
    touch = true;
modify pullVerb
    touch = true;
modify throwVerb
    touch = true;
modify standOnVerb
    touch = true;
modify showVerb
    touch = true;
modify cleanVerb
    touch = true;
modify moveVerb
    touch = true;
modify fastenVerb
    touch = true;
modify unfastenVerb
    touch = true;
modify unplugVerb
    touch = true;
modify typeVerb
    touch = true;
modify lockVerb
    touch = true;
modify unlockVerb
    touch = true;
modify detachVerb
    touch = true;
modify pokeVerb
    touch = true;
modify touchVerb
    touch = true;
modify centerVerb
    touch = true;
modify searchVerb
    touch = true;

