/* This is the Assembler Pre-Processor
   Copyright (C) 1987-2026 Free Software Foundation, Inc.

   This file is part of GAS, the GNU Assembler.

   GAS is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   GAS is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
   License for more details.

   You should have received a copy of the GNU General Public License
   along with GAS; see the file COPYING.  If not, write to the Free
   Software Foundation, 51 Franklin Street - Fifth Floor, Boston, MA
   02110-1301, USA.  */

/* Modified by Allen Wirfs-Brock, Instantiations Inc 2/90.  */
/* App, the assembler pre-processor.  This pre-processor strips out
   excess spaces, turns single-quoted characters into a decimal
   constant, and turns the # in # <number> <filename> <garbage> into a
   .linefile.  This needs better error-handling.  */

#include "as.h"

#if (__STDC__ != 1)
#ifndef const
#define const  /* empty */
#endif
#endif

#ifdef H_TICK_HEX
int enable_h_tick_hex = 0;
#endif

#ifdef TC_M68K
/* Whether we are scrubbing in m68k MRI mode.  This is different from
   flag_m68k_mri, because the two flags will be affected by the .mri
   pseudo-op at different times.  */
static int scrub_m68k_mri;

/* The pseudo-op which switches in and out of MRI mode.  See the
   comment in do_scrub_chars.  */
static const char mri_pseudo[] = ".mri 0";
static const char *mri_state;
static char mri_last_ch;
#else
#define scrub_m68k_mri 0
#endif

#if defined TC_ARM && defined OBJ_ELF
/* The pseudo-op for which we need to special-case `@' characters.
   See the comment in do_scrub_chars.  */
static const char   symver_pseudo[] = ".symver";
static const char * symver_state;
#endif

/* The pseudo-op (without leading dot) at which we want to (perhaps just
   temporarily) stop processing.  See the comments in do_scrub_chars().  */
static const char   end_pseudo[] = "end ";
static const char * end_state;

/* Whether, considering the state at start of assembly, NO_PSEUDO_DOT is
   active.  */
static bool no_pseudo_dot;

static char last_char;

#define LEX_IS_SYMBOL_COMPONENT		1
#define LEX_IS_WHITESPACE		2
#define LEX_IS_LINE_SEPARATOR		3
#define LEX_IS_COMMENT_START		4
#define LEX_IS_LINE_COMMENT_START	5
#define	LEX_IS_TWOCHAR_COMMENT_1ST	6
#define	LEX_IS_STRINGQUOTE		8
#define	LEX_IS_COLON			9
#define	LEX_IS_NEWLINE			10
#define	LEX_IS_ONECHAR_QUOTE		11
#ifdef TC_V850
#define LEX_IS_DOUBLEDASH_1ST		12
#endif
#ifdef DOUBLEBAR_PARALLEL
#define LEX_IS_DOUBLEBAR_1ST		13
#endif
#define LEX_IS_PARALLEL_SEPARATOR	14
#ifdef H_TICK_HEX
#define LEX_IS_H			15
#endif
#define IS_SYMBOL_COMPONENT(c)		(lex[c] == LEX_IS_SYMBOL_COMPONENT)
#define IS_WHITESPACE(c)		(lex[c] == LEX_IS_WHITESPACE)
#define IS_LINE_SEPARATOR(c)		(lex[c] == LEX_IS_LINE_SEPARATOR)
#define IS_PARALLEL_SEPARATOR(c)	(lex[c] == LEX_IS_PARALLEL_SEPARATOR)
#define IS_COMMENT(c)			(lex[c] == LEX_IS_COMMENT_START)
#define IS_LINE_COMMENT(c)		(lex[c] == LEX_IS_LINE_COMMENT_START)
#define IS_TWOCHAR_COMMENT_1ST(c)	(lex[c] == LEX_IS_TWOCHAR_COMMENT_1ST)
#define	IS_NEWLINE(c)			(lex[c] == LEX_IS_NEWLINE)

static char lex[256] = {
  [' ']  = LEX_IS_WHITESPACE,
  ['\t'] = LEX_IS_WHITESPACE,
#ifdef CR_EOL
  ['\r'] = LEX_IS_LINE_SEPARATOR,
#else
  ['\r'] = LEX_IS_WHITESPACE,
#endif
  ['\n'] = LEX_IS_NEWLINE,
  [':'] = LEX_IS_COLON,
  ['$'] = LEX_IS_SYMBOL_COMPONENT,
  ['.'] = LEX_IS_SYMBOL_COMPONENT,
  ['_'] = LEX_IS_SYMBOL_COMPONENT,
  ['A'] = LEX_IS_SYMBOL_COMPONENT, ['a'] = LEX_IS_SYMBOL_COMPONENT,
  ['B'] = LEX_IS_SYMBOL_COMPONENT, ['b'] = LEX_IS_SYMBOL_COMPONENT,
  ['C'] = LEX_IS_SYMBOL_COMPONENT, ['c'] = LEX_IS_SYMBOL_COMPONENT,
  ['D'] = LEX_IS_SYMBOL_COMPONENT, ['d'] = LEX_IS_SYMBOL_COMPONENT,
  ['E'] = LEX_IS_SYMBOL_COMPONENT, ['e'] = LEX_IS_SYMBOL_COMPONENT,
  ['F'] = LEX_IS_SYMBOL_COMPONENT, ['f'] = LEX_IS_SYMBOL_COMPONENT,
  ['G'] = LEX_IS_SYMBOL_COMPONENT, ['g'] = LEX_IS_SYMBOL_COMPONENT,
  ['H'] = LEX_IS_SYMBOL_COMPONENT, ['h'] = LEX_IS_SYMBOL_COMPONENT,
  ['I'] = LEX_IS_SYMBOL_COMPONENT, ['i'] = LEX_IS_SYMBOL_COMPONENT,
  ['J'] = LEX_IS_SYMBOL_COMPONENT, ['j'] = LEX_IS_SYMBOL_COMPONENT,
  ['K'] = LEX_IS_SYMBOL_COMPONENT, ['k'] = LEX_IS_SYMBOL_COMPONENT,
  ['L'] = LEX_IS_SYMBOL_COMPONENT, ['l'] = LEX_IS_SYMBOL_COMPONENT,
  ['M'] = LEX_IS_SYMBOL_COMPONENT, ['m'] = LEX_IS_SYMBOL_COMPONENT,
  ['N'] = LEX_IS_SYMBOL_COMPONENT, ['n'] = LEX_IS_SYMBOL_COMPONENT,
  ['O'] = LEX_IS_SYMBOL_COMPONENT, ['o'] = LEX_IS_SYMBOL_COMPONENT,
  ['P'] = LEX_IS_SYMBOL_COMPONENT, ['p'] = LEX_IS_SYMBOL_COMPONENT,
  ['Q'] = LEX_IS_SYMBOL_COMPONENT, ['q'] = LEX_IS_SYMBOL_COMPONENT,
  ['R'] = LEX_IS_SYMBOL_COMPONENT, ['r'] = LEX_IS_SYMBOL_COMPONENT,
  ['S'] = LEX_IS_SYMBOL_COMPONENT, ['s'] = LEX_IS_SYMBOL_COMPONENT,
  ['T'] = LEX_IS_SYMBOL_COMPONENT, ['t'] = LEX_IS_SYMBOL_COMPONENT,
  ['U'] = LEX_IS_SYMBOL_COMPONENT, ['u'] = LEX_IS_SYMBOL_COMPONENT,
  ['V'] = LEX_IS_SYMBOL_COMPONENT, ['v'] = LEX_IS_SYMBOL_COMPONENT,
  ['W'] = LEX_IS_SYMBOL_COMPONENT, ['w'] = LEX_IS_SYMBOL_COMPONENT,
  ['X'] = LEX_IS_SYMBOL_COMPONENT, ['x'] = LEX_IS_SYMBOL_COMPONENT,
  ['Y'] = LEX_IS_SYMBOL_COMPONENT, ['y'] = LEX_IS_SYMBOL_COMPONENT,
  ['Z'] = LEX_IS_SYMBOL_COMPONENT, ['z'] = LEX_IS_SYMBOL_COMPONENT,
  ['0'] = LEX_IS_SYMBOL_COMPONENT,
  ['1'] = LEX_IS_SYMBOL_COMPONENT,
  ['2'] = LEX_IS_SYMBOL_COMPONENT,
  ['3'] = LEX_IS_SYMBOL_COMPONENT,
  ['4'] = LEX_IS_SYMBOL_COMPONENT,
  ['5'] = LEX_IS_SYMBOL_COMPONENT,
  ['6'] = LEX_IS_SYMBOL_COMPONENT,
  ['7'] = LEX_IS_SYMBOL_COMPONENT,
  ['8'] = LEX_IS_SYMBOL_COMPONENT,
  ['9'] = LEX_IS_SYMBOL_COMPONENT,
#define INIT2(n) [n] = LEX_IS_SYMBOL_COMPONENT, \
		 [(n) + 1] = LEX_IS_SYMBOL_COMPONENT
#define INIT4(n)    INIT2 (n),  INIT2 ((n) +  2)
#define INIT8(n)    INIT4 (n),  INIT4 ((n) +  4)
#define INIT16(n)   INIT8 (n),  INIT8 ((n) +  8)
#define INIT32(n)  INIT16 (n), INIT16 ((n) + 16)
#define INIT64(n)  INIT32 (n), INIT32 ((n) + 32)
#define INIT128(n) INIT64 (n), INIT64 ((n) + 64)
  INIT128 (128),
#undef INIT128
#undef INIT64
#undef INIT32
#undef INIT16
#undef INIT8
#undef INIT4
#undef INIT2
};

void
do_scrub_begin (int m68k_mri ATTRIBUTE_UNUSED)
{
  const char *p;

  /* Latch this once at start.  xtensa uses a hook function, yet context isn't
     meaningful for scrubbing (or else we'd need to sync scrubber behavior as
     state changes).  */
  if (lex['/'] == 0)
    no_pseudo_dot = NO_PSEUDO_DOT;

#ifdef TC_M68K
  scrub_m68k_mri = m68k_mri;

  if (! m68k_mri)
#endif
    {
      lex['"'] = LEX_IS_STRINGQUOTE;

#if ! defined (TC_HPPA)
      lex['\''] = LEX_IS_ONECHAR_QUOTE;
#endif

#ifdef SINGLE_QUOTE_STRINGS
      lex['\''] = LEX_IS_STRINGQUOTE;
#endif
    }

  /* Note: if any other character can be LEX_IS_STRINGQUOTE, the loop
     in state 5 of do_scrub_chars must be changed.  */

  /* Note that these override the previous defaults, e.g. if ';' is a
     comment char, then it isn't a line separator.  */

#ifdef tc_symbol_chars
  /* This macro permits the processor to specify all characters which
     may appears in an operand.  This will prevent the scrubber from
     discarding meaningful whitespace in certain cases.  The i386
     backend uses this to support prefixes, which can confuse the
     scrubber as to whether it is parsing operands or opcodes.  */
  for (p = tc_symbol_chars; *p; ++p)
    lex[(unsigned char) *p] = LEX_IS_SYMBOL_COMPONENT;
#endif

  /* The m68k backend wants to be able to change comment_chars.  */
#ifndef tc_comment_chars
#define tc_comment_chars comment_chars
#endif
  for (p = tc_comment_chars; *p; p++)
    lex[(unsigned char) *p] = LEX_IS_COMMENT_START;

  /* While counter intuitive to have more special purpose line comment chars
     override more general purpose ordinary ones, logic in do_scrub_chars()
     depends on this ordering.   */
  for (p = line_comment_chars; *p; p++)
    lex[(unsigned char) *p] = LEX_IS_LINE_COMMENT_START;

#ifndef tc_line_separator_chars
#define tc_line_separator_chars line_separator_chars
#endif
  for (p = tc_line_separator_chars; *p; p++)
    lex[(unsigned char) *p] = LEX_IS_LINE_SEPARATOR;

#ifdef tc_parallel_separator_chars
  /* This macro permits the processor to specify all characters which
     separate parallel insns on the same line.  */
  for (p = tc_parallel_separator_chars; *p; p++)
    lex[(unsigned char) *p] = LEX_IS_PARALLEL_SEPARATOR;
#endif

  /* Only allow slash-star comments if slash is not in use.  Certain
     other cases are dealt with in LEX_IS_LINE_COMMENT_START handling.
     FIXME: This isn't right.  We should always permit them.  */
  if (lex['/'] == 0)
    lex['/'] = LEX_IS_TWOCHAR_COMMENT_1ST;

#ifdef TC_M68K
  if (m68k_mri)
    {
      lex['\''] = LEX_IS_STRINGQUOTE;
      lex[';'] = LEX_IS_COMMENT_START;
      lex['*'] = LEX_IS_LINE_COMMENT_START;
      /* The MRI documentation says '!' is LEX_IS_COMMENT_START, but
	 then it can't be used in an expression.  */
      lex['!'] = LEX_IS_LINE_COMMENT_START;
    }
#endif

#ifdef TC_V850
  lex['-'] = LEX_IS_DOUBLEDASH_1ST;
#endif
#ifdef DOUBLEBAR_PARALLEL
  lex['|'] = LEX_IS_DOUBLEBAR_1ST;
#endif

#ifdef H_TICK_HEX
  if (enable_h_tick_hex)
    {
      lex['h'] = LEX_IS_H;
      lex['H'] = LEX_IS_H;
    }
#endif
}

/* Saved state of the scrubber.  */
static int state;
static int old_state;
static const char *out_string;
static char out_buf[20];
static int add_newlines;
static char *saved_input;
static size_t saved_input_len;
static char input_buffer[32 * 1024];

/* Data structure for saving the state of app across #include's.  Note that
   app is called asynchronously to the parsing of the .include's, so our
   state at the time .include is interpreted is completely unrelated.
   That's why we have to save it all.  */

struct app_save
{
  int          state;
  int          old_state;
  const char * out_string;
  char         out_buf[sizeof (out_buf)];
  int          add_newlines;
  char *       saved_input;
  size_t       saved_input_len;
  const char * end_state;
#ifdef TC_M68K
  int          scrub_m68k_mri;
  const char * mri_state;
  char         mri_last_ch;
#endif
#if defined TC_ARM && defined OBJ_ELF
  const char * symver_state;
#endif
  char         last_char;
};

char *
app_push (void)
{
  struct app_save *saved;

  saved = XNEW (struct app_save);
  saved->state = state;
  saved->old_state = old_state;
  saved->out_string = out_string;
  memcpy (saved->out_buf, out_buf, sizeof (out_buf));
  saved->add_newlines = add_newlines;
  if (saved_input == NULL)
    saved->saved_input = NULL;
  else
    {
      saved->saved_input = XNEWVEC (char, saved_input_len);
      memcpy (saved->saved_input, saved_input, saved_input_len);
      saved->saved_input_len = saved_input_len;
    }
  saved->end_state = end_state;
#ifdef TC_M68K
  saved->scrub_m68k_mri = scrub_m68k_mri;
  saved->mri_state = mri_state;
  saved->mri_last_ch = mri_last_ch;
#endif
#if defined TC_ARM && defined OBJ_ELF
  saved->symver_state = symver_state;
#endif
  saved->last_char = last_char;

  /* do_scrub_begin() is not useful, just wastes time.  */

  state = 0;
  saved_input = NULL;
  add_newlines = 0;

  return (char *) saved;
}

void
app_pop (char *arg)
{
  struct app_save *saved = (struct app_save *) arg;

  /* There is no do_scrub_end ().  */
  state = saved->state;
  old_state = saved->old_state;
  out_string = saved->out_string;
  memcpy (out_buf, saved->out_buf, sizeof (out_buf));
  add_newlines = saved->add_newlines;
  if (saved->saved_input == NULL)
    saved_input = NULL;
  else
    {
      gas_assert (saved->saved_input_len <= sizeof (input_buffer));
      memcpy (input_buffer, saved->saved_input, saved->saved_input_len);
      saved_input = input_buffer;
      saved_input_len = saved->saved_input_len;
      free (saved->saved_input);
    }
  end_state = saved->end_state;
#ifdef TC_M68K
  scrub_m68k_mri = saved->scrub_m68k_mri;
  mri_state = saved->mri_state;
  mri_last_ch = saved->mri_last_ch;
#endif
#if defined TC_ARM && defined OBJ_ELF
  symver_state = saved->symver_state;
#endif
  last_char = saved->last_char;

  free (arg);
}

/* @@ This assumes that \n &c are the same on host and target.  This is not
   necessarily true.  */

static int
process_escape (int ch)
{
  switch (ch)
    {
    case 'b':
      return '\b';
    case 'f':
      return '\f';
    case 'n':
      return '\n';
    case 'r':
      return '\r';
    case 't':
      return '\t';
    case '\'':
      return '\'';
    case '"':
      return '\"';
    default:
      return ch;
    }
}

#define MULTIBYTE_WARN_COUNT_LIMIT 10
static unsigned int multibyte_warn_count = 0;

bool
scan_for_multibyte_characters (const unsigned char *  start,
			       const unsigned char *  end,
			       bool                   warn)
{
  if (end <= start)
    return false;

  if (warn && multibyte_warn_count > MULTIBYTE_WARN_COUNT_LIMIT)
    return false;

  bool found = false;

  while (start < end)
    {
      unsigned char c;

      if ((c = * start++) <= 0x7f)
	continue;

      if (!warn)
	return true;

      found = true;

      const char * filename;
      unsigned int lineno;

      filename = as_where (& lineno);
      if (filename == NULL)
	as_warn (_("multibyte character (%#x) encountered in input"), c);
      else if (lineno == 0)
	as_warn (_("multibyte character (%#x) encountered in %s"), c, filename);
      else
	as_warn (_("multibyte character (%#x) encountered in %s at or near line %u"), c, filename, lineno);

      if (++ multibyte_warn_count == MULTIBYTE_WARN_COUNT_LIMIT)
	{
	  as_warn (_("further multibyte character warnings suppressed"));
	  break;
	}
    }

  return found;
}

/* This function is called to process input characters.  The GET
   parameter is used to retrieve more input characters.  GET should
   set its parameter to point to a buffer, and return the length of
   the buffer; it should return 0 at end of file.  The scrubbed output
   characters are put into the buffer starting at TOSTART; the TOSTART
   buffer is TOLEN bytes in length.  The function returns the number
   of scrubbed characters put into TOSTART.  This will be TOLEN unless
   end of file was seen.  This function is arranged as a state
   machine, and saves its state so that it may return at any point.
   This is the way the old code used to work.  */

size_t
do_scrub_chars (size_t (*get) (char *, size_t), char *tostart, size_t tolen,
		bool check_multibyte)
{
  char *to = tostart;
  char *toend = tostart + tolen;
  char *from;
  char *fromend;
  size_t fromlen;
  int ch, ch2 = 0;
  /* Character that started the string we're working on.  */
  static char quotechar;

  /*State 0: beginning of normal line
	  1: After first whitespace on line (flush more white)
	  2: After first non-white (opcode) on line (keep 1white)
	  3: after second white on line (into operands) (flush white)
	  4: after putting out a .linefile, put out digits
	  5: parsing a string, then go to old-state
	  6: putting out \ escape in a "d string.
	  7: no longer used
	  8: no longer used
	  9: After seeing symbol char in state 3 (keep 1white after symchar)
	 10: After seeing whitespace in state 9 (keep white before symchar)
	 11: After seeing a symbol character in state 0 (eg a label definition)
	 -1: output string in out_string and go to the state in old_state
	 12: no longer used
#ifdef DOUBLEBAR_PARALLEL
	 13: After seeing a vertical bar, looking for a second
	     vertical bar as a parallel expression separator.
#endif
#ifdef TC_PREDICATE_START_CHAR
	 14: After seeing a predicate start character at state 0, looking
	     for a predicate end character as predicate.
	 15: After seeing a predicate start character at state 1, looking
	     for a predicate end character as predicate.
#endif
#ifdef TC_Z80
	 16: After seeing an 'a' or an 'A' at the start of a symbol
	 17: After seeing an 'f' or an 'F' in state 16
#endif
	  */

  /* I added states 9 and 10 because the MIPS ECOFF assembler uses
     constructs like ``.loc 1 20''.  This was turning into ``.loc
     120''.  States 9 and 10 ensure that a space is never dropped in
     between characters which could appear in an identifier.  Ian
     Taylor, ian@cygnus.com.

     I added state 11 so that something like "Lfoo add %r25,%r26,%r27" works
     correctly on the PA (and any other target where colons are optional).
     Jeff Law, law@cs.utah.edu.

     I added state 13 so that something like "cmp r1, r2 || trap #1" does not
     get squashed into "cmp r1,r2||trap#1", with the all important space
     between the 'trap' and the '#1' being eliminated.  nickc@cygnus.com  */

  /* This macro gets the next input character.  */

#define GET()							\
  (from < fromend						\
   ? * (unsigned char *) (from++)				\
   : (saved_input = NULL,					\
      fromlen = (*get) (input_buffer, sizeof input_buffer),	\
      from = input_buffer,					\
      fromend = from + fromlen,					\
      (fromlen == 0						\
       ? EOF							\
       : * (unsigned char *) (from++))))

  /* This macro pushes a character back on the input stream.  */

#define UNGET(uch) (*--from = (uch))

  /* This macro puts a character into the output buffer.  If this
     character fills the output buffer, this macro jumps to the label
     TOFULL.  We use this rather ugly approach because we need to
     handle two different termination conditions: EOF on the input
     stream, and a full output buffer.  It would be simpler if we
     always read in the entire input stream before processing it, but
     I don't want to make such a significant change to the assembler's
     memory usage.  */

#define PUT(pch)				\
  do						\
    {						\
      *to++ = (pch);				\
      if (to >= toend)				\
	goto tofull;				\
    }						\
  while (0)

  if (saved_input != NULL)
    {
      from = saved_input;
      fromend = from + saved_input_len;
    }
  else
    {
      fromlen = (*get) (input_buffer, sizeof input_buffer);
      if (fromlen == 0)
	return 0;
      from = input_buffer;
      fromend = from + fromlen;

      if (check_multibyte)
	(void) scan_for_multibyte_characters ((const unsigned char *) from,
					      (const unsigned char *) fromend,
					      true /* Generate warnings.  */);
    }

  while (1)
    {
      /* The cases in this switch end with continue, in order to
	 branch back to the top of this while loop and generate the
	 next output character in the appropriate state.  */
      switch (state)
	{
	case -1:
	  ch = *out_string++;
	  if (*out_string == '\0')
	    {
	      state = old_state;
	      old_state = 3;
	    }
	  PUT (ch);
	  continue;

	case 4:
	  ch = GET ();
	  if (ch == EOF)
	    goto fromeof;
	  else if (ch >= '0' && ch <= '9')
	    PUT (ch);
	  else
	    {
	      while (ch != EOF && IS_WHITESPACE (ch))
		ch = GET ();
	      if (ch == '"')
		{
		  quotechar = ch;
		  state = 5;
		  old_state = 3;
		  PUT (ch);
		}
	      else
		{
		  while (ch != EOF && ch != '\n')
		    ch = GET ();
		  state = 0;
		  PUT (ch);
		}
	    }
	  continue;

	case 5:
	  /* We are going to copy everything up to a quote character,
	     with special handling for a backslash.  We try to
	     optimize the copying in the simple case without using the
	     GET and PUT macros.  */
	  {
	    char *s;
	    ptrdiff_t len;

	    for (s = from; s < fromend; s++)
	      {
		ch = *s;
		if (ch == '\\'
		    || ch == quotechar
		    || ch == '\n')
		  break;
	      }
	    len = s - from;
	    if (len > toend - to)
	      len = toend - to;
	    if (len > 0)
	      {
		memcpy (to, from, len);
		to += len;
		from += len;
		if (to >= toend)
		  goto tofull;
	      }
	  }

	  ch = GET ();
	  if (ch == EOF)
	    {
	      /* This buffer is here specifically so
		 that the UNGET below will work.  */
	      static char one_char_buf[1];

	      as_warn (_("end of file in string; '%c' inserted"), quotechar);
	      state = old_state;
	      from = fromend = one_char_buf + 1;
	      fromlen = 1;
	      UNGET ('\n');
	      PUT (quotechar);
	    }
	  else if (ch == quotechar)
	    {
	      state = old_state;
	      PUT (ch);
	    }
	  else if (TC_STRING_ESCAPES && ch == '\\')
	    {
	      state = 6;
	      PUT (ch);
	    }
	  else if (scrub_m68k_mri && ch == '\n')
	    {
	      /* Just quietly terminate the string.  This permits lines like
		   bne	label	loop if we haven't reach end yet.  */
	      state = old_state;
	      UNGET (ch);
	      PUT ('\'');
	    }
	  else
	    {
	      PUT (ch);
	    }
	  continue;

	case 6:
	  state = 5;
	  ch = GET ();
	  switch (ch)
	    {
	      /* Handle strings broken across lines, by turning '\n' into
		 '\\' and 'n'.  */
	    case '\n':
	      UNGET ('n');
	      add_newlines++;
	      PUT ('\\');
	      continue;

	    case EOF:
	      as_warn (_("end of file in string; '%c' inserted"), quotechar);
	      PUT (quotechar);
	      continue;

	      /* These two are used inside macros.  */
	    case '@':
	    case '+':
	      break;

	    case '"':
	    case '\\':
	    case 'b':
	    case 'f':
	    case 'n':
	    case 'r':
	    case 't':
	    case 'v':
	    case 'x':
	    case 'X':
	    case '0':
	    case '1':
	    case '2':
	    case '3':
	    case '4':
	    case '5':
	    case '6':
	    case '7':
	      break;

	    default:
#ifdef ONLY_STANDARD_ESCAPES
	      as_warn (_("unknown escape '\\%c' in string; ignored"), ch);
#endif
	      break;
	    }
	  PUT (ch);
	  continue;

#ifdef DOUBLEBAR_PARALLEL
	case 13:
	  ch = GET ();
	  if (ch != '|')
	    abort ();

	  /* Reset back to state 1 and pretend that we are parsing a
	     line from just after the first white space.  */
	  state = 1;
	  PUT ('|');
	  continue;
#endif
#ifdef TC_Z80
	case 16:
	  /* We have seen an 'a' at the start of a symbol, look for an 'f'.  */
	  ch = GET ();
	  if (ch == 'f' || ch == 'F')
	    {
	      state = 17;
	      PUT (ch);
	    }
	  else
	    {
	      if (ch != EOF)
		UNGET (ch);
	      state = 9;
	      break;
	    }
	  /* Fall through.  */
	case 17:
	  /* We have seen "af" at the start of a symbol,
	     a ' here is a part of that symbol.  */
	  ch = GET ();
	  state = 9;
	  if (ch == '\'')
	    /* Change to avoid warning about unclosed string.  */
	    PUT ('`');
	  else if (ch != EOF)
	    UNGET (ch);
	  break;
#endif
	}

      /* OK, we are somewhere in states 0 through 4 or 9 through 11.  */

      /* flushchar: */
      ch = GET ();

#ifdef TC_PREDICATE_START_CHAR
      if (ch == TC_PREDICATE_START_CHAR && (state == 0 || state == 1))
	{
	  state += 14;
	  PUT (ch);
	  continue;
	}
      else if (state == 14 || state == 15)
	{
	  if (ch == TC_PREDICATE_END_CHAR)
	    {
	      state -= 14;
	      PUT (ch);
	      ch = GET ();
	    }
	  else
	    {
	      PUT (ch);
	      continue;
	    }
	}
#endif

    recycle:

      /* We need to watch out for .end directives: We should in particular not
	 issue diagnostics for anything after an active one.  */
      if (ch == EOF)
	end_state = NULL;
      else if (end_state == NULL)
	{
	  if ((state == 0 || state == 1)
	      && (ch == '.'
		  || (no_pseudo_dot && ch == end_pseudo[0])))
	    end_state = end_pseudo + (ch != '.');
	}
      else if (ch != '\0'
	       && (*end_state == ch
		   /* Avoid triggering on directives like .endif or .endr.  */
		   || (*end_state == ' ' && !IS_SYMBOL_COMPONENT (ch))))
	{
 	  if (IS_NEWLINE (ch) || IS_LINE_SEPARATOR (ch))
 	    goto end_end;
	  ++end_state;
	}
      else if (*end_state != '\0')
	/* We did not get the expected character, or we didn't
	   get a valid terminating character after seeing the
	   entire pseudo-op, so we must go back to the beginning.  */
	end_state = NULL;
      else if (IS_NEWLINE (ch) || IS_LINE_SEPARATOR (ch))
	{
	end_end:
	  /* We've read the entire pseudo-op.  If this is the end of the line,
	     bail out now by (ab)using the output-full path.  This allows the
	     caller to process input up to here and terminate processing if this
	     directive is actually active (not on the false branch of a
	     conditional and not in a macro definition).  */
	  end_state = NULL;
	  state = 0;
	  PUT (ch);
	  goto tofull;
	}

#if defined TC_ARM && defined OBJ_ELF
      /* We need to watch out for .symver directives.  See the comment later
	 in this function.  */
      if (ch == EOF)
	symver_state = NULL;
      else if (symver_state == NULL)
	{
	  if ((state == 0 || state == 1)
	      && strchr (tc_comment_chars, '@') != NULL
	      && ch == symver_pseudo[0])
	    symver_state = symver_pseudo + 1;
	}
      else
	{
	  /* We advance to the next state if we find the right
	     character.  */
	  if (ch != '\0' && (*symver_state == ch))
	    ++symver_state;
	  else if (*symver_state != '\0')
	    /* We did not get the expected character, or we didn't
	       get a valid terminating character after seeing the
	       entire pseudo-op, so we must go back to the beginning.  */
	    symver_state = NULL;
	  else
	    {
	      /* We've read the entire pseudo-op.  If this is the end
		 of the line, go back to the beginning.  */
	      if (IS_NEWLINE (ch) || IS_LINE_SEPARATOR (ch))
		symver_state = NULL;
	    }
	}
#endif /* TC_ARM && OBJ_ELF */

#ifdef TC_M68K
      /* We want to have pseudo-ops which control whether we are in
	 MRI mode or not.  Unfortunately, since m68k MRI mode affects
	 the scrubber, that means that we need a special purpose
	 recognizer here.  */
      if (ch == EOF)
	mri_state = NULL;
      else if (mri_state == NULL)
	{
	  if ((state == 0 || state == 1)
	      && ch == mri_pseudo[0])
	    mri_state = mri_pseudo + 1;
	}
      else
	{
	  /* We advance to the next state if we find the right
	     character, or if we need a space character and we get any
	     whitespace character, or if we need a '0' and we get a
	     '1' (this is so that we only need one state to handle
	     ``.mri 0'' and ``.mri 1'').  */
	  if (ch != '\0'
	      && (*mri_state == ch
		  || (*mri_state == ' '
		      && IS_WHITESPACE (ch))
		  || (*mri_state == '0'
		      && ch == '1')))
	    {
	      mri_last_ch = ch;
	      ++mri_state;
	    }
	  else if (*mri_state != '\0'
		   || (!IS_WHITESPACE (ch)
		       && !IS_LINE_SEPARATOR (ch)
		       && !IS_NEWLINE (ch)))
	    {
	      /* We did not get the expected character, or we didn't
		 get a valid terminating character after seeing the
		 entire pseudo-op, so we must go back to the
		 beginning.  */
	      mri_state = NULL;
	    }
	  else
	    {
	      /* We've read the entire pseudo-op.  mri_last_ch is
		 either '0' or '1' indicating whether to enter or
		 leave MRI mode.  */
	      do_scrub_begin (mri_last_ch == '1');
	      mri_state = NULL;

	      /* We continue handling the character as usual.  The
		 main gas reader must also handle the .mri pseudo-op
		 to control expression parsing and the like.  */
	    }
	}
#endif

      if (ch == EOF)
	{
	  if (state != 0)
	    {
	      as_warn (_("end of file not at end of a line; newline inserted"));
	      state = 0;
	      PUT ('\n');
	    }
	  goto fromeof;
	}

      switch (lex[ch])
	{
	case LEX_IS_WHITESPACE:
	  do
	    {
	      ch = GET ();
	    }
	  while (ch != EOF && IS_WHITESPACE (ch));
	  if (ch == EOF)
	    goto fromeof;

	  if (state == 0)
	    {
	      /* Preserve a single whitespace character at the
		 beginning of a line.  */
	      state = 1;
	      UNGET (ch);
	      PUT (' ');
	      break;
	    }

#ifdef KEEP_WHITE_AROUND_COLON
	  if (lex[ch] == LEX_IS_COLON)
	    {
	      /* Only keep this white if there's no white *after* the
		 colon.  */
	      ch2 = GET ();
	      if (ch2 != EOF)
		UNGET (ch2);
	      if (!IS_WHITESPACE (ch2))
		{
		  state = 9;
		  UNGET (ch);
		  PUT (' ');
		  break;
		}
	    }
#endif

	  /* Prune trailing whitespace.  */
	  if (IS_COMMENT (ch)
	      || (IS_LINE_COMMENT (ch)
	          && (state < 1 || strchr (tc_comment_chars, ch)))
	      || IS_NEWLINE (ch)
	      || IS_LINE_SEPARATOR (ch)
	      || IS_PARALLEL_SEPARATOR (ch))
	    {
	      if (scrub_m68k_mri)
		{
		  /* In MRI mode, we keep these spaces.  */
		  UNGET (ch);
		  PUT (' ');
		  break;
		}
	      goto recycle;
	    }
#ifdef DOUBLESLASH_LINE_COMMENTS
	  if (IS_TWOCHAR_COMMENT_1ST (ch))
	    {
	      ch2 = GET ();
	      if (ch2 != EOF)
	        UNGET (ch2);
	      if (ch2 == '/')
		goto recycle;
	    }
#endif

	  /* If we're in state 2 or 11, we've seen a non-white
	     character followed by whitespace.  If the next character
	     is ':', this is whitespace after a label name which we
	     normally must ignore.  In MRI mode, though, spaces are
	     not permitted between the label and the colon.  */
	  if ((state == 2 || state == 11)
	      && lex[ch] == LEX_IS_COLON
	      && ! scrub_m68k_mri)
	    {
	      state = 1;
	      PUT (ch);
	      break;
	    }

	  switch (state)
	    {
	    case 1:
	      /* We can arrive here if we leave a leading whitespace
		 character at the beginning of a line.  */
	      goto recycle;
	    case 2:
	      state = 3;
	      if (to + 1 < toend)
		{
		  /* Optimize common case by skipping UNGET/GET.  */
		  PUT (' ');	/* Sp after opco */
		  goto recycle;
		}
	      UNGET (ch);
	      PUT (' ');
	      break;
	    case 3:
#ifndef TC_KEEP_OPERAND_SPACES
	      /* For TI C6X, we keep these spaces as they may separate
		 functional unit specifiers from operands.  */
	      if (scrub_m68k_mri)
#endif
		{
		  /* In MRI mode, we keep these spaces.  */
		  UNGET (ch);
		  PUT (' ');
		  break;
		}
	      goto recycle;	/* Sp in operands */
	    case 9:
	    case 10:
#ifndef TC_KEEP_OPERAND_SPACES
	      if (scrub_m68k_mri)
#endif
		{
		  /* In MRI mode, we keep these spaces.  */
		  state = 3;
		  UNGET (ch);
		  PUT (' ');
		  break;
		}
	      state = 10;	/* Sp after symbol char */
	      goto recycle;
	    case 11:
	      if (LABELS_WITHOUT_COLONS || flag_m68k_mri)
		state = 1;
	      else
		{
		  /* We know that ch is not ':', since we tested that
		     case above.  Therefore this is not a label, so it
		     must be the opcode, and we've just seen the
		     whitespace after it.  */
		  state = 3;
		}
	      UNGET (ch);
	      PUT (' ');	/* Sp after label definition.  */
	      break;
	    default:
	      BAD_CASE (state);
	    }
	  break;

	case LEX_IS_TWOCHAR_COMMENT_1ST:
	  ch2 = GET ();
	  if (ch2 == '*')
	    {
	twochar_comment:
	      for (;;)
		{
		  do
		    {
		      ch2 = GET ();
		      if (ch2 != EOF && IS_NEWLINE (ch2))
			add_newlines++;
		    }
		  while (ch2 != EOF && ch2 != '*');

		  while (ch2 == '*')
		    ch2 = GET ();

		  if (ch2 == EOF || ch2 == '/')
		    break;

		  /* This UNGET will ensure that we count newlines
		     correctly.  */
		  UNGET (ch2);
		}

	      if (ch2 == EOF)
		as_warn (_("end of file in multiline comment"));

	      ch = ' ';
	      goto recycle;
	    }
#ifdef DOUBLESLASH_LINE_COMMENTS
	  else if (ch2 == '/')
	    {
	      do
		{
		  ch = GET ();
		}
	      while (ch != EOF && !IS_NEWLINE (ch));
	      if (ch == EOF)
		as_warn ("end of file in comment; newline inserted");
	      state = 0;
	      PUT ('\n');
	      break;
	    }
#endif
	  else
	    {
	      if (ch2 != EOF)
		UNGET (ch2);
	      if (state == 9 || state == 10)
		state = 3;
	      PUT (ch);
	    }
	  break;

	case LEX_IS_STRINGQUOTE:
	  quotechar = ch;
	  if (state == 10)
	    {
	      /* Preserve the whitespace in foo "bar".  */
	      UNGET (ch);
	      state = 3;
	      PUT (' ');

	      /* PUT didn't jump out.  We could just break, but we
		 know what will happen, so optimize a bit.  */
	      ch = GET ();
	      old_state = 9;
	    }
	  else if (state == 3)
	    old_state = 9;
	  else if (state == 0)
	    old_state = 11; /* Now seeing label definition.  */
	  else
	    old_state = state;
	  state = 5;
	  PUT (ch);
	  break;

	case LEX_IS_ONECHAR_QUOTE:
#ifdef H_TICK_HEX
	  if (state == 9 && enable_h_tick_hex)
	    {
	      char c;

	      c = GET ();
	      as_warn ("'%c found after symbol", c);
	      UNGET (c);
	    }
#endif
	  if (state == 10)
	    {
	      /* Preserve the whitespace in foo 'b'.  */
	      UNGET (ch);
	      state = 3;
	      PUT (' ');
	      break;
	    }
	  ch = GET ();
	  if (ch == EOF)
	    {
	      as_warn (_("end of file after a one-character quote; \\0 inserted"));
	      ch = 0;
	    }
	  if (ch == '\\')
	    {
	      ch = GET ();
	      if (ch == EOF)
		{
		  as_warn (_("end of file in escape character"));
		  ch = '\\';
		}
	      else
		ch = process_escape (ch);
	    }
	  sprintf (out_buf, "%d", ch & 0xff);

	  /* None of these 'x constants for us.  We want 'x'.  */
	  if ((ch = GET ()) != '\'')
	    {
#ifdef REQUIRE_CHAR_CLOSE_QUOTE
	      as_warn (_("missing close quote; (assumed)"));
#else
	      if (ch != EOF)
		UNGET (ch);
#endif
	    }
	  if (strlen (out_buf) == 1)
	    {
	      PUT (out_buf[0]);
	      break;
	    }
	  if (state == 9)
	    old_state = 3;
	  else
	    old_state = state;
	  state = -1;
	  out_string = out_buf;
	  PUT (*out_string++);
	  break;

	case LEX_IS_COLON:
#ifdef KEEP_WHITE_AROUND_COLON
	  state = 9;
#else
	  if (state == 9 || state == 10)
	    state = 3;
	  else if (state != 3)
	    state = 1;
#endif
	  PUT (ch);
	  break;

	case LEX_IS_NEWLINE:
	  /* Roll out a bunch of newlines from inside comments, etc.  */
	  if (add_newlines)
	    {
	      --add_newlines;
	      UNGET (ch);
	    }
	  /* Fall through.  */

	case LEX_IS_LINE_SEPARATOR:
	  state = 0;
	  PUT (ch);
	  break;

	case LEX_IS_PARALLEL_SEPARATOR:
	  state = 1;
	  PUT (ch);
	  break;

#ifdef TC_V850
	case LEX_IS_DOUBLEDASH_1ST:
	  ch2 = GET ();
	  if (ch2 != '-')
	    {
	      if (ch2 != EOF)
		UNGET (ch2);
	      goto de_fault;
	    }
	  /* Read and skip to end of line.  */
	  do
	    {
	      ch = GET ();
	    }
	  while (ch != EOF && ch != '\n');

	  if (ch == EOF)
	    as_warn (_("end of file in comment; newline inserted"));

	  state = 0;
	  PUT ('\n');
	  break;
#endif
#ifdef DOUBLEBAR_PARALLEL
	case LEX_IS_DOUBLEBAR_1ST:
	  ch2 = GET ();
	  if (ch2 != EOF)
	    UNGET (ch2);
	  if (ch2 != '|')
	    goto de_fault;

	  /* Handle '||' in two states as invoking PUT twice might
	     result in the first one jumping out of this loop.  We'd
	     then lose track of the state and one '|' char.  */
	  state = 13;
	  PUT ('|');
	  break;
#endif
	case LEX_IS_LINE_COMMENT_START:
	  /* FIXME-someday: The two character comment stuff was badly
	     thought out.  On i386, we want '/' as line comment start
	     AND we want C style comments.  hence this hack.  The
	     whole lexical process should be reworked.  xoxorich.  */
	  if (ch == '/')
	    {
	      ch2 = GET ();
	      if (ch2 == '*')
		goto twochar_comment;
	      if (ch2 != EOF)
		UNGET (ch2);
	    }

	  if (state == 0 || state == 1)	/* Only comment at start of line.  */
	    {
	      int startch;

	      startch = ch;

	      do
		{
		  ch = GET ();
		}
	      while (ch != EOF && IS_WHITESPACE (ch));

	      if (ch == EOF)
		{
		  as_warn (_("end of file in comment; newline inserted"));
		  PUT ('\n');
		  break;
		}

	      if (ch < '0' || ch > '9' || state != 0 || startch != '#')
		{
		  /* Not a cpp line.  */
		  while (ch != EOF && !IS_NEWLINE (ch))
		    ch = GET ();
		  if (ch == EOF)
		    {
		      as_warn (_("end of file in comment; newline inserted"));
		      PUT ('\n');
		    }
		  else /* IS_NEWLINE (ch) */
		    {
		      /* To process non-zero add_newlines.  */
		      UNGET (ch);
		    }
		  state = 0;
		  break;
		}
	      /* Looks like `# 123 "filename"' from cpp.  */
	      UNGET (ch);
	      old_state = 4;
	      state = -1;
	      if (scrub_m68k_mri)
		out_string = "\tlinefile ";
	      else
		out_string = "\t.linefile ";
	      PUT (*out_string++);
	      break;
	    }

#ifdef TC_D10V
	  /* All insns end in a char for which LEX_IS_SYMBOL_COMPONENT is true.
	     Trap is the only short insn that has a first operand that is
	     neither register nor label.
	     We must prevent exef0f ||trap #1 to degenerate to exef0f ||trap#1 .
	     We can't make '#' LEX_IS_SYMBOL_COMPONENT because it is
	     already LEX_IS_LINE_COMMENT_START.  However, it is the
	     only character in line_comment_chars for d10v, hence we
	     can recognize it as such.  */
	  /* An alternative approach would be to reset the state to 1 when
	     we see '||', '<'- or '->', but that seems to be overkill.  */
	  if (state == 10)
	    PUT (' ');
#endif
	  /* We have a line comment character which is not at the
	     start of a line.  If this is also a normal comment
	     character, fall through.  Otherwise treat it as a default
	     character.  */
	  if (strchr (tc_comment_chars, ch) == NULL)
	    goto de_fault;
	  if (scrub_m68k_mri
	      && (ch == '!' || ch == '*' || ch == '#'))
	    goto de_fault;
	  /* Fall through.  */
	case LEX_IS_COMMENT_START:
#if defined TC_ARM && defined OBJ_ELF
	  /* On the ARM, `@' is the comment character.
	     Unfortunately this is also a special character in ELF .symver
	     directives (and .type, though we deal with those another way).
	     So we check if this line is such a directive, and treat
	     the character as default if so.  This is a hack.  */
	  if ((symver_state != NULL) && (*symver_state == 0))
	    goto de_fault;
#endif

	  /* Care is needed not to damage occurrences of \<comment-char>
	     by stripping the <comment-char> onwards.  Yuck.  */
	  if ((to > tostart ? to[-1] : last_char) == '\\')
	    /* Do not treat the <comment-char> as a start-of-comment.  */
	    goto de_fault;

#ifdef WARN_COMMENTS
	  if (!found_comment)
	    found_comment_file = as_where (&found_comment);
#endif
	  do
	    {
	      ch = GET ();
	    }
	  while (ch != EOF && !IS_NEWLINE (ch));
	  if (ch == EOF)
	    as_warn (_("end of file in comment; newline inserted"));
	  state = 0;
	  PUT ('\n');
	  break;

#ifdef H_TICK_HEX
	case LEX_IS_H:
	  /* Look for strings like H'[0-9A-Fa-f] and if found, replace
	     the H' with 0x to make them gas-style hex characters.  */
	  if (enable_h_tick_hex)
	    {
	      char quot;

	      quot = GET ();
	      if (quot == '\'')
		{
		  UNGET ('x');
		  ch = '0';
		}
	      else
		UNGET (quot);
	    }
#endif
	  /* Fall through.  */

	case LEX_IS_SYMBOL_COMPONENT:
	  if (state == 10)
	    {
	      /* This is a symbol character following another symbol
		 character, with whitespace in between.  We skipped
		 the whitespace earlier, so output it now.  */
	      UNGET (ch);
	      state = 3;
	      PUT (' ');
	      break;
	    }

#ifdef TC_Z80
	  /* "af'" is a symbol containing '\''.  */
	  if (state == 3 && (ch == 'a' || ch == 'A'))
	    {
	      state = 16;
	      PUT (ch);
	      ch = GET ();
	      if (ch == 'f' || ch == 'F')
		{
		  state = 17;
		  PUT (ch);
		  break;
		}
	      else
		{
		  state = 9;
		  if (ch == EOF || !IS_SYMBOL_COMPONENT (ch))
		    {
		      if (ch != EOF)
			UNGET (ch);
		      break;
		    }
		}
	    }
#endif
	  if (state == 3)
	    state = 9;

	  /* This is a common case.  Quickly copy CH and all the
	     following symbol component or normal characters.  */
	  if (to + 1 < toend
#ifdef TC_M68K
	      && mri_state == NULL
#endif
#if defined TC_ARM && defined OBJ_ELF
	      && symver_state == NULL
#endif
	      && end_state == NULL)
	    {
	      char *s;
	      ptrdiff_t len;

	      for (s = from; s < fromend; s++)
		{
		  int type;

		  ch2 = *(unsigned char *) s;
		  type = lex[ch2];
		  if (type != 0
		      && type != LEX_IS_SYMBOL_COMPONENT)
		    break;
		}

	      if (s > from)
		/* Handle the last character normally, for
		   simplicity.  */
		--s;

	      len = s - from;

	      if (len > (toend - to) - 1)
		len = (toend - to) - 1;

	      if (len > 0)
		{
		  PUT (ch);
		  memcpy (to, from, len);
		  to += len;
		  from += len;
		  if (to >= toend)
		    goto tofull;
		  ch = GET ();
		}
	    }

	  /* Fall through.  */
	default:
	de_fault:
	  /* Some relatively `normal' character.  */
	  if (state == 0)
	    {
	      state = 11;	/* Now seeing label definition.  */
	    }
	  else if (state == 1)
	    {
	      state = 2;	/* Ditto.  */
	    }
	  else if (state == 9)
	    {
	      if (!IS_SYMBOL_COMPONENT (ch))
		state = 3;
	    }
	  else if (state == 10)
	    {
	      if (ch == '\\')
		{
		  /* Special handling for backslash: a backslash may
		     be the beginning of a formal parameter (of a
		     macro) following another symbol character, with
		     whitespace in between.  If that is the case, we
		     output a space before the parameter.  Strictly
		     speaking, correct handling depends upon what the
		     macro parameter expands into; if the parameter
		     expands into something which does not start with
		     an operand character, then we don't want to keep
		     the space.  We don't have enough information to
		     make the right choice, so here we are making the
		     choice which is more likely to be correct.  */
		  if (to + 1 >= toend)
		    {
		      /* If we're near the end of the buffer, save the
		         character for the next time round.  Otherwise
		         we'll lose our state.  */
		      UNGET (ch);
		      goto tofull;
		    }
		  *to++ = ' ';
		}

	      state = 3;
	    }
	  PUT (ch);
	  break;
	}
    }

  /*NOTREACHED*/

 fromeof:
  /* We have reached the end of the input.  */
  if (to > tostart)
    last_char = to[-1];
  return to - tostart;

 tofull:
  /* The output buffer is full.  Save any input we have not yet
     processed.  */
  if (fromend > from)
    {
      saved_input = from;
      saved_input_len = fromend - from;
    }
  else
    saved_input = NULL;

  if (to > tostart)
    last_char = to[-1];
  return to - tostart;
}

/* Return amount of pending input.  */

size_t
do_scrub_pending (void)
{
  size_t len = 0;
  if (saved_input)
    len += saved_input_len;
  if (state == -1)
    len += strlen (out_string);
  return len;
}
