/* readconf.c - routines used to read configuration file and set global vars
 * Copyright (C) 2005, 2006, 2007 Jia Wang <skyroam@gmail.com>
 *
 *
 *
 * This file is part of GNU Proxyknife.
 * GNU Proxyknife is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the 
 * Free Software Foundation; either version 3 of the License, or (at your 
 * option) any later version.
 *
 * GNU Proxyknife is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 * General Public License for more details.
 *
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/* all functions declared in pyhc.h */
#include <proxyknife.h>
/* functions defined in this file */
int readint (char *line, char *flagstring, int *i, int *num);
int readstr (char *line, char *flagstring, char **s, int *num);
void optstr (char *value, char **s, int *num);
void optint (char *value, int *i, int *num);
void * xstrdup(char *s);

#include <stdio.h>
#include <stdlib.h>
#include <netdb.h>
extern int h_errno;
#include <string.h>
/* for getopt */
#include <unistd.h>
extern char *optarg;
extern int optind, opterr, optopt;
#include <getopt.h>

#include <assert.h>
void
usage ()
{
  printf (_("\
GNU Proxyknife %s, a customizable proxy hunter.\n\
Usage: proxyknife [OPTION]...\n\
Special options:\n\
-V,  --version    display the version of Proxyknife and exit.\n\
     --help       print this help.\n\
\n\
All of the following options need arguments if no declarations.\n\
\n\
Commandline only options:\n\
-f \tthe argument is configuration file.\n\
\n\
Configuration options:\n\
-d \tdebug, turn on debug output.\n\
   \tNotice: no argument for this option.\n\
-t \tthreads\n\
-g \tuser_agent\n\
-I \tproxyknifeintype, the type of proxylist(local file or url).\n\
-i \tproxyknifein, the proxylist file or url used as input of proxyknife.\n\
   \t'-' meanings stdin.\n\
-m \tmytype, the type of myproxy.\n\
-h \tmyhost, the IPV4 address or host name of myproxy.\n\
-p \tmyport, the port of myproxy.\n\
-u \tmyuser, the username of myproxy if need.\n\
-s \tmypass, the password of myproxy if need.\n\
-c \tchecksite, the target site used to check testproxies. Google by default.\n\
-C \tcheckreqin, the file including req string.\n\
-k \tkey, the keyword used to check whether testproxies give correct response.\n\
-r \trcvtimeo, socket recv timeout in seconds.\n\
-n \tsndtimeo, socket send timeout in seconds.\n\
-S \tmyconreplysize\n\
-T \ttestconreplysize\n\
-G \ttesttagreplysize\n\
-H \thttptestmethod\n\
-5 \tsocks5testmethod, reserved now\n\
-4 \tsocks4testmethod, reserved now\n\
-l \tlogfilename\n\
-o \tproxyknifeout, the result is also printed to this file.\n\
\n\
Mail Bug reports and suggestions to <bug-proxyknife@gnu.org>.\n"), 
VERSION);
}

/* Read file PROXYKNIFE_CONF and configure global vars */
void
READCONF (int argc, char **argv)
{
  FILE *conf;
  char *proxyknife_conf = NULL;
  char *line;
  /* malloc at least linelen for a line */
  int linelen;
  int stringlen;
  ssize_t read;
  char *offset;
  char *p, *q;
  /* must be customized. Have default guess. */
  int set_mytype = 0, set_httptestmethod = 0, set_socks5testmethod =
    0, set_socks4testmethod = 0;
  /* Must too only if mytype != DIRECT. Without default value */
  int set_myport = 0, set_myhost = 0;
  /* Must too only if mytype = *_AUTH. Without default value */
  int set_myuser = 0, set_mypass = 0;
  /* The following is customizable and  is also ok to use the value in 
     example. But do not remove or comment any of the following.
   */
  /* These with hard-coded value by default. Must have value. */
  int set_proxyknifein = 0, set_proxyknifeout = 0, set_proxyknifeintype = 0;
  int set_debug = 0; 
  
  int set_threads = 0;
  int set_checksite = 0, set_key = 0, set_logfilename = 0;
  int set_user_agent = 0, set_myconreplysize = 0, set_testconreplysize = 0;
  int set_testtagreplysize = 0, set_rcvtimeo = 0, set_sndtimeo = 0;

  /* Full customized. It is ok whether it exists. */
  int set_checkreqin = 0;

  struct hostent *h;

  int slash, colon;
  unsigned char *filename, *home;
  int filenamelen;


  /* for command option */
  int c;
  /* for long options */
  int digit_optind = 0;
  int this_option_optind = optind ? optind : 1;
  int option_index = 0;
  static struct option long_options[] = {
    {"help", 0, NULL, 129},
    {"version", 0, NULL, 'V'},
    {0, 0, 0, 0}
  };
  
  /* Default value */
  /* Guess. User must consider these. */
  my.mytype = DIRECT;
  test.httptestmethod = HTTP_GET;

  /* Guess. But now only support the default: */
  test.socks5testmethod = SOCKS5_CONNECT;
  test.socks4testmethod = SOCKS4_CONNECT;


  /* Default value. No change is also ok. */
  threads = THREADS;
  
  target.target = xstrdup(CHECKSITE);
  target.key = xstrdup(KEY);
  
  logfilename = xstrdup(LOGFILENAME);
  
  user_agent = xstrdup(USER_AGENT);

  my.myconreplysize = MYCONREPLYSIZE;
  test.testconreplysize = TESTCONREPLYSIZE;
  test.testtagreplysize = TESTTAGREPLYSIZE;
  rcvtimeo = RCVTIMEO;
  sndtimeo = SNDTIMEO;
  


  while (1)
    {
      c = getopt_long (argc, argv,
		       "Vdf:i:I:o:t:g:m:p:h:u:s:c:C:k:r:n:S:T:G:H:5:4:l:",
		       long_options, &option_index);
      if (c == -1)
	break;
      switch (c)
	{
	case 129:
	  usage ();
	  exit (EXIT_SUCCESS);
	  break;
	case 'd':
	  optint ("1", &debug, &set_debug);
	  break;
	case 'f':
	  /* printf ("Check %s firstly\n", optarg); */
	  proxyknife_conf = xmalloc (strlen (optarg) + 1);
	  memset (proxyknife_conf, strlen (optarg) + 1, 0);
	  memmove (proxyknife_conf, optarg, strlen (optarg));
	  break;
	case 'V':
	  printf (_("\
GNU Proxyknife %s\n\
\n\
Copyright (C) 2005, 2006, 2007 %s <skyroam@gmail.com>.\n\
Proxyknife is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the\n\
GNU General Public License for more details.\n"), VERSION, _("Jia Wang"));
	  exit (EXIT_SUCCESS);
	case 'i':
	  optstr (optarg, &proxyknife_in, &set_proxyknifein);
	  break;
	case 'I':
	  optint (optarg, &proxyknife_in_type, &set_proxyknifeintype);
	  break;
	case 'o':
	  optstr (optarg, &proxyknife_out, &set_proxyknifeout);
	  break;
	case 't':
	  optint (optarg, &threads, &set_threads);
	  break;
	case 'g':
	  optstr (optarg, &user_agent, &set_user_agent);
	  break;
	case 'm':
	  optint (optarg, &(my.mytype), &set_mytype);
	  break;
	case 'p':
	  optint (optarg, &(my.myport), &set_myport);
	  break;
	case 'h':
	  optstr (optarg, &(my.myhost), &set_myhost);
	  break;
	case 'u':
	  optstr (optarg, &(my.myuser), &set_myuser);
	  break;
	case 's':
	  optstr (optarg, &(my.mypass), &set_mypass);
	  break;
	case 'c':		/* checksite */
	  optstr (optarg, &(target.target), &set_checksite);
	  break;
	case 'C':		/* "checkreqin=" */
	  optstr (optarg, &(target.checkreqin), &set_checkreqin);
	  break;
	case 'k':		/* "key=" */
	  optstr (optarg, &(target.key), &set_key);
	  break;
	case 'r':		/*"rcvtimeo=" */
	  optint (optarg, &(rcvtimeo), &set_rcvtimeo);
	  break;
	case 'n':		/*"sndtimeo=" */
	  optint (optarg, &(sndtimeo), &set_sndtimeo);
	  break;
	case 'S':		/*"myconreplysize=" */
	  optint (optarg, &(my.myconreplysize), &set_myconreplysize);
	  break;
	case 'T':		/* "testconreplysize=" */
	  optint (optarg, &(test.testconreplysize), &set_testconreplysize);
	  break;
	case 'G':		/* "testtagreplysize=" */
	  optint (optarg, &(test.testtagreplysize), &set_testtagreplysize);
	  break;
	case 'H':		/* "httptestmethod=" */
	  optint (optarg, &(test.httptestmethod), &set_httptestmethod);
	  break;
	case '5':		/* "socks5testmethod=" */
	  optint (optarg, &(test.socks5testmethod), &set_socks5testmethod);
	  break;
	case '4':		/*"socks4testmethod=" */
	  optint (optarg, &(test.socks4testmethod), &set_socks4testmethod);
	  break;
	case 'l':		/*"logfilename=" */
	  optstr (optarg, &(logfilename), &set_logfilename);
	  break;

	default:
	  usage ();
	  exit (EXIT_FAILURE);
	  break;
	}
    }

  if(proxyknife_conf!=NULL)conf = fopen (proxyknife_conf, "r");
  else conf = NULL;

  if (conf == NULL)
    {
      if (debug)
	perror (__FILE__
		" READCONF:fopen:the configuration file in commandline");

      conf = fopen (PROXYKNIFE_CONF, "r");
      if (conf == NULL)
	{
	  if (debug)
	    perror (__FILE__ " " "READCONF:fopen:" PROXYKNIFE_CONF);
	  /* check ~/.proxyknife.conf */
	  home = getenv ("HOME");
	  if (home != NULL)
	    {
	      filenamelen =
		strlen (home) + strlen (PROXYKNIFE_CONF) + strlen ("/.");
	      filename = xmalloc (filenamelen + 1);
	      memset (filename, 0, filenamelen);
	      snprintf (filename, filenamelen + 1, "%s/.%s", home, PROXYKNIFE_CONF);
	      conf = fopen (filename, "r");
	      if (conf == NULL)
		{
		  if (debug)
		    perror ("READCONF:fopen:~/." PROXYKNIFE_CONF);
		}
	      else
		{
		  if (debug)
		    printf ("Using ~/." PROXYKNIFE_CONF "\n");
		}
	      xfree ((void **) &filename);
	    }
	}
      else
	{
	  printf ("READCONF:Using " PROXYKNIFE_CONF "\n");
	}
    }
  else
    {
      printf ("READCONF:Using %s\n", proxyknife_conf);
    }

  if(proxyknife_conf!=NULL)xfree ((void **) &proxyknife_conf);
  /* check  /etc/proxyknife.conf */
  /* error  was deal above, 
     so there will not be  duplicate reports of errors  here */
  if (conf == NULL)
    {
      filename = SYSTEM_PROXYKNIFECONF;	/* so xfree is unnecessary... */
      conf = fopen (filename, "r");
      if (conf == NULL)
	{
	  if(debug){
              perror ("READCONF:fopen:/etc/" PROXYKNIFE_CONF);
	  fprintf (stderr, 
		   "READCONF:Can't find the configuration file!Use commadline options only!\n");
      }

	  /* xexit (EXIT_FAILURE); */
	}
      else
	{
	  if (debug)
	    printf ("Using " SYSTEM_PROXYKNIFECONF "\n");
	}
    }

  slash = '/';
  colon = ':';

  if( conf!= NULL){
  while (1)
    {
      line = NULL;
      linelen = 80;
      read = getaline ((unsigned char **) &line, &linelen, conf);
      if (read == -1)
	break;
      if (read > 0)
	{			/*read=0?  */
	  if (line[0] == '#'){
              xfree((void **)&line);
              continue;
      }
	  if (line[read - 1] == '\n')
	    line[read - 1] = 0;
	  !readint (line, "threads=", &threads, &set_threads)
	    || !readstr (line, "user_agent=", &user_agent, &set_user_agent)
	    || !readint (line, "mytype=", &(my.mytype), &set_mytype)
	    || !readint (line, "myport=", &(my.myport), &set_myport)
	    || !readstr (line, "myhost=", &(my.myhost), &set_myhost)
	    || !readstr (line, "myuser=", &(my.myuser), &set_myuser)
	    || !readstr (line, "mypass=", &(my.mypass), &set_mypass)
	    || !readstr (line, "checksite=", &(target.target), &set_checksite)
	    || !readstr (line, "checkreqin=", &(target.checkreqin),
			 &set_checkreqin)
	    || !readstr (line, "key=", &(target.key), &set_key)
	    || !readint (line, "rcvtimeo=", &(rcvtimeo), &set_rcvtimeo)
	    || !readint (line, "sndtimeo=", &(sndtimeo), &set_sndtimeo)
	    || !readint (line, "myconreplysize=", &(my.myconreplysize),
			 &set_myconreplysize)
	    || !readint (line, "testconreplysize=", &(test.testconreplysize),
			 &set_testconreplysize)
	    || !readint (line, "testtagreplysize=", &(test.testtagreplysize),
			 &set_testtagreplysize)
	    || !readint (line, "httptestmethod=", &(test.httptestmethod),
			 &set_httptestmethod)
	    || !readint (line, "socks5testmethod=", &(test.socks5testmethod),
			 &set_socks5testmethod)
	    || !readint (line, "socks4testmethod=", &(test.socks4testmethod),
			 &set_socks4testmethod)
	    || !readstr (line, "logfilename=", &(logfilename),
			 &set_logfilename)
	    || !readint (line, "proxyknifeintype=", &(proxyknife_in_type),
			 &set_proxyknifeintype)
	    || !readstr (line, "proxyknifein=", &(proxyknife_in),
			 &set_proxyknifein)
	    || !readstr (line, "proxyknifeout=", &(proxyknife_out),
			 &set_proxyknifeout)
	    || !readint (line, "debug=", &debug, &set_debug);
	}
      if (line)
	xfree ((void **)&line);
    }

      if (fclose (conf) != 0)
	{
	  perror (__FILE__ ": READCONF:fclose: configuration file" );
	  xexit (EXIT_FAILURE);
	}

    }
  if (set_mytype * set_httptestmethod * set_socks5testmethod  
      * set_socks4testmethod * set_rcvtimeo * set_sndtimeo 
      * set_checksite * set_key * set_user_agent * set_threads  
      * set_myconreplysize * set_testconreplysize * set_testtagreplysize
      * set_logfilename
     )
    {if(debug){
      fprintf (stderr, "%s: READCONF:not all parameters reset:", progname);
      fprintf (stderr, "set flags are checked:\n"
	       "\tmytype:%d\n"
	       "\thttptestmethod:%d\n"
	       "\tsocks5testmethod:%d\n"
	       "\tsocks4testmethod:%d\n"
	       "\trcvtimeo:%d\n"
	       "\tsndtimeo:%d\n"
	       "\tchecksite:%d\n"
           "\tkey:%d\n"
           "\tuser_agent:%d\n"
           "\tthreads:%d\n"
	       "\tmyconreplysize:%d\n"
	       "\ttestconreplysize:%d\n"
	       "\ttesttagreplysize:%d\n"
	       "\tlogfilename:%d\n",
	       set_mytype, set_httptestmethod, set_socks5testmethod,
	       set_socks4testmethod,
	       set_rcvtimeo, set_sndtimeo,
	       set_checksite, set_key,set_user_agent,set_threads,
	       set_myconreplysize, set_testconreplysize,
	       set_testtagreplysize,set_logfilename );
              }
      /* xexit (EXIT_FAILURE); */
    }

  if (my.mytype != DIRECT)
    {
      if ((my.mytype == HTTP_CONNECT_AUTH)
	  || (my.mytype == SOCKS5_CONNECT_AUTH))
	{
	  if ((set_myuser == 0) || (set_mypass == 0))
	    {
	      fprintf (stderr,
		       "%s: READCONF:Parameters:myuser or mypass isn't set\n",
		       progname);
	      xexit (EXIT_FAILURE);
	    }
	}
      else if ((my.mytype != HTTP_CONNECT) && (my.mytype != SOCKS5_CONNECT))
	{
	  fprintf (stderr,
		   "%s: READCONF: Parameters:mytype %d is not supported now!\n",
		   progname, my.mytype);
	  xexit (EXIT_FAILURE);
	}
      if (set_myhost * set_myport == 0)
	{
	  fprintf (stderr,
		   "%s: READCONF:Parameters:myhost or myport isn't set!\n",
		   progname);
	  xexit (EXIT_FAILURE);
	}

    }
  else
    {
/*fprintf (stderr,
	       "%s: READCONF:Parameters:mytype:DIRECT is not supported now\n",
	       progname);
      xexit (EXIT_FAILURE);
      */
    }

  /* http://$targethost:$targetport$targetpath */
  /* or host:port[path] */
  /* Parse it here to stop proxyknife early and let user know the error in conf */
  /* modify to accept protocol:// or even no // */
  p = strstr ((const char *) target.target, (const char *) "://");
  if (p == NULL)
    {				/* haven't dup now! */
      if (debug)
	fprintf (stderr, "%s %s: READCONF: No '://' found in %s\n",
		 progname, __FILE__, target.target);
      //xexit (EXIT_FAILURE);
      p = target.target;
    }
  else
    {
      stringlen = p - target.target;
      if (stringlen != 0)
	{			/* we do not accept NULL proto, so ignored it. */
	  target.protocol = xmalloc (stringlen + 1);
	  memmove (target.protocol, target.target, stringlen);
	  target.protocol[stringlen] = '\0';
	}

      p += (int) strlen ("://");
    }
  q = strchr (p, colon);
  if (q == NULL)
    {
      if (debug)
	fprintf (stderr, "%s %s: READCONF: No ':' found in %s\n",
		 progname, __FILE__, p);
      xexit (EXIT_FAILURE);
    }
  stringlen = q - p;
  target.targethost = xmalloc (stringlen + 1);
  memmove (target.targethost, p, stringlen);
  target.targethost[stringlen] = '\0';
  /* move p after ':' */
  p = q + 1;
  q = strchr (p, slash);
  if (q == NULL)
    {				/* add fix in the future */
      if (debug)
	fprintf (stderr, "%s %s: READCONF: No '/' found in %s\n",
		 progname, __FILE__, p);
      //xexit (EXIT_FAILURE);
      /* default to '/' */
      target.targetpath = xmalloc (2);
      target.targetpath[0] = '/';
      target.targetpath[1] = '\0';
      /* for port */
      stringlen = strlen (p);
    }
  else
    {				/* path */
      stringlen = strlen (q);
      target.targetpath = xmalloc (stringlen + 1);
      memmove (target.targetpath, q, stringlen);
      target.targetpath[stringlen] = '\0';
      /* for port */
      stringlen = q - p;
    }

  target.targetport = xmalloc (stringlen + 1);
  memmove (target.targetport, p, stringlen);
  target.targetport[stringlen] = '\0';

  /*sscanf ((const char *) target, "http://%[^:]:%[^/]%s", target_host,
     target_port, target_path); */

  h = gethostbyname (target.targethost);

  if (h == NULL)
    {
      herror (target.targethost);
      xexit (EXIT_FAILURE);
    }

  /* init target.targetaddr,used to validate socks5 proxy */
  /* copy it to sin_addr then we can use gethostbyname later */
  target.targetaddr.sin_addr = *(struct in_addr *) (h->h_addr);
  target.targetaddr.sin_port = htons (atoi (target.targetport));
  target.targetaddr.sin_family = AF_INET;
  memset (&(target.targetaddr.sin_zero), 0,
	  sizeof (target.targetaddr.sin_zero));

  /* proxyknife_in, proxyknife_out */

  /* If no NULL options or NULL configuration lines */
  if (proxyknife_in == NULL)
    {
      if (proxyknife_in_type == IN_HTTP)
	{
	  optstr (PROXYKNIFE_DEFAULT_LISTSITE,
		  &proxyknife_in, &set_proxyknifein);
	}
      else
	{
	  optstr (PROXYKNIFE_IN, &proxyknife_in, &set_proxyknifein);
	}
    }
  if (proxyknife_out == NULL)
    {
      optstr (PROXYKNIFE_OUT, &proxyknife_out, &set_proxyknifeout);
    }

  if (debug)
    {
      printf ("setting:\n"
	      "\tlogfilename=%s\n"
	      "\ttarget=%s\n"
	      "\ttargethost=%s\n"
	      "\ttargetport=%s\n"
	      "\ttargetpath=%s\n"
	      "\tkey=%s\n"
	      "\thttptestmethod=%d\n"
	      "\tsocks5testmethod=%d\n"
	      "\tsocks4testmethod=%d\n"
	      "\tthreads=%d\n"
	      "\tuser_agent=%s\n"
	      "\trcvtimeo=%d\n"
	      "\tsndtimeo=%d\n"
	      "\ttestconreplysize=%d\n"
	      "\ttesttagreplysize=%d\n"
	      "\tproxyknife_in_type=%d\n"
	      "\tproxyknife_in=%s\n"
	      "\tproxyknife_out=%s\n"
	      "\tdebug=%d\n",
	      logfilename, target.target, target.targethost,
	      target.targetport, target.targetpath, target.key,
	      test.httptestmethod, test.socks5testmethod,
	      test.socks4testmethod, threads, user_agent, rcvtimeo, sndtimeo,
	      test.testconreplysize, test.testtagreplysize,
	      proxyknife_in_type, proxyknife_in, proxyknife_out, debug);
      switch (my.mytype)
	{
	case DIRECT:
	  printf ("\tmytype=DIRECT\n");
	  break;
	case HTTP_CONNECT:
	  printf ("\tmytype=HTTP_CONNECT\n");
	  printf ("\tmyhost=%s\n\tmyport=%d\n", my.myhost, my.myport);
	  break;
	case HTTP_CONNECT_AUTH:
	  printf ("\tmytype=HTTP_CONNECT_AUTH\n");
	  printf ("\tmyhost=%s\n\tmyport=%d\n", my.myhost, my.myport);
	  printf ("\tmyuser=%s\n\tmypass=%s\n", my.myuser, my.mypass);
	  break;
	case SOCKS5_CONNECT:
	  printf ("\tmytype=SOCKS5_CONNECT\n");
	  printf ("\tmyhost=%s\n\tmyport=%d\n", my.myhost, my.myport);
	  break;
	case SOCKS5_CONNECT_AUTH:
	  printf ("\tmytype=SOCKS5_CONNECT_AUTH\n");
	  printf ("\tmyhost=%s\n\tmyport=%d\n", my.myhost, my.myport);
	  printf ("\tmyuser=%s\n\tmypass=%s\n", my.myuser, my.mypass);
	  break;
	default:		/* if no debug, unecessary to check this */
	  fprintf (stderr,
		   "%s:READCONF:Parameters:invalid mytype:%d!This shouldn't happen!\n",
		   progname, my.mytype);
	  xexit (EXIT_FAILURE);
	  break;
	}
    }
}

/* Read the int after flagstring.
 * 
 * Return value:
 * 0: success!
 * -1: no flagstring found at the head of line.
 * */
int
readint (char *line, char *flagstring, int *i, int *num)
{
  if (!strncmp (line, flagstring, strlen (flagstring)))
    {
      if (*num)
	{
	  if (debug)
	    fprintf (stderr, "Duplicate configuration:%s -- ignored!\n",
		     line);
	  return 0;		/* do not match it with other options */
	  //exit(EXIT_FAILURE);
	}
      else
	{
	  *i = atoi (line + strlen (flagstring));
	  (*num)++;
	  return 0;
	}
    }
  return -1;
}

/* Store  string after flagstring to *s 
 *
 * Return value:
 * 0: success!
 * -1: no flagstring found at the head of line.
 * */
int
readstr (char *line, char *flagstring, char **s, int *num)
{
  int flagsize, len;
  flagsize = strlen (flagstring);

  if (!strncmp (line, flagstring, flagsize))
    {
      /* duplication */
      if (*num)
	{
	  if (debug)
	    fprintf (stderr, "Duplicate configuration:%s -- ignored!\n",
		     line);
	  return 0;		/* do not check it  with other options */
	  //exit(EXIT_FAILURE);
	}
      else
	{

	  len = strlen (line) - flagsize + 1;
	  *s = xmalloc (len);
	  memmove (*s, line + flagsize, len);
	  (*num)++;
	  return 0;
	}
    }
  return -1;
}

/* Make a duplicate of a string with allocated dynamic memory */

void *
xstrdup(char *s)
{
        int len;
        char *p;
        assert(s!=NULL);
        len = strlen(s);
        assert(len!=0);
        len ++;
        p = xmalloc(len);
        memmove(p,s,len);
        return p;
}

/* read a string option from commandline */
void
optstr (char *value, char **s, int *num)
{

  int len;
  /* check  duplication */
  if (*num)
    {
      fprintf (stderr, __FILE__
	       ": Duplication commandline options: %s!\n", value);
      exit (EXIT_FAILURE);
    }
  else
    {
      assert (value != NULL);	/* This should not happen. */
      //xfree(s);
      len = strlen (value) + 1;
      *s = xmalloc (len); /* *s is unallocated before this. */
      memmove (*s, value, len);
      (*num)++;
    }
}

/* read a int option from commandline */
void
optint (char *value, int *i, int *num)
{

  if (*num)
    {
      fprintf (stderr, "Duplicate commandline options :%s \n", value);
      exit (EXIT_FAILURE);
    }
  else
    {

      *i = atoi (value);
      (*num)++;
    }
}
