/* cmd.c - Tcl commands for point repulsion simulation
 *
 * $Id: cmd.c,v 1.1 96/02/11 21:30:06 leech Exp $
 *
 * Copyright (C) 1996, Jonathan P. Leech
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is". Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Log:	cmd.c,v $
 * Revision 1.1  96/02/11  21:30:06  leech
 * Initial revision
 * 
 *
 */

#include <tk.h>
#include <malloc.h>
#include "SphereState.h"

// Append string representation of a vector (list of 3 floats)
//  to a Tcl interpreter.
void Tcl_AppendVector(Tcl_Interp *interp, const dvector &v) {
    char buf[3][20];

    Tcl_PrintDouble(interp, v(0), buf[0]);
    Tcl_PrintDouble(interp, v(1), buf[1]);
    Tcl_PrintDouble(interp, v(2), buf[2]);

    Tcl_AppendResult(interp, buf[0], " ", buf[1], " ", buf[2], (char *)NULL);
}

struct dfunc_cmd {
    const char *cmd;
    void (SphereState::*setfunc)(double);
    double (SphereState::*getfunc)();
};

static dfunc_cmd dcmds[] = {
    { "softening", &SphereState::SetSoftening,	&SphereState::GetSoftening    },
    { "kforce",  &SphereState::SetKForce,	&SphereState::GetKForce       },
    { "r0",	 &SphereState::SetR0,		&SphereState::GetR0	      },
    { "damping", &SphereState::SetDamping,	&SphereState::GetDamping      },
    { "timestep",&SphereState::SetTimestep,	&SphereState::GetTimestep     },
    { "cutoff",  &SphereState::SetCutoffRadius, &SphereState::GetCutoffRadius },
    { "ke",	 NULL,				&SphereState::KineticEnergy   },
    { "pe",	 NULL,				&SphereState::PotentialEnergy },
    { "vmax",	 NULL,				&SphereState::MaxVelocity     },
    { "minpair", NULL,				&SphereState::ClosestPair     },
    { "maxpair", NULL,				&SphereState::FurthestPair    },
    { "idealpair", NULL,			&SphereState::IdealPair       },
    { "idealtimestep", NULL,			&SphereState::DefaultTimestep }
};

struct ifunc_cmd {
    const char *cmd;
    void (SphereState::*setfunc)(int);
    int (SphereState::*getfunc)();
};

static ifunc_cmd icmds[] = {
    { "critdamp",     &SphereState::SetCriticalDamping, &SphereState::GetCriticalDamping },
    { "nbody",	      &SphereState::SetNumBodies,	&SphereState::GetNumBodies },
    { "interactions", NULL,				&SphereState::Interactions },
    { "debug",	      &SphereState::SetDebug,		&SphereState::GetDebug }
};

// General function for setting/getting numeric model parameters
// Called back with arg[0] (if present) = parameter value to set.
// If no argument present, just returns the value.
// Uses passed accessor functions.
int model_dfunc(
    SphereState *model, Tcl_Interp *interp, int ac, char *av[],
    char *cmdname,
    void (SphereState::*setfunc)(double),
    double (SphereState::*getfunc)())
{
    double value;

    if (ac > 0 && setfunc != NULL) {
	if (Tcl_GetDouble(interp, av[0], &value) == TCL_OK) {
	    (model->*setfunc)(value);
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, cmdname, ": bad value ", av[0], (char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	char buf[20];

	value = (model->*getfunc)();
	Tcl_PrintDouble(interp, value, buf);
	Tcl_AppendResult(interp, buf, (char *)NULL);
	return TCL_OK;
    }
}

// General function for setting/getting integer model parameters
// Called back with arg[0] (if present) = parameter value to set.
// If no argument present, just returns the value.
// Uses passed accessor functions.
int model_ifunc(
    SphereState *model, Tcl_Interp *interp, int ac, char *av[],
    char *cmdname,
    void (SphereState::*setfunc)(int),
    int (SphereState::*getfunc)())
{
    int value;

    if (ac > 0 && setfunc != NULL) {
	if (Tcl_GetInt(interp, av[0], &value) == TCL_OK) {
	    (model->*setfunc)(value);
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, cmdname, ": bad value ", av[0], (char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	char buf[20];

	value = (model->*getfunc)();
	sprintf(buf, "%d", value);
	Tcl_AppendResult(interp, buf, (char *)NULL);
	return TCL_OK;
    }
}

struct syment {
    char *name;
    int value;
};

// Lookup value corresponding to a name
// Returns success or failure
int lookup_value(int nsym, syment *sym, char *name, int *value)
{
    for (int i = 0; i < nsym; i++) {
	if (!strcmp(sym[i].name, name)) {
	    *value = sym[i].value;
	    return 1;
	}
    }
    return 0;
}

// Lookup name corresponding to a value
// Returns success or failure
int lookup_name(int nsym, syment *sym, int value, char **name)
{
    for (int i = 0; i < nsym; i++) {
	if (sym[i].value == value) {
	    *name = sym[i].name;
	    return 1;
	}
    }
    return 0;
}

int model_distfunc(SphereState *model, Tcl_Interp *interp, int ac, char *av[])
{
    static syment sym[] = {
	{ "arc", int(ForceLaw::kArcDistance) },
	{ "euclid", int(ForceLaw::kEuclidDistance) }
    };
    int nsym = sizeof(sym) / sizeof(sym[0]);
    int value;

    if (ac > 0) {
	if (lookup_value(nsym, sym, av[0], &value)) {
	    model->GetForceLaw()->SetDistanceFunction(
		ForceLaw::DistanceFunction(value));
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, "bad distfunc ", av[0], (char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	char *name;
	value = model->GetForceLaw()->GetDistanceFunction();
	if (lookup_name(nsym, sym, value, &name)) {
	    Tcl_AppendResult(interp, name, (char *)NULL);
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, "bad value for distfunc", (char *)NULL);
	    return TCL_ERROR;
	}
    }
}

int model_falloff(SphereState *model, Tcl_Interp *interp, int ac, char *av[])
{
    static syment sym[] = {
	{ "linear",  int(ForceLaw::kLinearLaw) },
	{ "square",  int(ForceLaw::kSquareLaw) },
	{ "lennart", int(ForceLaw::kLennartLaw) },
	{ "morse",   int(ForceLaw::kMorseLaw) }
    };
    int nsym = sizeof(sym) / sizeof(sym[0]);
    int value;

    if (ac > 0) {
	if (lookup_value(nsym, sym, av[0], &value)) {
	    model->GetForceLaw()->SetFalloffFunction(
		ForceLaw::FalloffFunction(value));
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, "bad falloff ", av[0], (char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	char *name;
	value = model->GetForceLaw()->GetFalloffFunction();
	if (lookup_name(nsym, sym, value, &name)) {
	    Tcl_AppendResult(interp, name, (char *)NULL);
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, "bad value for falloff", (char *)NULL);
	    return TCL_ERROR;
	}
    }
}

// General function for setting/getting vector model parameters
// Called back with arg[0] = index of point and arg[1-3] (if present) =
//  vector components.
// If no vector present, just returns the value.
// Uses passed accessor functions.
int model_vecfunc(
    SphereState *model, Tcl_Interp *interp, int ac, char *av[],
    char *cmdname,
    void (SphereState::*setfunc)(unsigned, const dvector &),
    dvector (SphereState::*getfunc)(unsigned))
{
    dvector v;
    int n;

    if (ac == 0 || Tcl_GetInt(interp, av[0], &n) != TCL_OK) {
	Tcl_AppendResult(interp, cmdname, ": missing or bad point # ",
	    av[0], (char *)NULL);
	return TCL_ERROR;
    }

    if (n < 0 || n >= model->GetNumBodies()) {
	Tcl_AppendResult(interp, cmdname, ": point # ", av[0],
	    " out of range", (char *)NULL);
	return TCL_ERROR;
    }

    if (ac >= 4 && setfunc != NULL) {
	if (Tcl_GetDouble(interp, av[1], &v[0]) == TCL_OK &&
	    Tcl_GetDouble(interp, av[2], &v[1]) == TCL_OK &&
	    Tcl_GetDouble(interp, av[3], &v[2]) == TCL_OK) {
	    (model->*setfunc)(unsigned(n), v);
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, cmdname, ": bad value ", av[0],
		av[1], av[2], av[3], (char *)NULL);
	    return TCL_ERROR;
	}
    } else if (ac == 1) {
	v = (model->*getfunc)(unsigned(n));
	Tcl_AppendVector(interp, v);
	return TCL_OK;
    } else {
	Tcl_AppendResult(interp, cmdname, ": need 3 vector components",
	    (char *)NULL);
	return TCL_ERROR;
    }
}

// General function for calling model functions taking a string parameter.
// Called back with av[0] = string (required)
int model_stringfunc(
    SphereState *model, Tcl_Interp *interp, int ac, char *av[],
    char *cmdname,
    boolean (SphereState::*func)(const char *))
{
    if (ac > 0) {
	boolean ok = (model->*func)(av[0]);
	if (ok)
	    return TCL_OK;
	else {
	    Tcl_AppendResult(interp, cmdname, ": failed", (char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	Tcl_AppendResult(interp, cmdname, ": missing filename",
	    av[0], (char *)NULL);
	return TCL_ERROR;
    }
}

#include "nnsort.h"
// Function for generating convex hull of model. Returns a list
//  of facets, each of which is 3 indices.
int model_hull(
    SphereState *model, Tcl_Interp *interp, int /*ac*/, char */*av*/[],
    char *cmdname)
{
    int npt = model->GetNumBodies();

    if (npt < 4) {
	Tcl_AppendResult(interp, cmdname, ": need >= 4 points to make hull", (char *)NULL);
	return TCL_ERROR;
    }

    dvector *pt = new dvector[npt];

    for (int i = 0; i < npt; i++)
	pt[i] = model->GetPosition(unsigned(i));

    int (*facets)[3];
    int nf = nnhull3((double (*)[3])pt, npt, &facets);

    for (i = 0; i < nf; i++) {
	char buf[50];
	sprintf(buf, "{ %d %d %d }%s",
	    facets[i][0], facets[i][1], facets[i][2],
	    (i < nf - 1) ? " " : "");
	Tcl_AppendResult(interp, buf, (char *)NULL);
    }
    free((void *)facets);
    return TCL_OK;
}

// Callback for command created for a sphere model; options are
// Set/Get integral params
//  nbody	[n]	- number of points
//  debug	[n]	- debug flag
// Set/Get numeric params
//  critdamp	[flag]	- whether critical damping is enabled
//  softening	[e]	- softening parameter
//  kforce	[k]	- spring constant
//  r0		[r0]	- Lennart-Jones/Morse distance
//  damping	[d]	- damping constant
//  timestep	[dt]	- time step
//  cutoff	[c]	- interaction cutoff distance
// Get integral params (readonly)
//  interactions    - # of pair interactions in most recent compute step
// Get numeric params (readonly)
//  ke		    - kinetic energy
//  pe		    - potential energy
//  vmax	    - maximum velocity
//  minpair	    - minimum nearest-neighbor distance
//  maxpair	    - maximum nearest-neighbor distance
//  idealpair	    - ideal nearest-neighbor distance
//  idealtimestep   - ideal timestep (if critical damping enabled)
// Other
//  compute	    - compute one timestep of dynamics
//  setparams	    - set natural timestep for critical damping
//  distfunc [func] - set/get functional form for computing distances
//  falloff [func]  - set/get functional form for computing force falloff
//  pos i [x y z]   - set/get position of body i
//  vel i [vx vy vz]- set/get velocity of body i
//  read file	    - load bodies from specified file
//  write file	    - save bodies to specified file
//  hull	    - returns list of facet indices
int spheremodel_cmd(ClientData data, Tcl_Interp *interp, int ac, char *av[])
{
    if (ac == 0) {
	Tcl_AppendResult(interp, "wrong # args: need \"option [args]\"", (char *)NULL);
	return TCL_ERROR;
    }

    SphereState *model = (SphereState *)data;
    for (int i = 0; i < sizeof(dcmds) / sizeof(dcmds[0]); i++) {
	if (!strcmp(dcmds[i].cmd, av[1]))
	    return model_dfunc(model, interp, ac-2, av+2,
		av[1], dcmds[i].setfunc, dcmds[i].getfunc);
    }
    for (i = 0; i < sizeof(icmds) / sizeof(icmds[0]); i++) {
	if (!strcmp(icmds[i].cmd, av[1]))
	    return model_ifunc(model, interp, ac-2, av+2,
		av[1], icmds[i].setfunc, icmds[i].getfunc);
    }
    if (!strcmp("compute", av[1])) {
	boolean ok = model->Timestep();
	if (ok)
	    return TCL_OK;
	else {
	    Tcl_AppendResult(interp, "error in SphereState::Compute", (char *)NULL);
	    return TCL_ERROR;
	}
    }
    if (!strcmp("distfunc", av[1]))
	return model_distfunc(model, interp, ac-2, av+2);
    if (!strcmp("falloff", av[1]))
	return model_falloff(model, interp, ac-2, av+2);
    if (!strcmp("pos", av[1]))
	return model_vecfunc(model, interp, ac-2, av+2, av[1],
		&SphereState::SetPosition, &SphereState::GetPosition);
    if (!strcmp("vel", av[1]))
	return model_vecfunc(model, interp, ac-2, av+2, av[1],
		&SphereState::SetVelocity, &SphereState::GetVelocity);
    if (!strcmp("read", av[1]))
	return model_stringfunc(model, interp, ac-2, av+2, av[1],
		&SphereState::ReadBodies);
    if (!strcmp("write", av[1]))
	return model_stringfunc(model, interp, ac-2, av+2, av[1],
		&SphereState::WriteBodies);
    if (!strcmp("hull", av[1]))
	return model_hull(model, interp, ac-2, av+2, av[1]);

    Tcl_AppendResult(interp, "unknown subcommand \"", av[1], "\"", (char *)NULL);
    return TCL_ERROR;
}

// Callback for "spherestate"; options are
//	name
int spherestate_cmd(ClientData, Tcl_Interp *interp, int ac, char *av[])
{
    if (ac == 0) {
	Tcl_AppendResult(interp, "wrong # args: should be \"model-name\"", (char *)NULL);
	return TCL_ERROR;
    }

    // Create a model and pass it as the ClientData parameter
    SphereState *model = new SphereState(4);

    Tcl_CreateCommand(interp, av[1], &spheremodel_cmd, (ClientData)model, NULL);
    Tcl_AppendResult(interp, av[1], (char *)NULL);

    return TCL_OK;
}

// Bind application commands to interpreter
int cmd_Init(Tcl_Interp *interp) {
    Tcl_CreateCommand(interp, "spherestate", &spherestate_cmd, 0, NULL);

    return TCL_OK;
}
