/* morph3d.c  --  morph3d main program */

/* contains: main(), GetImgHeader(), InputImage(), OutputImage(), */
/*           MakeNameList(), PushName()                           */

/* dependencies: GetArgs(), GetSE(), Morph3DSub(), GetRasHeader(),      */
/*               GetRasColorMap(), ReadRasterFile(), WriteRasterFile(), */
/*               ExtractLuminance(), CalcDelay()                        */

/* morph3d version 4.0    1 June 1993              */
/* 3D image morphology program                     */
/*                                                 */
/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */    
/* Nashville, TN 37235                             */
/* rap2@vuse.vanderbilt.edu                        */
/*                                                 */
/* This software is freely redistributable if      */
/* the author's name and affiliation are included. */


#include "morph3d.h"
#include "rasterio.h"

struct IOS
   {
   byte *Img;               /* image as list in row-major order */
   struct rasterfile RasHd; /* raster file header */
   byte *CMap;              /* colormap */
   char **NameList;         /* list of file names of images in memory  */
   int  BuffDepth;          /* depth of name buffer */
   char *basename;          /* output file basename */
   char *mnemo;             /* output file name mnemonic */
   int  Delay;              /* delay between input and output */
   int  eof;                /* end-of-file flag */
   };

int *GetSE();

static void GetImgHeader();
static int InputImage();
static int OutputImage();
static void MakeNameList();
static void PushName();

/* input "file" is expected on stdin. output "file" is to stdout. */

main( argc, argv ) 
   unsigned int argc;  /* count of arguments parsed by operating system    */
   char *argv[];       /* pointers to arguments parsed by operating system */
   {
   struct IOS IOStruct;

   int MorphOp;   /* operation to perform */

   int X,Y;       /* image horizontal, vertical dimensions */
   int ImgType;   /* gray-level or binary */
   int NZpad ;    /* flag.  F => zeropadding of input */
   int NoScale;   /* flag.  T => do not scale output of IntMorph */
   int Dsp;       /* flag.  T => display some info */

   int *SE;
   char *SEName;  /* structuring element (SE) path name */
   int sX,sY,sZ;  /* SE horizontal, vertical, time dimensions */
   int sorgx,sorgy,sorgz;
   int sV;        /* largest gray level in SE */
   int autoSE;    /* T => create a disk-shaped SE locally */
   int SEType;    /* binary or gray-level SE */
   int SorF;      /* Set operation or Function operation */

   int LThresh;   /* lower binary threshold value */
   int UThresh;   /* upper binary threshold value */
   int Rank;      /* rank for rank filter */

   float Slope,Incept;

   /* begin */

   if (argc < 5)
      {
      fprintf(stderr,
       "usage: morph3d <FileList >LogFile -m e|d|o|c|r|a|b|l|m|n|p|q \n\
       [-i g|b] [-s g|b] [-o s|f] [-l kkk lll] \n\
       [-t nnn [mmm]] [-r med|nnn] [-z] [-f slope inter] \n\
       [-n mnemonic] [-b basename] -k SEFile | 3x3x3 | 5x5x5 | plus | \n\
       [cone | cylinder | sphere xxx yyy zzz [vvv]]\n");
      exit(0);
      }

   /* initialize everything */
   IOStruct.CMap = NULL;
   IOStruct.NameList = NULL;
   IOStruct.basename = NULL;
   IOStruct.mnemo = NULL;
   IOStruct.eof = FALSE;
   X=Y=ImgType=NZpad=NoScale=Dsp=0;
   SEName = NULL;
   MorphOp=sX=sY=sZ=sV=autoSE=SEType=SorF=LThresh=UThresh=Rank=0;
   Slope=1.0;
   Incept=0.0;

   /* get arguments */
   GetArgs(argc,argv,&MorphOp,&SEName,&autoSE,&sX,&sY,&sZ,&sV,
           &SorF,&SEType,&ImgType,&Rank,&LThresh,&UThresh,&NZpad,
           &Slope,&Incept,&Dsp,&(IOStruct.basename),&(IOStruct.mnemo));

   /* get the structuring element */
   SE = GetSE(SEName,autoSE,&SEType,&SorF,&sX,&sY,&sZ,&sV,&sorgx,&sorgy,&sorgz);

   /* get output delay */
   IOStruct.Delay = CalcDelay( MorphOp, sZ, sorgz );

   /* get name buffer depth */
   IOStruct.BuffDepth = IOStruct.Delay + 1; 

   /* allocate and setup Name List structure */
   MakeNameList( &(IOStruct.NameList), IOStruct.BuffDepth );

   /* read the header from the first image; if needed, allocates CMap. */
   GetImgHeader( &(IOStruct.RasHd), &(IOStruct.CMap) );

   /* get image dimensions */
   X = IOStruct.RasHd.ras_width;
   Y = IOStruct.RasHd.ras_height;

   /* do the morph op */
   Morph3DSub( MorphOp, InputImage, &IOStruct, OutputImage, &IOStruct,
               X, Y, ImgType, NZpad, LThresh, UThresh,
               SE, sX, sY, sZ, sV, sorgx, sorgy, sorgz, SEType, 
               SorF, Rank);


   free( IOStruct.NameList );
   if ( IOStruct.CMap ) free( IOStruct.CMap );
   }



static void GetImgHeader( RasHd, CMap )
   struct rasterfile *RasHd;
   byte **CMap;
   {
   char InName[FNLGTH+1];
   FILE *fp;

   fscanf( stdin, "%s", InName );
   rewind( stdin );

   if ( InName[0] == '\0' )
      {
      fprintf( stderr, "Cannot read input file name via stdin.\n");
      exit(0);
      }

   fp = OpenFile( InName, INPATH, "r" );

   if ( GetRasHeader( fp, RasHd ) ) 
      { 
      fprintf( stderr, "Input file: %s\n", InName );
      exit(0);
      }

   /* this field might be zero for some images, so compute it */
   if ( !RasHd->ras_length )
      {
      if ( RasHd->ras_width % 2 ) /* if there are an odd # pix / line */
         {
         RasHd->ras_length = (RasHd->ras_width+1) * RasHd->ras_height *
                          RasHd->ras_depth / 8;
         }
      else /* there are an even # of pixels / line */
         {
         RasHd->ras_length = RasHd->ras_width * RasHd->ras_height *
                             RasHd->ras_depth / 8;
         }
      }

   if ( !RasHd->ras_length ) 
      {
      fprintf(stderr,"The first input image has zero length\n");
      exit(0);
      }

   if ( GetRasColorMap( fp, RasHd, CMap, TRUE ) ) 
      { 
      fprintf( stderr, "Input file: %s\n", InName );
      exit(0);
      }


   fclose( fp );

   return;
   }

static int InputImage( Img, NumReqs, eof, InStruct )
   byte *Img;
   int NumReqs;
   int *eof;
   struct IOS *InStruct;
   {
   char InName[FNLGTH+1];
   FILE *fp;
   struct rasterfile CurrHd;
   byte CMap[3*(WHITE+1)];
   byte *CM = CMap;
   byte *I = Img;

   /* get next file name */
   fscanf( stdin, "%s", InName );

   /* if at eof in name list, return */
   if ( feof(stdin) ) 
      {
      *eof = TRUE;
      InStruct->eof = TRUE;
      return( NOERR );
      }
    else 
      *eof = FALSE;

   /* open the file */
   if ( InName[0] != '\0' )
       fp = OpenFile( InName, INPATH, "r" );
   else
     {
      fprintf( stderr, "InputImage: Cannot read input file name via stdin.\n");
      return( ERROR );
      }

   /* read it */
   if (ReadRasterFile( fp, &CurrHd, &CM, &I, 0, 0, 0, 0, FALSE ))
      {
      fprintf( stderr, "InputImage: error reading input file %s\n",InName);
      return( ERROR );
      }

   /* close it */
   fclose( fp );

   /* print info */
   fprintf( stdout, "input %d: %s\n", NumReqs,InName ); 

   /* compare header to reference */
   if ( CurrHd.ras_width  != InStruct->RasHd.ras_width  ||
        CurrHd.ras_height != InStruct->RasHd.ras_height    )
         { 
         fprintf( stderr, "Input file: %s is of incorrect size.\n", InName );
         return( ERROR );
         }

   /* push name onto list */
   PushName( InName, InStruct->NameList, InStruct->BuffDepth );

   /* extract the luminance */
   ExtractLuminance( &CurrHd, CM, I, 0, 0, 0, 0 );

   return( NOERR );
   }


static int OutputImage( Img, NumReqs, OutStruct )
   byte *Img;
   int NumReqs;
   struct IOS *OutStruct;
   {
   FILE *fp;
   char OutName[FNLGTH+1];
   char Index[IXLGTH+2];
   char Suffix[SXLGTH+1];
   char Format[FTLGTH];
   char *dot;

   /* if we are at eof then the input procedure did not scroll the name list */
   if ( OutStruct->eof == TRUE ) 
      PushName( "xxx", OutStruct->NameList, OutStruct->BuffDepth ); 

   /* if a basename was specified, use it */
   if ( OutStruct->basename )
      {
      strncpy( OutName, OutStruct->basename, FNLGTH );
      strncpy( Suffix, ".ras", SXLGTH );

      /* write index field width in format string */
      sprintf( Format, "%%0%1dd", IXLGTH );

      /* create index field for output file name */
      Index[0] = '.';
      sprintf( Index+1, Format, NumReqs );

      /* make output file name from basename + mnemonic + index + suffix */
      if ( OutStruct->mnemo )
         strcat( OutName, OutStruct->mnemo );
      strcat( OutName, Index );
      strcat( OutName, Suffix );
      }
   else /* construct a filename from the input file name */
      {
      /* get input file name */
      strncpy( OutName, (OutStruct->NameList)[0], FNLGTH );

      /* find the suffix */
      dot = rindex( OutName, '.' );
      if ( (dot != NULL) && strcmp(dot,".Z") ) /* if suffix is .Z skip it */
         {
         *dot = '\000';
         *(dot+1) = '\000';
         dot = rindex( OutName, '.' );
         }
         
      if ( dot != NULL )  /* there is a suffix; save it */
         {
         strncpy( Suffix, dot, SXLGTH );
         *dot = '\000';
         }
      else  /* there is not a suffix; use .ras */
         strncpy( Suffix, ".ras", SXLGTH );

      /* write index field width in format string */
      sprintf( Format, "%%0%1dd", IXLGTH );

      /* create index field for output file name */
      Index[0] = '.';
      sprintf( Index+1, Format, NumReqs );

      /* make output file name from corresponding */ 
      /* input file name + mnemonic + index + suffix */
      if ( OutStruct->mnemo )
         strcat( OutName, OutStruct->mnemo );
      strcat( OutName, Index );
      strcat( OutName, Suffix );
      }

   fp = OpenFile( OutName, OUTPATH, "w" );

   /* write it */
   if (WriteRasterFile( 
       fp, &(OutStruct->RasHd), OutStruct->CMap, Img, 0, 0, 0, 0 ))
      {
      fprintf( stderr, "OutputImage: error writing output file %s\n",OutName);
      return( ERROR );
      }
  
   /* close it */
   fclose( fp );

   /* print info */
   fprintf( stdout, "output %d: %s\n", NumReqs,OutName ); 

   return( NOERR );
   }


static void MakeNameList( Names, N )
   char ***Names;
   int N;
   {
   int i;

   if ( !(*Names = 
            (char **)calloc( N*(sizeof(char *) + FNLGTH*sizeof(char)), 1 )) )
      {
      fprintf(stderr,"MakeNameList: unable to allocate name list\n");
      exit( 0 );
      }

   for ( i=0; i<N; ++i )
      (*Names)[i] = (char *)(*Names) + N*sizeof(char *) + i*FNLGTH*sizeof(char);
   }



static void PushName( Name, List, N )
   char *Name;
   char **List;
   int N;
   {
   int i;
   char *P;
   
   P = List[0];
   for ( i=1; i<N; ++i )
      List[i-1] = List[i];
   List[N-1] = P;
   strncpy( List[N-1], Name, FNLGTH );
   return;
   }
