/*******************************************************************************
+
+  LEDA  3.0
+
+
+  rs_tree.h
+
+
+  Copyright (c) 1992  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 6600 Saarbruecken, FRG     
+  All rights reserved.
+ 
*******************************************************************************/


#ifndef LEDA_RS_TREE_H
#define LEDA_RS_TREE_H

#include <math.h>
#include <LEDA/basic.h>

//
// (Generalized) Randomized Search Trees
// -------------------------------------
// by C.Aragon and R.Seidel
//
//
// variable probability distribution
// doubly-linked with pred-succ-pointer
//
// 6/92, M.Paul
//

/* still to implement: conc, split */



enum rst_dir { rst_left=1, rst_right=2 } ;



class rst_node
{
  friend class rs_tree ;

  GenPtr key ;
  int prio ;			// priority between 0 and MAXINT-1
  GenPtr inf ;
  rst_node* ptr[5] ;		// parent, left, right, pred, succ

  LEDA_MEMORY( rst_node ) ;
} ;

typedef rst_node* rst_item ;



class rs_tree
{
  rst_item root ;			// pointer to root AND leaf
  int count ;				// number of entries
  double log_prob ;			// log of probability 

  virtual int cmp(GenPtr x, GenPtr y) const { return compare(x,y) ; }
  virtual void copy_key( GenPtr& )  const {}
  virtual void copy_inf( GenPtr& )  const {}
  virtual void clear_key( GenPtr& ) const {}
  virtual void clear_inf( GenPtr& ) const {}
  virtual void print_key( GenPtr ) const {}
  virtual void print_inf( GenPtr ) const {}
  
  virtual int int_type() const { return 0 ; }

  inline rst_item& parent( rst_item p ) const { return p->ptr[0] ; }
  inline rst_item& l_child( rst_item p ) const { return p->ptr[1] ; } 
  inline rst_item& r_child( rst_item p ) const { return p->ptr[2] ; }
  inline rst_item& p_item( rst_item p ) const { return p->ptr[3] ; }
  inline rst_item& s_item( rst_item p ) const { return p->ptr[4] ; }
  inline rst_item& child( rst_item p, rst_dir d ) const ;

  rst_item search( GenPtr key, rst_item& parent ) const ;
  rst_item int_insert_at_item( rst_item p, GenPtr key, GenPtr inf ) ;
  rst_item gen_insert_at_item( rst_item p, GenPtr key, GenPtr inf ) ;
  void print_tree( rst_item p, string ins ) ;

  protected:

    rst_item item( GenPtr p ) const { return rst_item(p) ; }
    int rand_prio() const ;

  public:

    rs_tree( double prob = 1 ) ;	// default is "normal" RST,
					// which is also the fastest
    rs_tree( const rs_tree& rst ) ;
    rs_tree& operator=( const rs_tree& rst ) ;
    virtual ~rs_tree() ;

    void print_tree() ;			// should be private because
    void print_item( rst_item p ) ;	// priorities are protected
    void print_all_items() ;
    
    // sorted sequence operations

    rst_item lookup( GenPtr key ) const ;
    rst_item locate( GenPtr key ) const ;
    rst_item locate_pred( GenPtr key ) const ;
    rst_item insert_at_item( rst_item p, GenPtr key, GenPtr inf ) ;
    rst_item insert( GenPtr key, GenPtr inf ) ;
    void del_item( rst_item p ) ;
    void del( GenPtr key ) ;
    void reverse_items( rst_item pl, rst_item pr ) ;
    void change_inf( rst_item p, GenPtr inf ) ;
    void clear() ;

    rst_item pred( rst_item p ) const ;
    rst_item succ( rst_item p ) const ;
    rst_item first_item() const ;
    rst_item next_item( rst_item p ) const ;

    rst_item min() const ;
    rst_item max() const ;
    int size() const ;
    int empty() const ;

    GenPtr key( rst_item p ) const ;
    GenPtr inf( rst_item p ) const ;
    GenPtr& info( rst_item p ) const ;

    // additional priority queue operations

    rst_item find_min() const ;
    void del_min() ;
    void decrease_key( rst_item p, GenPtr key ) ;

    void conc(rs_tree&) 
    { error_handler(1,"sorry, rs_tree::conc not implemented"); }

    void split_at_item(rst_item,rs_tree&,rs_tree&) 
    { error_handler(1,"sorry, rs_tree::split_at_item not implemented"); }


} ;



inline rst_item& rs_tree::child( rst_item p, rst_dir d ) const 
{ return p->ptr[d] ; 
}



// choose random priority according to pr( prio==h ) = prob^h
// for prob==1 (log_prob==0) priorities are uniformly distributed
//
inline int rs_tree::rand_prio() const
{
  if( !log_prob )
    return random(0,MAXINT-1) ;
  else
    return int(log(rrandom())/log_prob) ;
}



// sorted sequence operations

inline rst_item rs_tree::pred( rst_item p ) const
{ rst_item q = p_item(p) ;
  return (q==root) ? 0 : q ;
}

inline rst_item rs_tree::succ( rst_item p ) const
{ rst_item q = s_item(p) ;
  return (q==root) ? 0 : q ;
}

inline rst_item rs_tree::first_item() const { return succ(root) ; }

inline rst_item rs_tree::next_item( rst_item p ) const { return succ(p) ; }



inline rst_item rs_tree::min() const { return succ(root) ; }

inline rst_item rs_tree::max() const { return pred(root) ; }

inline int rs_tree::size() const { return count ; }

inline int rs_tree::empty() const { return (count==0) ; }



inline GenPtr rs_tree::key( rst_item p ) const { return p->key ; }

inline GenPtr rs_tree::inf( rst_item p ) const { return p->inf ; }

inline GenPtr& rs_tree::info( rst_item p ) const { return p->inf ; }



inline rst_item rs_tree::lookup( GenPtr key ) const
{ rst_item pp ;
  return search( key, pp ) ; 
}

inline rst_item rs_tree::insert_at_item( rst_item p, 
					 GenPtr key, GenPtr inf )
{
  if( int_type() )
    return int_insert_at_item( p, key, inf ) ;
  else
    return gen_insert_at_item( p, key, inf ) ;
}

inline rst_item rs_tree::insert( GenPtr key, GenPtr inf )
{
  rst_item pp,
	   p = search( key, pp ) ;
  if( p ) {					// key exists
    clear_inf( p->inf ) ;  
    p->inf = inf ;  copy_inf( p->inf ) ;
    return p ;
  }
  else 
    return insert_at_item( pp, key, inf ) ;
}

inline void rs_tree::del( GenPtr key )
{ rst_item pp,
	   p = search( key, pp ) ;
  if( p )
    del_item( p ) ;
}

inline void rs_tree::change_inf( rst_item p, GenPtr inf ) { p->inf = inf ; }



// additional priority queue operations

inline rst_item rs_tree::find_min() const { return succ(root) ; }

inline void rs_tree::del_min()
{ rst_item p = find_min() ;
  if( p )
    del_item( p ) ;
}

inline void rs_tree::decrease_key( rst_item p, GenPtr key )
{
  insert( key, inf(p) ) ;
  del_item(p) ;
}



#endif
