/******************************************************************************
* Cagd_Wrt.c - Generic Curve/Surface writing to files.			      *
*******************************************************************************
* Written by Gershon Elber, July. 90.					      *
******************************************************************************/

#ifdef USE_VARARGS
#    include <varargs.h>
#else
#    include <stdarg.h>
#endif /* USE_VARARGS */

#include <string.h>
#include "prsr_loc.h"
#include "irit_soc.h"

/*****************************************************************************
* DESCRIPTION:                                                               M
* Generic routine to write curve(s) to the given file.			     M
*   If Comment is NULL, no comment is printed, if "" only internal comment.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Crvs:      To be saved in file f.                                        M
*   FileName:  File name where output should go to.                          M
*   Indent:    Column in which all printing starts at.                       M
*   Comment:   Optional comment to describe the geometry.                    M
*   ErrStr:    If failed, ErrStr will be set to describe the problem.        M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:        TRUE if succesful, FALSE otherwise.                          M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdCrvWriteToFile, files, write                                         M
*****************************************************************************/
int CagdCrvWriteToFile(CagdCrvStruct *Crvs,
		       char *FileName,
		       int Indent,
		       char *Comment,
		       char **ErrStr)
{
    int RetVal = TRUE;
    CagdCrvStruct *NextCrv;

    for (; Crvs != NULL && RetVal; Crvs = Crvs -> Pnext) {
	NextCrv = Crvs -> Pnext;      /* To make sure we dump one at a time. */
	Crvs -> Pnext = NULL;

	switch(Crvs -> GType) {
	    case CAGD_CBEZIER_TYPE:
		RetVal = BzrCrvWriteToFile(Crvs, FileName, Indent,
					   Comment, ErrStr);
		break;
	    case CAGD_CBSPLINE_TYPE:
		RetVal = BspCrvWriteToFile(Crvs, FileName, Indent,
					   Comment, ErrStr);
		break;
	    case CAGD_CPOWER_TYPE:
		CAGD_FATAL_ERROR(CAGD_ERR_POWER_NO_SUPPORT);
	    default:
		CAGD_FATAL_ERROR(CAGD_ERR_UNDEF_CRV);
		return FALSE;
	}

	Crvs -> Pnext = NextCrv;
    }

    return RetVal;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Generic routine to write curve(s) to the given stream.		     M
*   If Comment is NULL, no comment is printed, if "" only internal comment.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Crvs:      To be saved in stream.                                        M
*   Handler:   A handler to the open stream.				     M
*   Indent:    Column in which all printing starts at.                       M
*   Comment:   Optional comment to describe the geometry.                    M
*   ErrStr:    If failed, ErrStr will be set to describe the problem.        M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:        TRUE if succesful, FALSE otherwise.                          M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdCrvWriteToFile2, files, write                                        M
*****************************************************************************/
int CagdCrvWriteToFile2(CagdCrvStruct *Crvs,
			int Handler,
			int Indent,
			char *Comment,
			char **ErrStr)
{
    int RetVal = TRUE;
    CagdCrvStruct *NextCrv;

    for (; Crvs != NULL && RetVal; Crvs = Crvs -> Pnext) {
	NextCrv = Crvs -> Pnext;      /* To make sure we dump one at a time. */
	Crvs -> Pnext = NULL;

	switch(Crvs -> GType) {
	    case CAGD_CBEZIER_TYPE:
		RetVal = BzrCrvWriteToFile2(Crvs, Handler, Indent,
					    Comment, ErrStr);
		break;
	    case CAGD_CBSPLINE_TYPE:
		RetVal = BspCrvWriteToFile2(Crvs, Handler, Indent,
					    Comment, ErrStr);
		break;
	    case CAGD_CPOWER_TYPE:
		CAGD_FATAL_ERROR(CAGD_ERR_POWER_NO_SUPPORT);
		return FALSE;
	    default:
		CAGD_FATAL_ERROR(CAGD_ERR_UNDEF_CRV);
		return FALSE;
	}

	Crvs -> Pnext = NextCrv;
    }

    return RetVal;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Generic routine to write curve(s) to the given file.			     M
*   If Comment is NULL, no comment is printed, if "" only internal comment.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Crvs:      To be saved in file f.                                        M
*   f:         File descriptor where output should go to.                    M
*   Indent:    Column in which all printing starts at.                       M
*   Comment:   Optional comment to describe the geometry.                    M
*   ErrStr:    If failed, ErrStr will be set to describe the problem.        M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:        TRUE if succesful, FALSE otherwise.                          M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdCrvWriteToFile3, files, write                                        M
*****************************************************************************/
int CagdCrvWriteToFile3(CagdCrvStruct *Crvs,
			FILE *f,
			int Indent,
			char *Comment,
			char **ErrStr)
{
    int Handler = IritPrsrOpenStreamFromFile(f, FALSE, FALSE, FALSE),
	i = CagdCrvWriteToFile2(Crvs, Handler, Indent, Comment, ErrStr);

    IritPrsrCloseStream(Handler, TRUE);

    return i;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Generic routine to write surface(s) to the given file.		     M
*   If Comment is NULL, no comment is printed, if "" only internal comment.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Srfs:      To be saved in file f.                                        M
*   FileName:  File name where output should go to.                          M
*   Indent:    Column in which all printing starts at.                       M
*   Comment:   Optional comment to describe the geometry.                    M
*   ErrStr:    If failed, ErrStr will be set to describe the problem.        M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:        TRUE if succesful, FALSE otherwise.                          M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdSrfWriteToFile, files, write                                         M
*****************************************************************************/
int CagdSrfWriteToFile(CagdSrfStruct *Srfs,
		       char *FileName,
		       int Indent,
		       char *Comment,
		       char **ErrStr)
{
    int RetVal = TRUE;
    CagdSrfStruct *NextSrf;

    for (; Srfs != NULL && RetVal; Srfs = Srfs -> Pnext) {
	NextSrf = Srfs -> Pnext;      /* To make sure we dump one at a time. */
	Srfs -> Pnext = NULL;

	switch (Srfs -> GType) {
	    case CAGD_SBEZIER_TYPE:
		RetVal = BzrSrfWriteToFile(Srfs, FileName, Indent,
					   Comment, ErrStr);
		break;
	    case CAGD_SBSPLINE_TYPE:
		RetVal = BspSrfWriteToFile(Srfs, FileName, Indent,
					   Comment, ErrStr);
		break;
	    case CAGD_SPOWER_TYPE:
		CAGD_FATAL_ERROR(CAGD_ERR_POWER_NO_SUPPORT);
		return FALSE;
	    default:
		CAGD_FATAL_ERROR(CAGD_ERR_UNDEF_SRF);
		return FALSE;
	}

	Srfs -> Pnext = NextSrf;
    }

    return RetVal;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Generic routine to write surface(s) to the given stream.		     M
*   If Comment is NULL, no comment is printed, if "" only internal comment.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Srfs:      To be saved in stream.                                        M
*   Handler:   A handler to the open stream.				     M
*   Indent:    Column in which all printing starts at.                       M
*   Comment:   Optional comment to describe the geometry.                    M
*   ErrStr:    If failed, ErrStr will be set to describe the problem.        M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:        TRUE if succesful, FALSE otherwise.                          M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdSrfWriteToFile2, files, write, stream                                M
*****************************************************************************/
int CagdSrfWriteToFile2(CagdSrfStruct *Srfs,
			int Handler,
			int Indent,
			char *Comment,
			char **ErrStr)
{
    int RetVal = TRUE;
    CagdSrfStruct *NextSrf;

    for (; Srfs != NULL && RetVal; Srfs = Srfs -> Pnext) {
	NextSrf = Srfs -> Pnext;      /* To make sure we dump one at a time. */
	Srfs -> Pnext = NULL;

	switch (Srfs -> GType) {
	    case CAGD_SBEZIER_TYPE:
		RetVal = BzrSrfWriteToFile2(Srfs, Handler, Indent,
					    Comment, ErrStr);
		break;
	    case CAGD_SBSPLINE_TYPE:
		RetVal = BspSrfWriteToFile2(Srfs, Handler, Indent,
					    Comment, ErrStr);
		break;
	    case CAGD_SPOWER_TYPE:
		CAGD_FATAL_ERROR(CAGD_ERR_POWER_NO_SUPPORT);
		return FALSE;
	    default:
		CAGD_FATAL_ERROR(CAGD_ERR_UNDEF_SRF);
		return FALSE;
	}

	Srfs -> Pnext = NextSrf;
    }

    return RetVal;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Generic routine to write surface(s) to the given file.		     M
*   If Comment is NULL, no comment is printed, if "" only internal comment.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Srfs:      To be saved in file f.                                        M
*   f:         File descriptor where output should go to.                    M
*   Indent:    Column in which all printing starts at.                       M
*   Comment:   Optional comment to describe the geometry.                    M
*   ErrStr:    If failed, ErrStr will be set to describe the problem.        M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:        TRUE if succesful, FALSE otherwise.                          M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdSrfWriteToFile3, files, write                                        M
*****************************************************************************/
int CagdSrfWriteToFile3(CagdSrfStruct *Srfs,
			FILE *f,
			int Indent,
			char *Comment,
			char **ErrStr)
{
    int Handler = IritPrsrOpenStreamFromFile(f, FALSE, FALSE, FALSE),
	i = CagdSrfWriteToFile2(Srfs, Handler, Indent, Comment, ErrStr);

    IritPrsrCloseStream(Handler, TRUE);

    return i;
}
