/*****************************************************************************
* Filter to convert IRIT data files to REND386 (IBMPC only) Plg format.	     *
*									     *
* Written by:  Gershon Elber				Ver 1.0, Sep 1991    *
*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "irit_sm.h"
#include "iritprsr.h"
#include "allocate.h"
#include "attribut.h"
#include "iritgrap.h"
#include "getarg.h"
#include "genmat.h"
#include "ffcnvrt.h"
#include "ip_cnvrt.h"

#define GLOBAL_SCALE	200.0  /* Scale obj space -1..1 to -200..200 pixels. */

#ifdef NO_CONCAT_STR
static char *VersionStr =
	"Irit2Plg		Version 5.0,	Gershon Elber,\n\
	 (C) Copyright 1989/90-95 Gershon Elber, Non commercial use only.";
#else
static char *VersionStr = "Irit2Plg	" VERSION ",	Gershon Elber,	"
	__DATE__ ",   " __TIME__ "\n" COPYRIGHT ", Non commercial use only.";
#endif /* NO_CONCAT_STR */

static char
#ifdef DOUBLE
    *CtrlStr = "irit2plg l%- 4%- F%-PolyOpti|FineNess!d!F T%- z%- DFiles!*s";
#else
    *CtrlStr = "irit2plg l%- 4%- F%-PolyOpti|FineNess!d!f T%- z%- DFiles!*s";
#endif /* DOUBLE */

static MatrixType CrntViewMat;			/* This is the current view! */

static int TransColorTable[][4] = {
    { /* BLACK		*/ 0,    0,   0,   0 },
    { /* BLUE		*/ 1,    0,   0, 255 },
    { /* GREEN		*/ 2,    0, 255,   0 },
    { /* CYAN		*/ 3,    0, 255, 255 },
    { /* RED		*/ 4,  255,   0,   0 },
    { /* MAGENTA 	*/ 5,  255,   0, 255 },
    { /* BROWN		*/ 6,   50,   0,   0 },
    { /* LIGHTGRAY	*/ 7,  127, 127, 127 },
    { /* DARKGRAY	*/ 8,   63,  63,  63 },
    { /* LIGHTBLUE	*/ 9,    0,   0, 255 },
    { /* LIGHTGREEN	*/ 10,   0, 255,   0 },
    { /* LIGHTCYAN	*/ 11,   0, 255, 255 },
    { /* LIGHTRED	*/ 12, 255,   0,   0 },
    { /* LIGHTMAGENTA	*/ 13, 255,   0, 255 },
    { /* YELLOW		*/ 14, 255, 255,   0 },
    { /* WHITE		*/ 15, 255, 255, 255 },
    { /* BROWN		*/ 20,  50,   0,   0 },
    { /* DARKGRAY	*/ 56,  63,  63,  63 },
    { /* LIGHTBLUE	*/ 57,   0,   0, 255 },
    { /* LIGHTGREEN	*/ 58,   0, 255,   0 },
    { /* LIGHTCYAN	*/ 59,   0, 255, 255 },
    { /* LIGHTRED	*/ 60, 255,   0,   0 },
    { /* LIGHTMAGENTA	*/ 61, 255,   0, 255 },
    { /* YELLOW		*/ 62, 255, 255,   0 },
    { /* WHITE		*/ 63, 255, 255, 255 },
    {			   -1,   0,   0,   0 }
};

static int PlgColorTable[16] = {
	/* IRIT     PLG   */
    15, /* BLOCK -> WHITE */
    11, /* BLUE */
    7,  /* GREEN */
    8,  /* CYAN */
    1,  /* RED */
    13, /* MAGENTA */
    2,  /* BROWN */
    14, /* LIGHTGRAY */
    14, /* DARKGRAY */
    10, /* LIGHT BLUE */
    9,  /* LIGHT GREEN */
    8,  /* LIGHT CYAN */
    1,  /* LIGHT RED */
    13, /* LIGHT MAGENTA */
    6,  /* YELLOW */
    15, /* WHITE */    
};

static void DumpDataForPlg(IPObjectStruct *PObjects);
static void DumpOneObject(FILE *f,
			  IPObjectStruct *PObject,
			  int DumpVertices,
			  int *IncVertices);
static void DumpOnePolygon(FILE *f,
			   IPPolygonStruct *PPolygon,
			   int Color,
			   int DumpVertices,
			   int *IncVertices,
			   int IsPolygon);
static RealType *MapPoint(RealType *Pt);
static void Irit2PlgExit(int ExitCode);

/*****************************************************************************
* DESCRIPTION:                                                               M
* Main module of irit2plg - Read command line and do what is needed...	     M
*                                                                            *
* PARAMETERS:                                                                M
*   argc, argv:  Command line.                                               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   main                                                                     M
*****************************************************************************/
void main(int argc, char **argv)
{
    int Error,
	FineNessFlag = FALSE,
	VerFlag = FALSE,
	NumFiles = 0;
    char
	**FileNames = NULL;
    IPObjectStruct *PObjects;

    if ((Error = GAGetArgs(argc, argv, CtrlStr,
			   &FFCState.LinearOnePolyFlag,
			   &FFCState.FourPerFlat, &FineNessFlag,
			   &FFCState.OptimalPolygons,
			   &FFCState.FineNess, &FFCState.Talkative,
			   &VerFlag, &NumFiles, &FileNames)) != 0) {
	GAPrintErrMsg(Error);
	GAPrintHowTo(CtrlStr);
	Irit2PlgExit(1);
    }

    if (VerFlag) {
	fprintf(stderr, "\n%s\n\n", VersionStr);
	GAPrintHowTo(CtrlStr);
	Irit2PlgExit(0);
    }

    if (FFCState.LinearOnePolyFlag) {
	CagdSetLinear2Poly(CAGD_ONE_POLY_PER_COLIN);
    }
    else
        CagdSetLinear2Poly(CAGD_REG_POLY_PER_LIN);

    if (!NumFiles) {
	fprintf(stderr, "No data file names were given, exit.\n");
	GAPrintHowTo(CtrlStr);
	Irit2PlgExit(1);
    }

    /* Get the data files: */
    IritPrsrSetFlattenObjects(TRUE);
    if ((PObjects = IritPrsrGetDataFiles(FileNames, NumFiles, TRUE, FALSE)) ==
									NULL)
	Irit2PlgExit(1);

    if (IritPrsrWasPrspMat)
        MatMultTwo4by4(CrntViewMat, IritPrsrViewMat, IritPrsrPrspMat);
    else
	GEN_COPY(CrntViewMat, IritPrsrViewMat, sizeof(MatrixType));

    DumpDataForPlg(PObjects);

    Irit2PlgExit(0);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps the data for REND386 to stdout.                                      *
*                                                                            *
* PARAMETERS:                                                                *
*   PObjects:   To dump into file.                                           *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void DumpDataForPlg(IPObjectStruct *PObjects)
{
    int IncVertices,
        TotalVertices = 0,
	TotalPolys = 0;
    char
	*Name = NULL;
    IPVertexStruct *PVertex;
    IPPolygonStruct *PPoly;
    IPObjectStruct *PObj;
    FILE *f = stdout;	       /* Change it if you want it into "real" file. */
    

    /* Process freeform geometry into polys. */
    for (PObj = PObjects; PObj != NULL; PObj = PObj -> Pnext) {
	if (IP_IS_FFGEOM_OBJ(PObj))
	    PObjects = ProcessFreeForm(PObj, &FFCState);
    }

    /* Count how many polygons/vertices we have in this data set and print. */
    for (PObj = PObjects; PObj != NULL; PObj = PObj -> Pnext) {
	if (!IP_IS_POLY_OBJ(PObj) || !IP_IS_POLYGON_OBJ(PObj))
	    continue;

	if (Name == NULL && strlen(PObj -> Name) > 0)
	    Name = PObj -> Name;

	for (PPoly = PObj -> U.Pl;
	     PPoly != NULL;
	     PPoly = PPoly -> Pnext) {
	    TotalPolys++;

	    for (PVertex = PPoly -> PVertex;
		 PVertex != NULL;
		 PVertex = PVertex -> Pnext) {
		TotalVertices++;
	    }
	}
    }

    if (Name != NULL)
        fprintf(f, "IRIT %d %d\n", TotalVertices, TotalPolys);
    else
	fprintf(f, "%s_IRIT %d %d\n", Name, TotalVertices, TotalPolys);

    for (IncVertices = 0, PObj = PObjects; PObj != NULL; PObj = PObj -> Pnext)
	DumpOneObject(f, PObj, TRUE, &IncVertices);
    if (IncVertices != TotalVertices) {
	fprintf(stderr, "Inconsistent vertices count.\n");
	Irit2PlgExit(1);
    }

    for (IncVertices = 0, PObj = PObjects; PObj != NULL; PObj = PObj -> Pnext)
	DumpOneObject(f, PObj, FALSE, &IncVertices);
    if (IncVertices != TotalVertices) {
	fprintf(stderr, "Inconsistent vertices count.\n");
	Irit2PlgExit(1);
    }

    fclose(f);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps one object PObject to file f.                                        *
*                                                                            *
* PARAMETERS:                                                                *
*   f:            File to dump object to.                                    *
*   PObject:      Object to dump to file f.                                  *
*   DumpVertices: Do we dump vertices now?				     *
*   IncVertices:  Count number of vertices.				     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void DumpOneObject(FILE *f,
			  IPObjectStruct *PObject,
			  int DumpVertices,
			  int *IncVertices)
{
    int i, Color;
    IPPolygonStruct *PList;

    if (!IP_IS_POLY_OBJ(PObject))
	return;

    PList = PObject -> U.Pl;

    if ((Color = AttrGetObjectColor(PObject)) != IP_ATTR_NO_COLOR) {
	for (i = 0; TransColorTable[i][0] >= 0; i++) {
	    if (TransColorTable[i][0] == Color) {
		Color = PlgColorTable[i];
		break;
	    }
	}
    }
    else
        Color = IG_DEFAULT_COLOR;

    while (PList) {
	DumpOnePolygon(f, PList, Color, DumpVertices, IncVertices,
						IP_IS_POLYGON_OBJ(PObject));
	PList =	PList -> Pnext;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps one polygon, using global Matrix transform CrntViewMat.		     *
*                                                                            *
* PARAMETERS:                                                                *
*   f:            File to dump polygon to.    		                     *
*   PPolygon:     Polygon to dump to file f.                                 *
*   Color:	  Attribute of polygon.					     *
*   DumpVertices: Do we dump vertices now?				     *
*   IncVertices:  Count number of vertices.				     *
*   IsPolygon:    Is it a polygon or a polyline?                             *
*                                                                            *
* RETURN VALUE:                                                              *
*   void	                                                             *
*****************************************************************************/
static void DumpOnePolygon(FILE *f,
			   IPPolygonStruct *PPolygon,
			   int Color,
			   int DumpVertices,
			   int *IncVertices,
			   int IsPolygon)
{
    int CountVertices;
    RealType *MappedPoint;
    IPVertexStruct *V,
	*VList = PPolygon -> PVertex;

    if (VList == NULL)
	return;

    if (DumpVertices && !IritPrsrIsConvexPolygon(PPolygon)) {
	static int
	    Printed = FALSE;

	if (!Printed) {
	    fprintf(stderr,
		    "\nWARNING: Non convex polygon(s) might be in data (see CONVEX in IRIT),\n\t\t\t\toutput can be wrong as the result!\n");
	    Printed = TRUE;
	}
    }

    if (IsPolygon) {
	if (DumpVertices) {
	    for (V = VList; V != NULL; V = V -> Pnext) {
		MappedPoint = MapPoint(V -> Coord);
		fprintf(f, "%4d %4d %4d\n",
			(int) (MappedPoint[0] * GLOBAL_SCALE),
			(int) (MappedPoint[1] * GLOBAL_SCALE),
			(int) (MappedPoint[2] * GLOBAL_SCALE));
		(*IncVertices)++;
	    }
	}
	else {
	    for (CountVertices = 0, V = VList; V != NULL; V = V -> Pnext)
	        CountVertices++;
	    fprintf(f, "0x%02xff %d", Color | 0x0010, CountVertices);
	    while (CountVertices-- > 0)
	        fprintf(f, " %d", (*IncVertices)++);
	    fprintf(f, "\n");
	}
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Maps the given E3 point using the CrntViewMat.			     *
*                                                                            *
* PARAMETERS:                                                                *
*   Pt:        Point to map.                                                 *
*                                                                            *
* RETURN VALUE:                                                              *
*   RealType *:   Mapped point, in local static place.                       *
*****************************************************************************/
static RealType *MapPoint(RealType *Pt)
{
    static RealType MappedPt[3];

    MatMultVecby4by4(MappedPt, Pt, CrntViewMat);

    return MappedPt;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Irit2Plg exit routine.						     *
*                                                                            *
* PARAMETERS:                                                                *
*   ExitCode:                                                                *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void Irit2PlgExit(int ExitCode)
{
    exit(ExitCode);
}

#ifdef DEBUG

/*****************************************************************************
* DESCRIPTION:                                                               *
*    Dummy function to link at debugging time.                               *
*                                                                            *
* PARAMETERS:                                                                *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*                                                                            *
* KEYWORDS:                                                                  *
*****************************************************************************/
void DummyLinkCagdDebug(void)
{
    IritPrsrDbg();
}

#endif /* DEBUG */
