
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*************************************************************
*
*     file:     symmetry.c
*
*     Purpose:  Functions dealing with symmetries of surface.
*               Has the three functions for torus.
*               If user wants to define his/her own symmetry
*               group, put your functions in this file,
*               and change names right below here.  But leave
*               torus functions intact, so torus domain can
*               be used.
*
*/

#include "include.h"

/*
       Group elements are encoded in two-bit fields in "wrap",
       one for each dimension, with "001" for positive wrap
       and "011" for negative wrap.  You are free to encode
       any way you want, but use wrap = 0 for the identity.
*/

#define ALLWRAPMASK 03333333333

/*******************************************************************
*
*  function: torus_wrap
*
*  purpose:  Provide adjusted coordinates for vertices that get
*            wrapped around torus.  Serves as example for user-written
*            symmetry function.
*
*            This function uses the values of the torus periods read
*            in from the data file. Yours doesn't have the privilege
*            of defining its own datafile syntax, but you can access
*            values of parameters defined in the datafile as
*            web.params[n].value.
*/

void torus_wrap(x,y,wrap)
REAL *x;   /* original coordinates */
REAL *y;   /* wrapped coordinates  */
WRAPTYPE wrap;  /* encoded symmetry group element, 3 bits per dimension */
{
  int i,j;

  memcpy((char*)y,(char*)x,web.sdim*sizeof(REAL));
  for ( i = 0 ; wrap != 0 ; i++, wrap >>= TWRAPBITS )
    switch ( wrap & WRAPMASK )
      {  case 0: break;
	 case POSWRAP: for ( j = 0 ; j < web.sdim ; j++ )
		   y[j] += web.torus_period[i][j];
		 break;
	 case NEGWRAP: for ( j = 0 ; j < web.sdim ; j++ )
		   y[j] -= web.torus_period[i][j];
		 break;
         default: error("Unexpectedly large torus wrap.\n",WARNING);
      }
}


/*******************************************************************
*
*  function: torus_form_pullback
*
*  purpose:  Pull back differentail forms at vertices that get
*            wrapped around torus.  Serves as example for user-written
*            symmetry function.
*
*            Torus is flat, so implementation is trivial copy.
*/

void torus_form_pullback(x,xform,yform,wrap)
REAL *x;   /* original coordinates */
REAL *xform; /* pulled back form */
REAL *yform;   /* wrapped form, input  */
WRAPTYPE wrap;  /* encoded symmetry group element, 3 bits per dimension */
{
  memcpy((char*)xform,(char*)yform,web.sdim*sizeof(REAL));
}

/********************************************************************
*
*  function: torus_compose()
*
*  purpose:  do composition of two group elements
*
*/

WRAPTYPE torus_compose(wrap1,wrap2)
WRAPTYPE wrap1,wrap2;  /* the elements to compose */
{ int w = (wrap1 + wrap2) & ALLWRAPMASK;
  if ( ((w & 03)==2) || ((w&030)==020) || ((w&0300)==0200) )
    error("Bad wrap.\n",WARNING);
  return w;
}


/********************************************************************
*
*  function: torus_inverse()
*
*  purpose:  return inverse of group element.
*
*/

WRAPTYPE torus_inverse(wrap)
WRAPTYPE wrap;  /* the element invert */
{
  return ((~ALLWRAPMASK)-wrap) & ALLWRAPMASK;
}
  

  
