/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/************************************************************************
*
*  File: stringq.c
*
*  Purpose:
*  Functions to calculate area, length, and energy according to
*  the QUADRATIC STRING model.
*/

#include "include.h"

/* combining coefficients for coordinates of interpolation pts */
/* initialized in scoeff_init() */
REAL gcombo[EDGE_CTRL][EDGE_INTERP];

/* antisymmetric coefficients for area integral */
/* with diagonal elements to give area over x-axis for torus ease */
REAL scoeff[EDGE_CTRL][EDGE_CTRL] = {
     { 0.5, -2.0/3, 1.0/6 },{2.0/3, 0.0, -2.0/3},{-1.0/6, 2.0/3, -0.5} } ;

/* length gradient coefficients for string forces */
/* sdip[control][eval] - eval is integration index
                         control is control point index for grad */

REAL sdip[EDGE_CTRL][EDGE_INTERP];

/* integration coefficients for string forces */
/* ssimp[control][eval]                       */
REAL ssimp[EDGE_CTRL][EDGE_INTERP];

/* three-point gaussian integration on [0,2] */
REAL gauss2pt[EDGE_INTERP] = { 0.225403330758517, 1.0, 1.774596669241483 };
REAL gauss2wt[EDGE_INTERP] = { 5.0/9, 8.0/9, 5.0/9 };

REAL interpoly(k,u)
int  k;
REAL u;
{
  switch ( k )
   {
     case  0:  return (1 - u)*(2 - u)/2;
     case  1:  return u*(2 - u);
     case  2:  return u*(u - 1)/2;
   }

  return  0.0;  /* error return */
}

REAL interpolyderiv(k,u)
int  k;
REAL u;
{
  switch ( k )
   {
     case  0:  return u - 1.5;
     case  1:  return 2 - 2*u;
     case  2:  return u - 0.5;
   }

  return  0.0;  /* error return */
}

void scoeff_init()
{
   int i,j;

   for ( i = 0 ; i < EDGE_CTRL ;  i++ )
     for ( j = 0 ; j < EDGE_INTERP ; j++ )
       {
         gcombo[i][j] = interpoly(i,gauss2pt[j]);
         sdip[i][j] = interpolyderiv(i,gauss2pt[j]);
         ssimp[i][j] = gauss2wt[j]*sdip[i][j];
       }
}

/************************************************************************
*
*  Calculates all forces on control points due to edge and
*  accumulates them at each control point.
*  Quadratic version.
*/

void edge_force_q(e_id)
edge_id e_id;
{
  REAL *x[EDGE_CTRL],*force[EDGE_CTRL],tang[EDGE_INTERP][MAXCOORD];
  REAL norm[MAXCOORD];
  REAL len;
  REAL density = get_edge_density(e_id);
  vertex_id hv = get_edge_headv(e_id);
  vertex_id mv = get_edge_midv(e_id);
  vertex_id tv = get_edge_tailv(e_id);

  int i,j,k;

  x[0] = get_coord(tv);
  x[1] = get_coord(mv);
  x[2] = get_coord(hv);
  force[0] = get_force(tv);
  force[1] = get_force(mv);
  force[2] = get_force(hv);
  for ( i = 0 ; i < EDGE_INTERP ; i++ )
    {
      for ( j = 0 ; j < web.sdim ; j ++ )
        {
          tang[i][j] = 0.0;
          for ( k = 0 ; k < EDGE_CTRL ; k++ )
            tang[i][j] += sdip[k][i]*x[k][j];
        }
      norm[i] = sqrt(dot(tang[i],tang[i],web.sdim));
      for ( j = 0 ; j < web.sdim ; j ++ )
        tang[i][j] /= norm[i];
    }
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    for ( j = 0 ; j < web.sdim ; j++ )
      for ( k = 0 ; k < EDGE_INTERP ; k++ )
        force[i][j] -= density*ssimp[i][k]*tang[k][j];

  len = 0.0;
  for ( i = 0 ; i < EDGE_INTERP ; i++ )
    len += gauss2wt[i]*norm[i];

  set_edge_length(e_id,len);
  if ( web.area_norm_flag )
    { add_vertex_star(hv,len);
      add_vertex_star(mv,len);
      add_vertex_star(tv,len);
    }

  /* calculate gravitational forces */
  if ( web.gravflag )
    {
      REAL z;
      REAL density = edge_grav_density(e_id);

      if ( density != 0.0 )
        {
          for ( i = 0 ; i < EDGE_INTERP ; i++ )
            { z = 0.0;
              for ( j = 0 ; j < EDGE_CTRL ; j++ )
                z += gcombo[j][i]*x[j][1];
              for ( j = 0 ; j < EDGE_CTRL ; j++ )
               { force[j][1] -= -density*z*tang[i][0]*gcombo[j][i];
                 force[j][0] -= -0.5*density*z*z*sdip[j][i];
               }
            }              
        }
    }               

}

/************************************************************************
*
*  Returns energy due to one edge.
*
*  Quadratic version.
*/

void edge_energy_q(e_id)
edge_id e_id;
{
  REAL x[EDGE_CTRL][MAXCOORD];
  REAL *pt[EDGE_CTRL];
  REAL tang[MAXCOORD];
  vertex_id v[EDGE_CTRL];
  REAL len;
  int i,j,k;
  REAL z;
  REAL density = edge_grav_density(e_id);
              
  v[0] = get_edge_tailv(e_id);
  v[1] = get_edge_midv(e_id);
  v[2] = get_edge_headv(e_id);
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    {
      pt[i] = get_coord(v[i]);
    }
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    for ( j = 0 ; j < web.sdim ; j++ )
      x[i][j] = pt[i][j];
          
    
  /* calculate tangents at integration points and accumulate */
  len = 0.0;
  for ( i = 0 ; i < EDGE_INTERP ; i++ )
    {
      for ( j = 0 ; j < web.sdim ; j ++ )
        {
          tang[j] = 0.0;
          for ( k = 0 ; k < EDGE_CTRL ; k++ )
            tang[j] += sdip[k][i]*x[k][j];
        }
      len += gauss2wt[i]*sqrt(dot(tang,tang,web.sdim));

      /* calculate gravitational energy */
      if ( web.gravflag &&  ( density != 0.0 ) )
        {
          for ( i = 0 ; i < EDGE_INTERP ; i++ )
            { z = 0.0;
              for ( j = 0 ; j < EDGE_CTRL ; j++ )
                z += gcombo[j][i]*x[j][1];
              web.total_energy += -0.5*density*z*z*tang[0];
            }              
        }
    }
  set_edge_length(e_id,len);

  /* following three lines for area normalization option */
  add_vertex_star(v[0],len);
  add_vertex_star(v[1],len);
  add_vertex_star(v[2],len);
    
  web.total_energy += len*get_edge_density(e_id);
  if ( web.dimension == STRING ) /* don't count triple junction as area */
     web.total_area   += len;


}
 
/**********************************************************************
*
*  Adds contribution of edge to areas of neighboring facets, via
*  Green's theorem.   x-y plane only.
*
*/

void edge_area_q(fe_id)
facetedge_id fe_id;
{
  body_id b_id0;
  REAL *x[EDGE_CTRL];
  REAL area = 0.0;
  vertex_id v[EDGE_CTRL];
  int i,j;
  edge_id e_id = get_fe_edge(fe_id);
  REAL torusx[MAXCOORD];  /* for unwrapping head in torus */
  REAL *truehead;   /* used for torus wrap area adjust */
              
  /* be sure have original orientation of edge so torus midpoints
     have same wrap as tail */
  if ( inverted(e_id) )
   { invert(e_id); invert(fe_id); }

  v[0] = get_edge_tailv(e_id);
  v[1] = get_edge_midv(e_id);
  v[2] = get_edge_headv(e_id);
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    x[i] = get_coord(v[i]);
          
  if ( web.torus_flag ) 
   { /* unwrap head */
     truehead = x[2];
     (*sym_wrap)(x[2],torusx,get_edge_wrap(e_id));
     x[2] = torusx;
   }
	 
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    for ( j = 0 ; j < EDGE_CTRL ; j ++ )
      area += scoeff[i][j]*x[i][1]*x[j][0];

  if ( web.torus_flag )
    { /* adjust for wrapping */
      int w = get_edge_wrap(e_id);
      for ( i = 0 ; i < 2 ; i++,w>>=3 )
        {
	  if ( (w&3) == 0 ) continue;
          if ( (w&3) == 1 )   /* can get true position of head x[0] */
            area += x[2][0]*web.torus_period[i][1];
          else 
            area -= truehead[0]*web.torus_period[i][1];
        }
    }
   
 

  /* add to body areas */
  b_id0 = get_facet_body(get_fe_facet(fe_id));
  if ( valid_id(b_id0 ) )
    set_body_volume(b_id0,get_body_volume(b_id0) + area);
  
  b_id0 = get_facet_body(facet_inverse(get_fe_facet(fe_id)));
  if ( valid_id(b_id0 ) )
    set_body_volume(b_id0,get_body_volume(b_id0) - area);
}

/*****************************************************************
*
*  Function: string_grad_q()
*
*  Purpose:  Calculate area gradients at vertices.
*/

void string_grad_q()
{
  body_id bi_id;  /* identifier for body i */
  body_id bj_id;  /* identifier for body j */
  facetedge_id fe_id;
  REAL *x[EDGE_CTRL];
  int i,k;
  vertex_id v[EDGE_CTRL];
  volgrad *vgptr;

  FOR_ALL_FACETEDGES(fe_id)
   {
     v[0] = get_fe_tailv(fe_id);
     v[1] = get_fe_midv(fe_id);
     v[2] = get_fe_headv(fe_id);
     for ( i = 0 ; i < EDGE_CTRL ; i++ )
       {
         x[i] = get_coord(v[i]);
       }

     bi_id = get_facet_body(get_fe_facet(fe_id));
     bj_id = get_facet_body(get_fe_facet(get_next_facet(fe_inverse(fe_id))));

     if ( valid_id(bi_id) && (get_battr(bi_id) & FIXEDVOL) )
       for ( i = 0 ; i < EDGE_CTRL ; i++ )
         { 
           vgptr = get_bv_new_vgrad(bi_id,v[i]);
           for ( k = 0 ; k < EDGE_CTRL ; k++ )
             {
               vgptr->grad[0] +=  scoeff[k][i]*x[k][1];
               vgptr->grad[1] +=  scoeff[i][k]*x[k][0];
             }
       }

     if ( valid_id(bj_id) && !equal_id(bj_id,bi_id) && 
                                            (get_battr(bj_id) & FIXEDVOL) )
       for ( i = 0 ; i < EDGE_CTRL ; i++ )
         { 
           vgptr = get_bv_new_vgrad(bj_id,v[i]);
           for ( k = 0 ; k < EDGE_CTRL ; k++ )
             {
               vgptr->grad[0] -=  scoeff[k][i]*x[k][1];
               vgptr->grad[1] -=  scoeff[i][k]*x[k][0];
             }
       }
  }
}
